<?php
/**
 * Test Database Helper Class
 * Handles test database setup, teardown, and data seeding
 */

class TestDatabase {
    private static $pdo = null;
    
    /**
     * Get PDO connection to test database
     */
    public static function getConnection() {
        if (self::$pdo === null) {
            try {
                $dsn = "mysql:host=" . TEST_DB_HOST . ";dbname=" . TEST_DB_NAME . ";charset=" . TEST_DB_CHARSET;
                $options = [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false,
                ];
                self::$pdo = new PDO($dsn, TEST_DB_USER, TEST_DB_PASS, $options);
            } catch (PDOException $e) {
                throw new Exception("Test database connection failed: " . $e->getMessage());
            }
        }
        return self::$pdo;
    }
    
    /**
     * Setup test database - create tables from schema
     */
    public static function setup() {
        $pdo = self::getConnection();
        
        // Read unified schema file
        if (file_exists(TEST_SCHEMA_PATH)) {
            $schema = file_get_contents(TEST_SCHEMA_PATH);
            // Remove CREATE DATABASE and USE statements
            $schema = preg_replace('/CREATE DATABASE[^;]+;/i', '', $schema);
            $schema = preg_replace('/USE[^;]+;/i', '', $schema);
        } else {
            throw new Exception("Unified schema file not found at: " . TEST_SCHEMA_PATH . ". Please ensure database/schema_main.sql exists.");
        }
        
        // Remove comments and execute statements
        $statements = self::parseSqlStatements($schema);
        
        try {
            $pdo->beginTransaction();
            
            foreach ($statements as $statement) {
                if (!empty(trim($statement))) {
                    $pdo->exec($statement);
                }
            }
            
            $pdo->commit();
            
            // Seed test data if enabled
            if (TEST_SEED_DATA) {
                self::seedTestData();
            }
            
        } catch (PDOException $e) {
            $pdo->rollBack();
            throw new Exception("Failed to setup test database: " . $e->getMessage());
        }
    }
    
    /**
     * Parse SQL statements from schema file
     */
    private static function parseSqlStatements($sql) {
        // Remove SQL comments (but preserve content within quotes)
        $sql = preg_replace('/--[^\r\n]*/', '', $sql);
        $sql = preg_replace('/\/\*.*?\*\//s', '', $sql);
        
        // Remove phpMyAdmin specific statements
        $sql = preg_replace('/SET SQL_MODE[^;]*;/i', '', $sql);
        $sql = preg_replace('/START TRANSACTION[^;]*;/i', '', $sql);
        $sql = preg_replace('/COMMIT[^;]*;/i', '', $sql);
        $sql = preg_replace('/SET time_zone[^;]*;/i', '', $sql);
        $sql = preg_replace('/SET @OLD_[^;]*;/i', '', $sql);
        $sql = preg_replace('/\/\*![^}]*\*\//', '', $sql);
        
        // Split by semicolons, but be careful with semicolons in strings
        $statements = [];
        $current = '';
        $inString = false;
        $stringChar = '';
        
        for ($i = 0; $i < strlen($sql); $i++) {
            $char = $sql[$i];
            $current .= $char;
            
            // Handle string delimiters
            if (($char === '"' || $char === "'") && ($i === 0 || $sql[$i-1] !== '\\')) {
                if (!$inString) {
                    $inString = true;
                    $stringChar = $char;
                } elseif ($char === $stringChar) {
                    $inString = false;
                }
            }
            
            // Split on semicolon if not in string
            if ($char === ';' && !$inString) {
                $stmt = trim($current);
                if (!empty($stmt) && strlen($stmt) > 5) {
                    $statements[] = $stmt;
                }
                $current = '';
            }
        }
        
        // Add remaining statement if any
        $stmt = trim($current);
        if (!empty($stmt) && strlen($stmt) > 5) {
            $statements[] = $stmt;
        }
        
        return array_filter($statements);
    }
    
    /**
     * Seed test database with sample data
     */
    public static function seedTestData() {
        $pdo = self::getConnection();
        
        try {
            $pdo->beginTransaction();
            
            // Seed subscription tiers (if not already seeded)
            $stmt = $pdo->query("SELECT COUNT(*) FROM subscription_tiers");
            if ($stmt->fetchColumn() == 0) {
                $sql = "INSERT INTO subscription_tiers 
                        (tier_name, tier_level, monthly_price, google_ads_budget, facebook_ads_budget, 
                         email_contacts_limit, email_campaigns_limit, discount_codes_limit, 
                         direct_mail_quantity, direct_mail_frequency, features) VALUES
                        ('Starter', 1, 299.00, 200.00, 150.00, 1000, 4, 5, 500, 'quarterly', 'Basic discount codes, Monthly metrics report'),
                        ('Professional', 2, 599.00, 500.00, 400.00, 5000, NULL, NULL, 1500, 'monthly', 'Advanced discount management, Real-time dashboard, A/B testing, Campaign optimization'),
                        ('Enterprise', 3, 999.00, 1200.00, 800.00, NULL, NULL, NULL, 5000, 'monthly', 'Premium discount system, Full API access, Dedicated account manager, Advanced analytics, Multi-channel automation, Custom integrations')";
                $pdo->exec($sql);
            }
            
            // Seed test admin user
            $stmt = $pdo->query("SELECT COUNT(*) FROM users WHERE email = 'admin@test.com'");
            if ($stmt->fetchColumn() == 0) {
                $passwordHash = password_hash('test123', PASSWORD_DEFAULT);
                $sql = "INSERT INTO users (email, password_hash, business_name, contact_name, role, status) VALUES
                        ('admin@test.com', ?, 'Test Admin', 'Test Administrator', 'admin', 'active')";
                $pdo->prepare($sql)->execute([$passwordHash]);
            }
            
            // Seed test client user
            $stmt = $pdo->query("SELECT COUNT(*) FROM users WHERE email = 'client@test.com'");
            if ($stmt->fetchColumn() == 0) {
                $passwordHash = password_hash('test123', PASSWORD_DEFAULT);
                $sql = "INSERT INTO users (email, password_hash, business_name, contact_name, role, status) VALUES
                        ('client@test.com', ?, 'Test Business', 'Test Client', 'client', 'active')";
                $pdo->prepare($sql)->execute([$passwordHash]);
            }
            
            $pdo->commit();
        } catch (PDOException $e) {
            $pdo->rollBack();
            throw new Exception("Failed to seed test data: " . $e->getMessage());
        }
    }
    
    /**
     * Truncate all tables (clean slate for tests)
     */
    public static function truncateAll() {
        $pdo = self::getConnection();
        
        // Disable foreign key checks
        $pdo->exec("SET FOREIGN_KEY_CHECKS = 0");
        
        $tables = [
            'user_metrics', 'ad_performance', 'direct_mail_campaigns', 'email_campaigns',
            'email_subscribers', 'discount_codes', 'transactions', 'subscriptions',
            'campaigns', 'api_credentials', 'users'
        ];
        
        foreach ($tables as $table) {
            try {
                $pdo->exec("TRUNCATE TABLE `$table`");
            } catch (PDOException $e) {
                // Table might not exist, continue
            }
        }
        
        // Re-enable foreign key checks
        $pdo->exec("SET FOREIGN_KEY_CHECKS = 1");
    }
    
    /**
     * Reset database to initial state
     */
    public static function reset() {
        self::truncateAll();
        if (TEST_SEED_DATA) {
            self::seedTestData();
        }
    }
}

