<?php
/**
 * Test Campaign and Ad Flow
 * Tests: 1. Create campaign, 2. Create ad, 3. Save ad to campaign, 4. Check email campaign loading, 5. Verify acceleratedadz.php access
 */

require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/includes/database.php';

session_start();

// Check if user is logged in
if (!isLoggedIn()) {
    die("ERROR: Please log in first to run this test.\n");
}

$userId = $_SESSION['user_id'];
$db = getDB();

echo "=== Campaign and Ad Flow Test ===\n\n";

$errors = [];
$results = [];
$baseUrl = 'http' . (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 's' : '') . '://' . $_SERVER['HTTP_HOST'];

try {
    // Step 1: Create a test campaign
    echo "Step 1: Creating test campaign...\n";
    $campaignName = 'Test Campaign ' . date('Y-m-d H:i:s');
    $campaignType = 'multi_channel'; // Multi-channel campaign (works with Ad Builder and Email Campaigns)
    
    $stmt = $db->prepare("
        INSERT INTO campaigns (user_id, campaign_name, campaign_type, budget, start_date, end_date, description, status)
        VALUES (?, ?, ?, ?, ?, ?, ?, 'active')
    ");
    $stmt->execute([
        $userId,
        $campaignName,
        $campaignType,
        null, // budget
        date('Y-m-d'), // start_date
        null, // end_date
        'Test campaign for ad flow testing' // description
    ]);
    
    $campaignId = $db->lastInsertId();
    echo "✓ Campaign created: ID = {$campaignId}, Name = '{$campaignName}'\n\n";
    $results['campaign_id'] = $campaignId;
    $results['campaign_name'] = $campaignName;
    
    // Step 2: Create an ad and save it
    echo "Step 2: Creating and saving ad...\n";
    $adData = [
        'name' => 'Test Ad ' . date('H:i:s'),
        'platform' => 'google_ads',
        'template' => 'retail',
        'headline' => 'Amazing Test Product',
        'description' => 'This is a test ad description for testing purposes.',
        'cta' => 'Shop Now',
        'ctaColor' => '#ff6b6b',
        'bgColor' => '#ffffff',
        'textColor' => '#2c3e50',
        'bgImage' => '',
        'productImage' => '',
        'destinationUrl' => 'https://example.com',
        'size' => '1200x628'
    ];
    
    // Save ad to saved_ads table
    $stmt = $db->prepare("
        INSERT INTO saved_ads (user_id, ad_name, ad_data, platform, template_type)
        VALUES (?, ?, ?, ?, ?)
    ");
    $stmt->execute([
        $userId,
        $adData['name'],
        json_encode($adData),
        $adData['platform'],
        $adData['template']
    ]);
    
    $savedAdId = $db->lastInsertId();
    echo "✓ Ad saved: ID = {$savedAdId}, Name = '{$adData['name']}'\n\n";
    $results['ad_id'] = $savedAdId;
    $results['ad_name'] = $adData['name'];
    
    // Step 3: Add ad to campaign
    echo "Step 3: Adding ad to campaign...\n";
    $stmt = $db->prepare("
        INSERT INTO campaign_ads (campaign_id, saved_ad_id, ad_name, ad_platform, status)
        VALUES (?, ?, ?, ?, 'active')
        ON DUPLICATE KEY UPDATE 
            status = 'active',
            updated_at = NOW()
    ");
    $stmt->execute([
        $campaignId,
        $savedAdId,
        $adData['name'],
        $adData['platform']
    ]);
    
    $campaignAdId = $db->lastInsertId();
    // If lastInsertId returns 0, it might be an UPDATE, so get the ID
    if (!$campaignAdId) {
        $stmt = $db->prepare("SELECT id FROM campaign_ads WHERE campaign_id = ? AND saved_ad_id = ?");
        $stmt->execute([$campaignId, $savedAdId]);
        $campaignAd = $stmt->fetch();
        $campaignAdId = $campaignAd ? $campaignAd['id'] : null;
    }
    
    echo "✓ Ad added to campaign: Campaign Ad ID = " . ($campaignAdId ?: 'N/A') . "\n\n";
    $results['campaign_ad_id'] = $campaignAdId;
    
    // Step 4: Check if email campaign can see the ad
    echo "Step 4: Checking if email campaign can load the ad...\n";
    $stmt = $db->prepare("
        SELECT DISTINCT c.id, c.campaign_name
        FROM campaigns c
        INNER JOIN campaign_ads ca ON c.id = ca.campaign_id
        WHERE c.user_id = ? AND ca.status = 'active' AND c.id = ?
    ");
    $stmt->execute([$userId, $campaignId]);
    $campaignsWithAds = $stmt->fetchAll();
    
    if (count($campaignsWithAds) > 0) {
        echo "✓ Email campaign can see the ad! Found campaign: " . $campaignsWithAds[0]['campaign_name'] . "\n";
        
        // Get ads for this campaign
        $stmt = $db->prepare("
            SELECT ca.*, sa.ad_data 
            FROM campaign_ads ca
            LEFT JOIN saved_ads sa ON ca.saved_ad_id = sa.id
            WHERE ca.campaign_id = ? AND ca.status = 'active'
        ");
        $stmt->execute([$campaignId]);
        $ads = $stmt->fetchAll();
        
        echo "  Found " . count($ads) . " ad(s) in this campaign\n";
        foreach ($ads as $ad) {
            echo "  - Ad: " . $ad['ad_name'] . " (ID: " . $ad['id'] . ")\n";
        }
        echo "\n";
        $results['email_campaign_ads_found'] = true;
        $results['ad_count'] = count($ads);
    } else {
        echo "✗ Email campaign cannot see the ad\n";
        $errors[] = "Email campaign cannot see ads";
        $results['email_campaign_ads_found'] = false;
    }
    
    // Step 5: Verify acceleratedadz.php can access the ad
    echo "Step 5: Verifying acceleratedadz.php can access the ad...\n";
    
    if ($campaignAdId) {
        // Check if ad can be retrieved via campaign_ads
        $stmt = $db->prepare("
            SELECT ca.*, sa.ad_data, c.id as campaign_id, c.campaign_name, c.user_id
            FROM campaign_ads ca
            LEFT JOIN saved_ads sa ON ca.saved_ad_id = sa.id
            JOIN campaigns c ON ca.campaign_id = c.id
            WHERE ca.id = ? AND ca.status = 'active'
        ");
        $stmt->execute([$campaignAdId]);
        $campaignAd = $stmt->fetch();
        
        if ($campaignAd) {
            echo "✓ Ad can be retrieved from campaign_ads table\n";
            echo "  Campaign Ad ID: " . $campaignAd['id'] . "\n";
            echo "  Ad Name: " . $campaignAd['ad_name'] . "\n";
            echo "  Campaign: " . $campaignAd['campaign_name'] . "\n";
            
            $adDataRetrieved = json_decode($campaignAd['ad_data'] ?? '{}', true);
            echo "  Headline: " . ($adDataRetrieved['headline'] ?? 'N/A') . "\n";
            echo "  Description: " . ($adDataRetrieved['description'] ?? 'N/A') . "\n";
            echo "\n";
            
            // Generate URL for acceleratedadz.php
            require_once __DIR__ . '/includes/base_path.php';
            $acceleratedAdzPath = url('acceleratedadz.php');
            $acceleratedAdzUrl = $baseUrl . $acceleratedAdzPath . '?campaign_ad=' . $campaignAdId . '&source=campaign';
            
            echo "✓ AcceleratedAdz URL generated:\n";
            echo "  " . $acceleratedAdzUrl . "\n\n";
            
            $results['acceleratedadz_url'] = $acceleratedAdzUrl;
            $results['acceleratedadz_accessible'] = true;
            
            // Also try by campaign ID
            $acceleratedAdzUrlByCampaign = $baseUrl . $acceleratedAdzPath . '?campaign=' . $campaignId;
            echo "  Alternative URL (by campaign): " . $acceleratedAdzUrlByCampaign . "\n\n";
            $results['acceleratedadz_url_by_campaign'] = $acceleratedAdzUrlByCampaign;
            
        } else {
            echo "✗ Ad cannot be retrieved from campaign_ads table\n";
            $errors[] = "Ad retrieval failed";
            $results['acceleratedadz_accessible'] = false;
        }
    } else {
        echo "✗ Campaign Ad ID is missing\n";
        $errors[] = "Campaign Ad ID missing";
        $results['acceleratedadz_accessible'] = false;
    }
    
    // Summary
    echo "=== Test Summary ===\n";
    echo "Campaign ID: " . ($results['campaign_id'] ?? 'N/A') . "\n";
    echo "Ad ID: " . ($results['ad_id'] ?? 'N/A') . "\n";
    echo "Campaign Ad ID: " . ($results['campaign_ad_id'] ?? 'N/A') . "\n";
    echo "Email Campaign Can See Ads: " . ($results['email_campaign_ads_found'] ? 'Yes' : 'No') . "\n";
    echo "AcceleratedAdz Accessible: " . ($results['acceleratedadz_accessible'] ? 'Yes' : 'No') . "\n";
    
    if (!empty($errors)) {
        echo "\nErrors:\n";
        foreach ($errors as $error) {
            echo "  - {$error}\n";
        }
    }
    
    if (!empty($results['acceleratedadz_url'])) {
        echo "\n=== Next Steps ===\n";
        echo "1. Open the AcceleratedAdz URL to view the ad:\n";
        echo "   " . $results['acceleratedadz_url'] . "\n\n";
        echo "2. Or open email campaigns page and use 'Insert Ad' button:\n";
        $emailCampaignsPath = url('client/email_campaigns.php');
        echo "   " . $baseUrl . $emailCampaignsPath . "\n\n";
        echo "3. The ad should be available for insertion in email campaigns.\n";
    }
    
    echo "\n=== Test Complete ===\n";
    
} catch (Exception $e) {
    echo "\n✗ ERROR: " . $e->getMessage() . "\n";
    echo "File: " . $e->getFile() . "\n";
    echo "Line: " . $e->getLine() . "\n";
    if (!empty($results)) {
        echo "\nPartial Results:\n";
        print_r($results);
    }
    exit(1);
}

