<?php
/**
 * Interactive Setup Wizard
 * Steps through each configuration and tests it
 */

session_start();

// Initialize step tracking
if (!isset($_SESSION['setup_step'])) {
    $_SESSION['setup_step'] = 1;
    $_SESSION['setup_results'] = [];
}

$step = isset($_GET['step']) ? (int)$_GET['step'] : $_SESSION['setup_step'];
$action = $_GET['action'] ?? '';

// Handle step actions
if ($action === 'test' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    
    switch ($step) {
        case 1:
            echo json_encode(testEnvFile());
            exit;
        case 2:
            echo json_encode(testDatabase());
            exit;
        case 3:
            echo json_encode(testGoogleAds());
            exit;
        case 4:
            echo json_encode(testFacebookAds());
            exit;
        case 5:
            echo json_encode(testEmail());
            exit;
        case 6:
            echo json_encode(finalizeSetup());
            exit;
    }
}

// Handle saving .env file
if ($action === 'save_env' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    echo json_encode(saveEnvFile($_POST));
    exit;
}

// Handle database schema import
if ($action === 'import_schema' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    echo json_encode(importDatabaseSchema());
    exit;
}

// Handle database clear/remove
if ($action === 'clear_database' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    echo json_encode(clearDatabase());
    exit;
}

// Handle running tests
if ($action === 'run_tests' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    echo json_encode(runTests());
    exit;
}

function testEnvFile() {
    $envPath = __DIR__ . '/.env';
    $envExamplePath = __DIR__ . '/env.example';
    
    if (file_exists($envPath)) {
        // Check if it has required values
        $content = file_get_contents($envPath);
        $hasDbConfig = strpos($content, 'DB_HOST=') !== false && 
                       strpos($content, 'DB_NAME=') !== false &&
                       strpos($content, 'DB_USER=') !== false &&
                       strpos($content, 'DB_PASS=') !== false;
        
        return [
            'success' => $hasDbConfig,
            'message' => $hasDbConfig 
                ? '✓ .env file exists and has database configuration'
                : '⚠ .env file exists but missing database configuration',
            'details' => 'File found at: ' . $envPath
        ];
    }
    
    return [
        'success' => false,
        'message' => '✗ .env file not found',
        'details' => 'Please create .env file from env.example'
    ];
}

function saveEnvFile($data) {
    $envPath = __DIR__ . '/.env';
    $examplePath = __DIR__ . '/env.example';
    
    // If .env exists, read it; otherwise start from example
    if (file_exists($envPath)) {
        $content = file_get_contents($envPath);
    } else {
        if (!file_exists($examplePath)) {
            return ['success' => false, 'message' => 'env.example file not found'];
        }
        $content = file_get_contents($examplePath);
    }
    
    // Map of keys to update
    $replacements = [
        'DB_HOST' => $data['db_host'] ?? null,
        'DB_NAME' => $data['db_name'] ?? null,
        'DB_USER' => $data['db_user'] ?? null,
        'DB_PASS' => $data['db_pass'] ?? null,
        'SECRET_KEY' => $data['secret_key'] ?? null,
        'GOOGLE_ADS_CLIENT_ID' => $data['google_client_id'] ?? null,
        'GOOGLE_ADS_CLIENT_SECRET' => $data['google_client_secret'] ?? null,
        'GOOGLE_ADS_DEVELOPER_TOKEN' => $data['google_developer_token'] ?? null,
        'FACEBOOK_APP_ID' => $data['facebook_app_id'] ?? null,
        'FACEBOOK_APP_SECRET' => $data['facebook_app_secret'] ?? null,
        'FACEBOOK_ACCESS_TOKEN' => $data['facebook_access_token'] ?? null,
        'EMAIL_SERVICE' => $data['email_service'] ?? null,
        'SMTP_HOST' => $data['smtp_host'] ?? null,
        'SMTP_PORT' => $data['smtp_port'] ?? null,
        'SMTP_USER' => $data['smtp_user'] ?? null,
        'SMTP_PASS' => $data['smtp_pass'] ?? null,
    ];
    
    // Handle SECRET_KEY special case - generate if empty
    if (isset($data['secret_key']) && empty($data['secret_key'])) {
        $replacements['SECRET_KEY'] = bin2hex(random_bytes(32));
    }
    
    // Update each key in the content
    foreach ($replacements as $key => $value) {
        if ($value !== null) {
            // Replace existing line or add new one
            if (preg_match("/^$key=.*/m", $content)) {
                $content = preg_replace("/^$key=.*/m", "$key=$value", $content);
            } else {
                // Add new line if key doesn't exist
                $content .= "\n$key=$value";
            }
        }
    }
    
    // Write file
    if (file_put_contents($envPath, $content) !== false) {
        // Set secure permissions (600 = owner read/write only)
        @chmod($envPath, 0600);
        return ['success' => true, 'message' => '✓ .env file saved successfully'];
    }
    
    return ['success' => false, 'message' => '✗ Failed to save .env file. Check file permissions.'];
}

function testDatabase() {
    // Enhanced error handling
    try {
        // Check if .env exists
        if (!file_exists(__DIR__ . '/.env')) {
            return [
                'success' => false,
                'message' => '✗ .env file not found',
                'details' => 'Please complete Step 1 first to create the .env file with database credentials.',
                'needsEnv' => true
            ];
        }
        
        // Reload config after .env is created
        require_once __DIR__ . '/config/env_loader.php';
        require_once __DIR__ . '/config/database.php';
        
        // Validate configuration constants
        if (!defined('DB_HOST') || !defined('DB_NAME') || !defined('DB_USER') || !defined('DB_PASS')) {
            return [
                'success' => false,
                'message' => '✗ Database configuration incomplete',
                'details' => 'Missing required database configuration in .env file (DB_HOST, DB_NAME, DB_USER, DB_PASS).'
            ];
        }
        
        // Test connection with detailed error handling
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $pdo = new PDO($dsn, DB_USER, DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_TIMEOUT => 10,
                PDO::ATTR_PERSISTENT => false
            ]);
            
            // Test query
            $stmt = $pdo->query("SELECT 1");
            $stmt->fetch();
            
            // Get database version
            $version = $pdo->query("SELECT VERSION()")->fetchColumn();
            
            // Check if tables exist
            try {
                $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
                $hasTables = count($tables) > 0;
                
                $details = 'Connected to database "' . DB_NAME . '" on ' . DB_HOST . ' (MySQL ' . $version . ').';
                $details .= $hasTables 
                    ? ' Found ' . count($tables) . ' table(s).'
                    : ' No tables found - schema needs to be imported.';
                
                return [
                    'success' => true,
                    'message' => '✓ Database connection successful',
                    'details' => $details,
                    'needsSchema' => !$hasTables,
                    'tableCount' => count($tables),
                    'dbVersion' => $version
                ];
            } catch (PDOException $e) {
                return [
                    'success' => false,
                    'message' => '✗ Cannot query database tables',
                    'details' => 'Connection successful but cannot list tables. Error: ' . $e->getMessage()
                ];
            }
        } catch (PDOException $e) {
            // Provide helpful error messages
            $errorCode = $e->getCode();
            $errorMsg = $e->getMessage();
            
            $details = $errorMsg;
            
            if ($errorCode == 1045) {
                $details .= ' Check your DB_USER and DB_PASS in .env file.';
            } elseif ($errorCode == 1049) {
                $details .= ' Database "' . DB_NAME . '" does not exist. Please create it first.';
            } elseif ($errorCode == 2002 || strpos($errorMsg, 'Connection refused') !== false) {
                $details .= ' Cannot connect to database server at ' . DB_HOST . '. Is MySQL running?';
            } elseif ($errorCode == 2003 || strpos($errorMsg, 'timed out') !== false) {
                $details .= ' Connection to ' . DB_HOST . ' timed out. Check firewall settings.';
            }
            
            return [
                'success' => false,
                'message' => '✗ Database connection failed',
                'details' => $details,
                'errorCode' => $errorCode
            ];
        }
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => '✗ Unexpected error',
            'details' => 'Error: ' . $e->getMessage() . ' (Line: ' . $e->getLine() . ')'
        ];
    }
}

function importDatabaseSchema() {
    $schemaPath = __DIR__ . '/database/schema_main.sql';
    
    if (!file_exists($schemaPath)) {
        return [
            'success' => false, 
            'message' => '✗ Schema file not found',
            'details' => 'Expected file at: ' . $schemaPath . '. Please ensure the schema file exists.'
        ];
    }
    
    // Reload config
    if (file_exists(__DIR__ . '/.env')) {
        require_once __DIR__ . '/config/env_loader.php';
    }
    require_once __DIR__ . '/config/database.php';
    
    try {
        // Validate database connection first
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
        $pdo = new PDO($dsn, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_TIMEOUT => 30 // Longer timeout for large schemas
        ]);
        
        // Read SQL file with error handling
        $sql = @file_get_contents($schemaPath);
        if ($sql === false) {
            throw new Exception("Failed to read schema file. Check file permissions.");
        }
        
        if (empty(trim($sql))) {
            throw new Exception("Schema file is empty.");
        }
        
        // Remove CREATE DATABASE and USE statements
        $sql = preg_replace('/CREATE DATABASE[^;]+;/i', '', $sql);
        $sql = preg_replace('/USE[^;]+;/i', '', $sql);
        
        // Split into individual statements with better parsing
        $statements = array_filter(
            array_map('trim', explode(';', $sql)),
            function($stmt) {
                $stmt = trim($stmt);
                return !empty($stmt) && 
                       !preg_match('/^--/', $stmt) &&
                       !preg_match('/^\/\*/', $stmt) &&
                       strlen($stmt) > 5;
            }
        );
        
        if (empty($statements)) {
            throw new Exception("No valid SQL statements found in schema file.");
        }
        
        $pdo->beginTransaction();
        $successCount = 0;
        $errorCount = 0;
        $warnings = [];
        
        foreach ($statements as $index => $stmt) {
            try {
                if (!empty(trim($stmt))) {
                    $pdo->exec($stmt);
                    $successCount++;
                }
            } catch (PDOException $e) {
                $errorMsg = $e->getMessage();
                // Some errors are okay (like table already exists)
                if (stripos($errorMsg, 'already exists') !== false || 
                    stripos($errorMsg, 'duplicate') !== false) {
                    // Skip duplicate/already exists errors
                    continue;
                } else {
                    $errorCount++;
                    $warnings[] = "Statement " . ($index + 1) . ": " . substr($errorMsg, 0, 100);
                }
            }
        }
        
        $pdo->commit();
        
        // Verify tables were created
        $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
        
        // Create default admin if doesn't exist
        try {
            if (in_array('users', $tables)) {
                $checkAdmin = $pdo->query("SELECT COUNT(*) FROM users WHERE role = 'admin'")->fetchColumn();
                if ($checkAdmin == 0) {
                    $email = 'admin@marketingplatform.com';
                    $hash = password_hash('admin123', PASSWORD_BCRYPT);
                    $pdo->prepare("INSERT INTO users (email, password_hash, business_name, contact_name, role, status) VALUES (?, ?, 'Marketing Platform Admin', 'System Administrator', 'admin', 'active')")
                        ->execute([$email, $hash]);
                }
            }
        } catch (PDOException $e) {
            $warnings[] = "Admin user creation: " . $e->getMessage();
        }
        
        $details = "Executed $successCount statements. Created " . count($tables) . " table(s).";
        if ($errorCount > 0) {
            $details .= " $errorCount error(s) encountered.";
        }
        if (!empty($warnings)) {
            $details .= " Warnings: " . implode('; ', array_slice($warnings, 0, 3));
        }
        
        return [
            'success' => true,
            'message' => '✓ Database schema imported successfully',
            'details' => $details,
            'tableCount' => count($tables),
            'runTests' => true // Flag to suggest running tests
        ];
    } catch (PDOException $e) {
        return [
            'success' => false,
            'message' => '✗ Database error during schema import',
            'details' => 'PDO Error: ' . $e->getMessage() . ' (Code: ' . $e->getCode() . ')'
        ];
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => '✗ Failed to import schema',
            'details' => $e->getMessage()
        ];
    }
}

function testGoogleAds() {
    if (!file_exists(__DIR__ . '/.env')) {
        return ['success' => false, 'message' => 'Please complete .env setup first'];
    }
    
    require_once __DIR__ . '/config/env_loader.php';
    require_once __DIR__ . '/config/google_ads_config.php';
    
    $hasClientId = !empty(GOOGLE_ADS_CLIENT_ID);
    $hasClientSecret = !empty(GOOGLE_ADS_CLIENT_SECRET);
    $hasDeveloperToken = !empty(GOOGLE_ADS_DEVELOPER_TOKEN);
    
    if (!$hasClientId || !$hasClientSecret) {
        return [
            'success' => false,
            'message' => '⚠ Google Ads not configured',
            'details' => 'Client ID and Secret are required. This is optional - you can skip for now.',
            'optional' => true
        ];
    }
    
    // Try to validate credentials (basic check)
    return [
        'success' => true,
        'message' => '✓ Google Ads credentials configured',
        'details' => $hasDeveloperToken 
            ? 'Client ID, Secret, and Developer Token are set'
            : 'Client ID and Secret are set (Developer Token pending)'
    ];
}

function testFacebookAds() {
    if (!file_exists(__DIR__ . '/.env')) {
        return ['success' => false, 'message' => 'Please complete .env setup first'];
    }
    
    require_once __DIR__ . '/config/env_loader.php';
    require_once __DIR__ . '/config/facebook_config.php';
    
    $hasAppId = !empty(FACEBOOK_APP_ID);
    $hasAppSecret = !empty(FACEBOOK_APP_SECRET);
    $hasAccessToken = !empty(FACEBOOK_ACCESS_TOKEN);
    
    if (!$hasAppId || !$hasAppSecret) {
        return [
            'success' => false,
            'message' => '⚠ Facebook Ads not configured',
            'details' => 'App ID and Secret are required. This is optional - you can skip for now.',
            'optional' => true
        ];
    }
    
    return [
        'success' => true,
        'message' => '✓ Facebook Ads credentials configured',
        'details' => $hasAccessToken 
            ? 'App ID, Secret, and Access Token are set'
            : 'App ID and Secret are set (Access Token pending)'
    ];
}

function testEmail() {
    if (!file_exists(__DIR__ . '/.env')) {
        return ['success' => false, 'message' => 'Please complete .env setup first'];
    }
    
    require_once __DIR__ . '/config/env_loader.php';
    require_once __DIR__ . '/config/email_config.php';
    
    $service = defined('EMAIL_SERVICE') ? EMAIL_SERVICE : 'smtp';
    
    if ($service === 'smtp') {
        $hasHost = !empty(SMTP_HOST);
        $hasUser = !empty(SMTP_USER);
        $hasPass = !empty(SMTP_PASS);
        
        if (!$hasHost || !$hasUser || !$hasPass) {
            return [
                'success' => false,
                'message' => '⚠ SMTP not fully configured',
                'details' => 'SMTP Host, User, and Password are required. This is optional - you can skip for now.',
                'optional' => true
            ];
        }
        
        // Try connection (without actually sending)
        return [
            'success' => true,
            'message' => '✓ SMTP configuration complete',
            'details' => 'SMTP settings are configured. Test email sending from the app.'
        ];
    }
    
    return [
        'success' => false,
        'message' => 'Email service not configured',
        'optional' => true
    ];
}

function clearDatabase() {
    try {
        // Reload config
        if (file_exists(__DIR__ . '/.env')) {
            require_once __DIR__ . '/config/env_loader.php';
        }
        require_once __DIR__ . '/config/database.php';
        
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
        $pdo = new PDO($dsn, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ]);
        
        // Get all tables
        $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
        
        if (empty($tables)) {
            return [
                'success' => true,
                'message' => '✓ Database is already empty',
                'details' => 'No tables found to remove.'
            ];
        }
        
        // Disable foreign key checks
        $pdo->exec("SET FOREIGN_KEY_CHECKS = 0");
        
        $droppedCount = 0;
        $errors = [];
        
        foreach ($tables as $table) {
            try {
                $pdo->exec("DROP TABLE IF EXISTS `$table`");
                $droppedCount++;
            } catch (PDOException $e) {
                $errors[] = "Failed to drop table $table: " . $e->getMessage();
            }
        }
        
        // Re-enable foreign key checks
        $pdo->exec("SET FOREIGN_KEY_CHECKS = 1");
        
        return [
            'success' => true,
            'message' => '✓ Database cleared successfully',
            'details' => "Dropped $droppedCount table(s). " . (!empty($errors) ? 'Warnings: ' . implode(', ', $errors) : '')
        ];
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => '✗ Failed to clear database',
            'details' => $e->getMessage()
        ];
    }
}

function runTests() {
    try {
        // Check if we're on localhost:8081 (test server) or localhost
        $isTestServer = (isset($_SERVER['HTTP_HOST']) && (strpos($_SERVER['HTTP_HOST'], 'localhost:8081') !== false || strpos($_SERVER['HTTP_HOST'], 'localhost') !== false)) ||
                        (isset($_SERVER['SERVER_NAME']) && strpos($_SERVER['SERVER_NAME'], 'localhost') !== false);
        
        if (!$isTestServer) {
            return [
                'success' => false,
                'message' => '⚠ Tests can only run on localhost',
                'details' => 'For security, tests are disabled on production servers.',
                'optional' => true
            ];
        }
        
        // PHP executable path
        $phpExe = 'C:\\xampp\\php\\php.exe';
        if (!file_exists($phpExe)) {
            // Fallback to system PHP
            $phpExe = 'php';
        }
        
        // Reload config
        if (file_exists(__DIR__ . '/.env')) {
            require_once __DIR__ . '/config/env_loader.php';
        }
        require_once __DIR__ . '/config/database.php';
        
        // Check if test configuration exists
        $testConfigPath = __DIR__ . '/config/database.test.php';
        if (!file_exists($testConfigPath)) {
            return [
                'success' => false,
                'message' => '✗ Test configuration not found',
                'details' => 'Please create config/database.test.php first.'
            ];
        }
        
        $results = [
            'tests' => [],
            'passed' => 0,
            'failed' => 0,
            'total' => 0
        ];
        
        // Test 1: Database Connection (direct test, no need for test config)
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $pdo = new PDO($dsn, DB_USER, DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_TIMEOUT => 5
            ]);
            $pdo->query("SELECT 1");
            $results['tests'][] = ['name' => 'Database Connection', 'status' => 'passed', 'message' => 'Connected successfully'];
            $results['passed']++;
        } catch (Exception $e) {
            $results['tests'][] = ['name' => 'Database Connection', 'status' => 'failed', 'message' => $e->getMessage()];
            $results['failed']++;
        }
        $results['total']++;
        
        // Test 2: Check Tables Exist
        try {
            $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
            $requiredTables = ['users', 'subscription_tiers', 'campaigns', 'subscriptions'];
            $missing = array_diff($requiredTables, $tables);
            
            if (empty($missing)) {
                $results['tests'][] = ['name' => 'Required Tables', 'status' => 'passed', 'message' => 'All required tables exist (' . count($tables) . ' total)'];
                $results['passed']++;
            } else {
                $results['tests'][] = ['name' => 'Required Tables', 'status' => 'failed', 'message' => 'Missing tables: ' . implode(', ', $missing)];
                $results['failed']++;
            }
        } catch (Exception $e) {
            $results['tests'][] = ['name' => 'Required Tables', 'status' => 'failed', 'message' => $e->getMessage()];
            $results['failed']++;
        }
        $results['total']++;
        
        // Test 3: Admin User Exists
        try {
            $stmt = $pdo->query("SELECT COUNT(*) FROM users WHERE role = 'admin'");
            $adminCount = $stmt->fetchColumn();
            
            if ($adminCount > 0) {
                $results['tests'][] = ['name' => 'Admin User', 'status' => 'passed', 'message' => "Found $adminCount admin user(s)"];
                $results['passed']++;
            } else {
                $results['tests'][] = ['name' => 'Admin User', 'status' => 'warning', 'message' => 'No admin user found'];
                $results['passed']++; // Warning, not failure
            }
        } catch (Exception $e) {
            $results['tests'][] = ['name' => 'Admin User', 'status' => 'failed', 'message' => $e->getMessage()];
            $results['failed']++;
        }
        $results['total']++;
        
        // Test 4: Subscription Tiers
        try {
            $stmt = $pdo->query("SELECT COUNT(*) FROM subscription_tiers");
            $tierCount = $stmt->fetchColumn();
            
            if ($tierCount >= 3) {
                $results['tests'][] = ['name' => 'Subscription Tiers', 'status' => 'passed', 'message' => "Found $tierCount tier(s)"];
                $results['passed']++;
            } else {
                $results['tests'][] = ['name' => 'Subscription Tiers', 'status' => 'warning', 'message' => "Only $tierCount tier(s) found (expected 3)"];
                $results['passed']++;
            }
        } catch (Exception $e) {
            $results['tests'][] = ['name' => 'Subscription Tiers', 'status' => 'failed', 'message' => $e->getMessage()];
            $results['failed']++;
        }
        $results['total']++;
        
        // Test 5: Foreign Key Constraints
        try {
            $pdo->exec("SET FOREIGN_KEY_CHECKS = 0");
            $pdo->exec("SET FOREIGN_KEY_CHECKS = 1");
            $results['tests'][] = ['name' => 'Foreign Key Support', 'status' => 'passed', 'message' => 'Foreign keys are supported'];
            $results['passed']++;
        } catch (Exception $e) {
            $results['tests'][] = ['name' => 'Foreign Key Support', 'status' => 'failed', 'message' => $e->getMessage()];
            $results['failed']++;
        }
        $results['total']++;
        
        $success = $results['failed'] == 0;
        
        return [
            'success' => $success,
            'message' => $success 
                ? '✓ All tests passed (' . $results['passed'] . '/' . $results['total'] . ')'
                : '⚠ Some tests failed (' . $results['passed'] . '/' . $results['total'] . ' passed)',
            'details' => $results,
            'testResults' => true
        ];
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => '✗ Test execution failed',
            'details' => $e->getMessage()
        ];
    }
}

function finalizeSetup() {
    // Mark setup as complete
    $setupFile = __DIR__ . '/.setup_complete';
    file_put_contents($setupFile, date('c') . "\n");
    
    return [
        'success' => true,
        'message' => '✓ Setup completed successfully!',
        'details' => 'Your Marketing Platform is ready to use.'
    ];
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Setup Wizard - Marketing Platform</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        .step-card { transition: all 0.3s; }
        .step-card.active { border-color: #0d6efd; box-shadow: 0 0 0 0.2rem rgba(13, 110, 253, 0.25); }
        .step-card.completed { border-color: #198754; background: #f8fff9; }
        .step-card.error { border-color: #dc3545; background: #fff8f8; }
        .step-number { width: 40px; height: 40px; border-radius: 50%; display: inline-flex; align-items: center; justify-content: center; font-weight: bold; }
        .progress-bar-container { height: 8px; }
        .test-result { padding: 12px; border-radius: 8px; margin-top: 10px; }
        .test-success { background: #d1e7dd; color: #0f5132; border: 1px solid #badbcc; }
        .test-error { background: #f8d7da; color: #842029; border: 1px solid #f5c2c7; }
        .test-warning { background: #fff3cd; color: #664d03; border: 1px solid #ffc107; }
        .spinner-border-sm { width: 1rem; height: 1rem; }
        .form-label { font-weight: 600; margin-bottom: 8px; }
        .optional-badge { font-size: 0.75rem; }
    </style>
</head>
<body class="bg-light">
    <div class="container py-5">
        <div class="row justify-content-center">
            <div class="col-lg-10">
                <div class="card shadow-lg border-0">
                    <div class="card-header bg-primary text-white">
                        <h2 class="mb-0"><i class="bi bi-magic"></i> Setup Wizard</h2>
                        <p class="mb-0 text-white-50">Follow these steps to configure your Marketing Platform</p>
                    </div>
                    <div class="card-body p-4">
                        <!-- Progress Bar -->
                        <div class="mb-4">
                            <div class="d-flex justify-content-between mb-2">
                                <small class="text-muted">Progress</small>
                                <small class="text-muted" id="progress-text">Step <?= $step ?> of 6</small>
                            </div>
                            <div class="progress progress-bar-container">
                                <div class="progress-bar" role="progressbar" style="width: <?= ($step / 6) * 100 ?>%"></div>
                            </div>
                        </div>

                        <!-- Step 1: .env File -->
                        <div class="step-card card mb-3 <?= $step >= 1 ? 'active' : '' ?>" id="step-1">
                            <div class="card-body">
                                <div class="d-flex align-items-center mb-3">
                                    <span class="step-number bg-primary text-white me-3">1</span>
                                    <div class="flex-grow-1">
                                        <h5 class="mb-0">Create .env File <span class="optional-badge badge bg-secondary">Required</span></h5>
                                        <small class="text-muted">Secure configuration file for passwords and API keys</small>
                                    </div>
                                </div>
                                <div id="step-1-content">
                                    <?php if ($step == 1): ?>
                                        <form id="env-form" class="mt-3">
                                            <div class="row">
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Database Host</label>
                                                    <input type="text" name="db_host" class="form-control" value="localhost" required>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Database Name</label>
                                                    <input type="text" name="db_name" class="form-control" value="mindsybq_market" required>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Database User</label>
                                                    <input type="text" name="db_user" class="form-control" value="mindsybq_appuser" required>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Database Password</label>
                                                    <input type="password" name="db_pass" class="form-control" required>
                                                </div>
                                                <div class="col-12 mb-3">
                                                    <label class="form-label">Secret Key <small class="text-muted">(Auto-generated if empty)</small></label>
                                                    <input type="text" name="secret_key" class="form-control" placeholder="Leave empty to auto-generate">
                                                </div>
                                            </div>
                                            <button type="submit" class="btn btn-primary">
                                                <i class="bi bi-save"></i> Create .env File
                                            </button>
                                        </form>
                                    <?php endif; ?>
                                    <div id="step-1-result"></div>
                                </div>
                            </div>
                        </div>

                        <!-- Step 2: Database -->
                        <div class="step-card card mb-3" id="step-2">
                            <div class="card-body">
                                <div class="d-flex align-items-center mb-3">
                                    <span class="step-number bg-secondary text-white me-3">2</span>
                                    <div class="flex-grow-1">
                                        <h5 class="mb-0">Test Database Connection <span class="optional-badge badge bg-danger">Required</span></h5>
                                        <small class="text-muted">Verify database connection and import schema</small>
                                    </div>
                                </div>
                                <div id="step-2-content">
                                    <?php if ($step == 2): ?>
                                        <div class="d-flex gap-2 mb-3">
                                            <button type="button" class="btn btn-primary" onclick="testStep(2)">
                                                <i class="bi bi-database-check"></i> Test Connection
                                            </button>
                                            <button type="button" class="btn btn-danger" onclick="clearDatabase()" id="clear-db-btn" style="display:none;">
                                                <i class="bi bi-trash"></i> Clear Database
                                            </button>
                                        </div>
                                        <div id="step-2-result"></div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <!-- Step 3: Google Ads -->
                        <div class="step-card card mb-3" id="step-3">
                            <div class="card-body">
                                <div class="d-flex align-items-center mb-3">
                                    <span class="step-number bg-secondary text-white me-3">3</span>
                                    <div class="flex-grow-1">
                                        <h5 class="mb-0">Google Ads API <span class="optional-badge badge bg-info">Optional</span></h5>
                                        <small class="text-muted">Configure Google Ads integration</small>
                                    </div>
                                </div>
                                <div id="step-3-content">
                                    <?php if ($step == 3): ?>
                                        <form id="google-form" class="mt-3">
                                            <div class="row">
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Client ID</label>
                                                    <input type="text" name="google_client_id" class="form-control">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">Client Secret</label>
                                                    <input type="password" name="google_client_secret" class="form-control">
                                                </div>
                                                <div class="col-12 mb-3">
                                                    <label class="form-label">Developer Token</label>
                                                    <input type="text" name="google_developer_token" class="form-control">
                                                </div>
                                            </div>
                                            <button type="submit" class="btn btn-primary me-2">
                                                <i class="bi bi-save"></i> Save & Test
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary" onclick="skipStep(3)">
                                                Skip for Now
                                            </button>
                                        </form>
                                        <div id="step-3-result"></div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <!-- Step 4: Facebook Ads -->
                        <div class="step-card card mb-3" id="step-4">
                            <div class="card-body">
                                <div class="d-flex align-items-center mb-3">
                                    <span class="step-number bg-secondary text-white me-3">4</span>
                                    <div class="flex-grow-1">
                                        <h5 class="mb-0">Facebook Ads API <span class="optional-badge badge bg-info">Optional</span></h5>
                                        <small class="text-muted">Configure Facebook Marketing API</small>
                                    </div>
                                </div>
                                <div id="step-4-content">
                                    <?php if ($step == 4): ?>
                                        <form id="facebook-form" class="mt-3">
                                            <div class="row">
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">App ID</label>
                                                    <input type="text" name="facebook_app_id" class="form-control">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">App Secret</label>
                                                    <input type="password" name="facebook_app_secret" class="form-control">
                                                </div>
                                                <div class="col-12 mb-3">
                                                    <label class="form-label">Access Token</label>
                                                    <input type="text" name="facebook_access_token" class="form-control">
                                                </div>
                                            </div>
                                            <button type="submit" class="btn btn-primary me-2">
                                                <i class="bi bi-save"></i> Save & Test
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary" onclick="skipStep(4)">
                                                Skip for Now
                                            </button>
                                        </form>
                                        <div id="step-4-result"></div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <!-- Step 5: Email -->
                        <div class="step-card card mb-3" id="step-5">
                            <div class="card-body">
                                <div class="d-flex align-items-center mb-3">
                                    <span class="step-number bg-secondary text-white me-3">5</span>
                                    <div class="flex-grow-1">
                                        <h5 class="mb-0">Email Configuration <span class="optional-badge badge bg-info">Optional</span></h5>
                                        <small class="text-muted">Set up SMTP or email service</small>
                                    </div>
                                </div>
                                <div id="step-5-content">
                                    <?php if ($step == 5): ?>
                                        <form id="email-form" class="mt-3">
                                            <div class="mb-3">
                                                <label class="form-label">Email Service</label>
                                                <select name="email_service" class="form-select">
                                                    <option value="smtp">SMTP</option>
                                                    <option value="sendgrid">SendGrid</option>
                                                    <option value="mailchimp">Mailchimp</option>
                                                </select>
                                            </div>
                                            <div class="row">
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">SMTP Host</label>
                                                    <input type="text" name="smtp_host" class="form-control" placeholder="smtp.gmail.com">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">SMTP Port</label>
                                                    <input type="number" name="smtp_port" class="form-control" value="587">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">SMTP User</label>
                                                    <input type="text" name="smtp_user" class="form-control">
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <label class="form-label">SMTP Password</label>
                                                    <input type="password" name="smtp_pass" class="form-control">
                                                </div>
                                            </div>
                                            <button type="submit" class="btn btn-primary me-2">
                                                <i class="bi bi-save"></i> Save & Test
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary" onclick="skipStep(5)">
                                                Skip for Now
                                            </button>
                                        </form>
                                        <div id="step-5-result"></div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <!-- Step 6: Complete -->
                        <div class="step-card card mb-3" id="step-6">
                            <div class="card-body">
                                <div class="d-flex align-items-center mb-3">
                                    <span class="step-number bg-secondary text-white me-3">6</span>
                                    <div class="flex-grow-1">
                                        <h5 class="mb-0">Finalize Setup</h5>
                                        <small class="text-muted">Complete the setup process</small>
                                    </div>
                                </div>
                                <div id="step-6-content">
                                    <?php if ($step == 6): ?>
                                        <button type="button" class="btn btn-success btn-lg" onclick="testStep(6)">
                                            <i class="bi bi-check-circle"></i> Complete Setup
                                        </button>
                                        <div id="step-6-result"></div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <!-- Navigation -->
                        <div class="d-flex justify-content-between mt-4">
                            <button type="button" class="btn btn-outline-secondary" onclick="prevStep()" id="prev-btn" style="display:none;">
                                <i class="bi bi-arrow-left"></i> Previous
                            </button>
                            <button type="button" class="btn btn-primary" onclick="nextStep()" id="next-btn" style="display:none;">
                                Next <i class="bi bi-arrow-right"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let currentStep = <?= $step ?>;
        const totalSteps = 6;

        function updateProgress() {
            const percent = (currentStep / totalSteps) * 100;
            document.querySelector('.progress-bar').style.width = percent + '%';
            document.getElementById('progress-text').textContent = `Step ${currentStep} of ${totalSteps}`;
            
            // Update step cards
            for (let i = 1; i <= totalSteps; i++) {
                const card = document.getElementById('step-' + i);
                card.classList.remove('active', 'completed', 'error');
                if (i < currentStep) {
                    card.classList.add('completed');
                } else if (i === currentStep) {
                    card.classList.add('active');
                }
            }
            
            // Show/hide navigation
            document.getElementById('prev-btn').style.display = currentStep > 1 ? 'block' : 'none';
            document.getElementById('next-btn').style.display = currentStep < totalSteps ? 'block' : 'none';
        }

        function nextStep() {
            if (currentStep < totalSteps) {
                currentStep++;
                window.location.href = '?step=' + currentStep;
            }
        }

        function prevStep() {
            if (currentStep > 1) {
                currentStep--;
                window.location.href = '?step=' + currentStep;
            }
        }

        function skipStep(step) {
            if (step < totalSteps) {
                currentStep = step + 1;
                window.location.href = '?step=' + currentStep;
            }
        }

        async function testStep(step) {
            const resultDiv = document.getElementById('step-' + step + '-result');
            resultDiv.innerHTML = '<div class="test-result"><i class="spinner-border spinner-border-sm"></i> Testing...</div>';
            
            try {
                const response = await fetch(`?step=${step}&action=test`, {
                    method: 'POST'
                });
                const result = await response.json();
                
                let className = result.success ? 'test-success' : (result.optional ? 'test-warning' : 'test-error');
                let icon = result.success ? '✓' : (result.optional ? '⚠' : '✗');
                
                let html = `<div class="test-result ${className}">
                    <strong>${icon} ${result.message}</strong>
                    ${result.details ? '<br><small>' + result.details + '</small>' : ''}
                    ${result.needsSchema ? '<br><button class="btn btn-sm btn-primary mt-2" onclick="importSchema()"><i class="bi bi-database-add"></i> Import Database Schema</button>' : ''}`;
                
                // Show clear database button if tables exist
                if (result.success && result.tableCount > 0) {
                    document.getElementById('clear-db-btn').style.display = 'inline-block';
                } else {
                    document.getElementById('clear-db-btn').style.display = 'none';
                }
                
                html += '</div>';
                resultDiv.innerHTML = html;
                
                if (result.success && step === 6) {
                    setTimeout(() => {
                        window.location.href = 'admin/login.php';
                    }, 2000);
                }
            } catch (error) {
                resultDiv.innerHTML = `<div class="test-result test-error"><strong>✗ Error:</strong> ${error.message}</div>`;
            }
        }

        async function importSchema() {
            const resultDiv = document.getElementById('step-2-result');
            resultDiv.innerHTML += '<div class="test-result"><i class="spinner-border spinner-border-sm"></i> Importing schema...</div>';
            
            try {
                const response = await fetch('?step=2&action=import_schema', {
                    method: 'POST'
                });
                const result = await response.json();
                
                let className = result.success ? 'test-success' : 'test-error';
                let icon = result.success ? '✓' : '✗';
                
                let html = `<div class="test-result ${className}">
                    <strong>${icon} ${result.message}</strong>
                    ${result.details ? '<br><small>' + result.details + '</small>' : ''}`;
                
                // Show clear database button if tables exist
                if (result.success && result.tableCount > 0) {
                    document.getElementById('clear-db-btn').style.display = 'inline-block';
                }
                
                // Suggest running tests if on localhost:8081
                if (result.success && result.runTests && window.location.hostname === 'localhost' && (window.location.port === '8081' || window.location.port === '')) {
                    html += `<br><button class="btn btn-sm btn-success mt-2" onclick="runTests()">
                        <i class="bi bi-check-circle"></i> Run Tests After Installation
                    </button>`;
                }
                
                html += '</div>';
                resultDiv.innerHTML += html;
            } catch (error) {
                resultDiv.innerHTML += `<div class="test-result test-error"><strong>✗ Error:</strong> ${error.message}</div>`;
            }
        }
        
        async function clearDatabase() {
            if (!confirm('⚠️ WARNING: This will DELETE ALL TABLES and DATA from the database. This action cannot be undone!\n\nAre you absolutely sure you want to proceed?')) {
                return;
            }
            
            const resultDiv = document.getElementById('step-2-result');
            resultDiv.innerHTML += '<div class="test-result"><i class="spinner-border spinner-border-sm"></i> Clearing database...</div>';
            
            try {
                const response = await fetch('?step=2&action=clear_database', {
                    method: 'POST'
                });
                const result = await response.json();
                
                let className = result.success ? 'test-success' : 'test-error';
                let icon = result.success ? '✓' : '✗';
                
                resultDiv.innerHTML += `<div class="test-result ${className}">
                    <strong>${icon} ${result.message}</strong>
                    ${result.details ? '<br><small>' + result.details + '</small>' : ''}
                </div>`;
                
                // Hide clear button if database is empty
                if (result.success && result.message.includes('already empty')) {
                    document.getElementById('clear-db-btn').style.display = 'none';
                }
            } catch (error) {
                resultDiv.innerHTML += `<div class="test-result test-error"><strong>✗ Error:</strong> ${error.message}</div>`;
            }
        }
        
        async function runTests() {
            const resultDiv = document.getElementById('step-2-result');
            const originalHtml = resultDiv.innerHTML;
            resultDiv.innerHTML += '<div class="test-result"><i class="spinner-border spinner-border-sm"></i> Running tests...</div>';
            
            try {
                const response = await fetch('?step=2&action=run_tests', {
                    method: 'POST'
                });
                const result = await response.json();
                
                if (result.testResults) {
                    // Display detailed test results
                    let html = '<div class="test-result ' + (result.success ? 'test-success' : 'test-warning') + ' mt-3">';
                    html += '<strong>' + result.message + '</strong><br><br>';
                    html += '<div class="table-responsive"><table class="table table-sm table-bordered">';
                    html += '<thead><tr><th>Test</th><th>Status</th><th>Message</th></tr></thead><tbody>';
                    
                    result.details.tests.forEach(function(test) {
                        let statusBadge = '';
                        if (test.status === 'passed') {
                            statusBadge = '<span class="badge bg-success">✓ Passed</span>';
                        } else if (test.status === 'failed') {
                            statusBadge = '<span class="badge bg-danger">✗ Failed</span>';
                        } else {
                            statusBadge = '<span class="badge bg-warning">⚠ Warning</span>';
                        }
                        
                        html += '<tr><td>' + test.name + '</td><td>' + statusBadge + '</td><td><small>' + test.message + '</small></td></tr>';
                    });
                    
                    html += '</tbody></table></div>';
                    html += '<div class="mt-2"><strong>Summary:</strong> ' + result.details.passed + ' passed, ' + result.details.failed + ' failed out of ' + result.details.total + ' tests.</div>';
                    html += '</div>';
                    
                    resultDiv.innerHTML = originalHtml + html;
                } else {
                    let className = result.success ? 'test-success' : (result.optional ? 'test-warning' : 'test-error');
                    let icon = result.success ? '✓' : (result.optional ? '⚠' : '✗');
                    
                    resultDiv.innerHTML += `<div class="test-result ${className}">
                        <strong>${icon} ${result.message}</strong>
                        ${result.details ? '<br><small>' + result.details + '</small>' : ''}
                    </div>`;
                }
            } catch (error) {
                resultDiv.innerHTML += `<div class="test-result test-error"><strong>✗ Error:</strong> ${error.message}</div>`;
            }
        }

        // Form handlers
        document.getElementById('env-form')?.addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            
            try {
                const response = await fetch('?action=save_env', {
                    method: 'POST',
                    body: formData
                });
                const result = await response.json();
                
                const resultDiv = document.getElementById('step-1-result');
                let className = result.success ? 'test-success' : 'test-error';
                resultDiv.innerHTML = `<div class="test-result ${className}"><strong>${result.message}</strong></div>`;
                
                if (result.success) {
                    setTimeout(() => testStep(1), 500);
                    setTimeout(() => nextStep(), 1500);
                }
            } catch (error) {
                alert('Error: ' + error.message);
            }
        });

        document.getElementById('google-form')?.addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            
            try {
                await fetch('?action=save_env', {
                    method: 'POST',
                    body: formData
                });
                await testStep(3);
            } catch (error) {
                alert('Error: ' + error.message);
            }
        });

        document.getElementById('facebook-form')?.addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            
            try {
                await fetch('?action=save_env', {
                    method: 'POST',
                    body: formData
                });
                await testStep(4);
            } catch (error) {
                alert('Error: ' + error.message);
            }
        });

        document.getElementById('email-form')?.addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(e.target);
            
            try {
                await fetch('?action=save_env', {
                    method: 'POST',
                    body: formData
                });
                await testStep(5);
            } catch (error) {
                alert('Error: ' + error.message);
            }
        });

        // Initialize
        updateProgress();
        
        // Auto-test on page load for current step
        if (currentStep === 2 || currentStep === 3 || currentStep === 4 || currentStep === 5) {
            setTimeout(() => testStep(currentStep), 500);
        }
    </script>
</body>
</html>

