<?php
/**
 * Setup Wizard
 * - Tests DB connection
 * - Runs schema if missing
 * - Ensures default admin exists
 * - Marks setup complete
 */

require_once __DIR__ . '/includes/functions.php';

$errors = [];
$messages = [];
$dbOk = false;
$schemaOk = false;
$adminOk = false;

// Try DB connection
try {
    $db = getDB();
    $dbOk = true;
    $messages[] = 'Database connection successful.';
} catch (Exception $e) {
    $errors[] = 'Database connection failed: ' . $e->getMessage();
}

if ($dbOk) {
    // Check required tables
    $requiredTables = ['users', 'subscription_tiers'];
    $missing = [];
    foreach ($requiredTables as $table) {
        try {
            $db->query('SELECT 1 FROM ' . $table . ' LIMIT 1');
        } catch (Exception $e) {
            $missing[] = $table;
        }
    }

    if (!empty($missing)) {
        // Run unified schema_main.sql
        $schemaFile = __DIR__ . '/database/schema_main.sql';
        if (file_exists($schemaFile)) {
            $sql = file_get_contents($schemaFile);
            // Remove CREATE DATABASE and USE statements
            $sql = preg_replace('/CREATE DATABASE[^;]+;/i', '', $sql);
            $sql = preg_replace('/USE[^;]+;/i', '', $sql);
            try {
                $db->exec($sql);
                $schemaOk = true;
                $messages[] = 'Database schema created successfully.';
            } catch (Exception $e) {
                $errors[] = 'Failed to run schema_main.sql: ' . $e->getMessage();
            }
        } else {
            $errors[] = 'Schema file not found at database/schema_main.sql';
        }
    } else {
        $schemaOk = true;
        $messages[] = 'Database schema already present.';
    }

    // Ensure default admin exists
    if ($schemaOk) {
        $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE role = 'admin'");
        $stmt->execute();
        $hasAdmin = $stmt->fetchColumn() > 0;
        if (!$hasAdmin) {
            $email = 'admin@marketingplatform.com';
            $hash = password_hash('admin123', PASSWORD_BCRYPT);
            $insert = $db->prepare("INSERT INTO users (email, password_hash, business_name, contact_name, role, status) VALUES (?, ?, 'Marketing Platform Admin', 'System Administrator', 'admin', 'active')");
            try {
                $insert->execute([$email, $hash]);
                $messages[] = 'Default admin user created (admin@marketingplatform.com / admin123). Please change the password after login.';
                $adminOk = true;
            } catch (Exception $e) {
                $errors[] = 'Failed to create default admin: ' . $e->getMessage();
            }
        } else {
            $adminOk = true;
            $messages[] = 'Admin user already exists.';
        }
    }
}

// Mark setup complete when everything OK
$canMarkComplete = $dbOk && $schemaOk && $adminOk && empty($errors);
if ($canMarkComplete) {
    @file_put_contents(__DIR__ . '/.setup_complete', date('c'));
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Setup Wizard - Marketing Platform</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
    <div class="container py-5">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card shadow">
                    <div class="card-body p-4">
                        <h3 class="mb-3">Setup Wizard</h3>
                        <p class="text-muted">This will verify your database and create the default admin if needed.</p>

                        <?php if (!empty($messages)): ?>
                            <div class="alert alert-success">
                                <ul class="mb-0">
                                    <?php foreach ($messages as $m): ?>
                                        <li><?= escape($m) ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        <?php endif; ?>

                        <?php if (!empty($errors)): ?>
                            <div class="alert alert-danger">
                                <ul class="mb-0">
                                    <?php foreach ($errors as $e): ?>
                                        <li><?= escape($e) ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        <?php endif; ?>

                        <?php if ($canMarkComplete): ?>
                            <div class="alert alert-info">Setup is complete. You can now log in.</div>
                            <div class="d-flex gap-2">
                                <a class="btn btn-primary" href="<?= url('admin/login.php') ?>">Go to Admin Login</a>
                                <a class="btn btn-outline-secondary" href="<?= url('login.php') ?>">Go to Client Login</a>
                            </div>
                        <?php else: ?>
                            <p class="mb-0">Fix the errors above and refresh this page.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>


