<?php
/**
 * User Registration Page
 */

require_once 'includes/auth.php';
require_once 'includes/functions.php';
require_once 'includes/csrf.php';

// Ensure base path is loaded
getBasePath();

$auth = new Auth();
$error = '';
$success = '';

// Redirect if already logged in
if ($auth->isLoggedIn()) {
    $role = $_SESSION['user_role'];
    header('Location: ' . url($role === 'admin' ? 'admin/dashboard.php' : 'client/dashboard.php'));
    exit;
}

// Handle registration form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    requireCSRFToken();
    
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';
    $businessName = trim($_POST['business_name'] ?? '');
    $contactName = trim($_POST['contact_name'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $address = trim($_POST['address'] ?? '');
    $city = trim($_POST['city'] ?? '');
    $state = trim($_POST['state'] ?? '');
    $zipCode = trim($_POST['zip_code'] ?? '');
    
    // Validation
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Valid email address is required';
    } elseif (empty($password) || strlen($password) < 8) {
        $error = 'Password must be at least 8 characters long';
    } elseif ($password !== $confirmPassword) {
        $error = 'Passwords do not match';
    } elseif (empty($businessName)) {
        $error = 'Business name is required';
    } elseif (empty($contactName)) {
        $error = 'Contact name is required';
    } else {
        try {
            $db = getDB();
            
            // Check if email already exists
            $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE email = ?");
            $stmt->execute([$email]);
            if ($stmt->fetchColumn() > 0) {
                $error = 'Email address already exists. Please use a different email or try logging in.';
            } else {
                // Create user
                $passwordHash = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $db->prepare("
                    INSERT INTO users (email, password_hash, business_name, contact_name, phone, address, city, state, zip_code, role, status)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'client', 'active')
                ");
                $stmt->execute([
                    $email,
                    $passwordHash,
                    $businessName,
                    $contactName,
                    $phone ?: null,
                    $address ?: null,
                    $city ?: null,
                    $state ?: null,
                    $zipCode ?: null
                ]);
                
                $newUserId = $db->lastInsertId();
                
                // Check for affiliate referral and convert lead
                if (isset($_SESSION['affiliate_code']) && isset($_SESSION['affiliate_partner_id'])) {
                    require_once __DIR__ . '/includes/ReferralProcessor.php';
                    $processor = new ReferralProcessor();
                    
                    // Find lead for this email
                    $leadStmt = $db->prepare("SELECT id FROM client_leads WHERE lead_email = ? AND status != 'converted' ORDER BY created_at DESC LIMIT 1");
                    $leadStmt->execute([$email]);
                    $lead = $leadStmt->fetch();
                    
                    if ($lead) {
                        // Convert lead to client
                        $processor->convertLeadToClient($lead['id'], $newUserId, null);
                    }
                }
                
                // Auto-login the user after registration
                if ($auth->login($email, $password)) {
                    setFlashMessage('success', 'Account created successfully! Welcome to Marketing Platform.');
                    header('Location: ' . url('client/dashboard.php'));
                    exit;
                } else {
                    $success = 'Account created successfully! Please log in.';
                }
            }
        } catch (PDOException $e) {
            $error = 'Registration failed: ' . $e->getMessage();
        }
    }
}

$flash = getFlashMessage();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - Marketing Platform</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body class="bg-light">
    <div class="container">
        <div class="row justify-content-center align-items-center min-vh-100 py-5">
            <div class="col-md-8 col-lg-6">
                <div class="card shadow">
                    <div class="card-body p-5">
                        <h2 class="text-center mb-4">Marketing Platform</h2>
                        <h4 class="text-center mb-4 text-muted">Create Account</h4>
                        
                        <?php if ($error): ?>
                            <div class="alert alert-danger"><?= escape($error) ?></div>
                        <?php endif; ?>
                        
                        <?php if ($success): ?>
                            <div class="alert alert-success"><?= escape($success) ?></div>
                        <?php endif; ?>
                        
                        <?php if ($flash): ?>
                            <div class="alert alert-<?= escape($flash['type']) ?>"><?= escape($flash['message']) ?></div>
                        <?php endif; ?>
                        
                        <form method="POST" action="<?= url('register.php') ?>">
                            <?= csrfField() ?>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="email" class="form-label">Email Address <span class="text-danger">*</span></label>
                                    <input type="email" class="form-control" id="email" name="email" value="<?= escape($_POST['email'] ?? '') ?>" required autofocus>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="contact_name" class="form-label">Contact Name <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="contact_name" name="contact_name" value="<?= escape($_POST['contact_name'] ?? '') ?>" required>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="business_name" class="form-label">Business Name <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="business_name" name="business_name" value="<?= escape($_POST['business_name'] ?? '') ?>" required>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="password" class="form-label">Password <span class="text-danger">*</span></label>
                                    <input type="password" class="form-control" id="password" name="password" required minlength="8">
                                    <small class="form-text text-muted">Must be at least 8 characters</small>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="confirm_password" class="form-label">Confirm Password <span class="text-danger">*</span></label>
                                    <input type="password" class="form-control" id="confirm_password" name="confirm_password" required minlength="8">
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="phone" class="form-label">Phone Number</label>
                                <input type="tel" class="form-control" id="phone" name="phone" value="<?= escape($_POST['phone'] ?? '') ?>">
                            </div>
                            
                            <div class="mb-3">
                                <label for="address" class="form-label">Address</label>
                                <input type="text" class="form-control" id="address" name="address" value="<?= escape($_POST['address'] ?? '') ?>">
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="city" class="form-label">City</label>
                                    <input type="text" class="form-control" id="city" name="city" value="<?= escape($_POST['city'] ?? '') ?>">
                                </div>
                                
                                <div class="col-md-3 mb-3">
                                    <label for="state" class="form-label">State</label>
                                    <input type="text" class="form-control" id="state" name="state" value="<?= escape($_POST['state'] ?? '') ?>">
                                </div>
                                
                                <div class="col-md-3 mb-3">
                                    <label for="zip_code" class="form-label">Zip Code</label>
                                    <input type="text" class="form-control" id="zip_code" name="zip_code" value="<?= escape($_POST['zip_code'] ?? '') ?>">
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary w-100 mb-3">Create Account</button>
                            
                            <div class="text-center">
                                <small class="text-muted">
                                    Already have an account? <a href="<?= url('login.php') ?>">Login here</a>
                                </small>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

