<?php
/**
 * Webhook Helper Functions
 * Utility functions to trigger webhooks from anywhere in the application
 */

require_once __DIR__ . '/database.php';

/**
 * Trigger webhooks for a specific event
 */
function triggerWebhookEvent($eventType, $payload) {
    if (!isset($_SESSION['user_id'])) {
        return; // No user context
    }
    
    $db = getDB();
    $userId = $_SESSION['user_id'];
    
    // Get active webhooks for user that listen to this event
    $stmt = $db->prepare("
        SELECT * FROM webhooks 
        WHERE user_id = ? 
        AND status = 'active'
        AND JSON_CONTAINS(events, ?, '$')
    ");
    $stmt->execute([$userId, json_encode($eventType)]);
    $webhooks = $stmt->fetchAll();
    
    foreach ($webhooks as $webhook) {
        $events = json_decode($webhook['events'], true);
        if (in_array($eventType, $events)) {
            sendWebhookRequest($db, $webhook, $eventType, $payload);
        }
    }
}

function sendWebhookRequest($db, $webhook, $eventType, $payload) {
    $ch = curl_init($webhook['webhook_url']);
    
    $postData = [
        'event' => $eventType,
        'timestamp' => date('c'),
        'payload' => $payload
    ];
    
    // Add signature if secret token exists
    if (!empty($webhook['secret_token'])) {
        $signature = hash_hmac('sha256', json_encode($postData), $webhook['secret_token']);
        $postData['signature'] = $signature;
    }
    
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($postData),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'User-Agent: MarketingPlatform/1.0'
        ],
        CURLOPT_TIMEOUT => 10,
        CURLOPT_CONNECTTIMEOUT => 5
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    // Log webhook call
    $stmt = $db->prepare("
        INSERT INTO webhook_logs 
        (webhook_id, event_type, payload, response_status, response_body, error_message)
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    
    $errorMessage = $error ?: null;
    $responseBody = $response ? substr($response, 0, 1000) : null;
    
    $stmt->execute([
        $webhook['id'],
        $eventType,
        json_encode($payload),
        $httpCode,
        $responseBody,
        $errorMessage
    ]);
    
    // Update webhook status if it failed
    if ($httpCode < 200 || $httpCode >= 300) {
        $failureCount = $webhook['failure_count'] + 1;
        $status = ($failureCount >= 5) ? 'failed' : 'active';
        
        $stmt = $db->prepare("
            UPDATE webhooks 
            SET failure_count = ?, status = ?, last_triggered_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$failureCount, $status, $webhook['id']]);
    } else {
        $stmt = $db->prepare("UPDATE webhooks SET last_triggered_at = NOW(), failure_count = 0 WHERE id = ?");
        $stmt->execute([$webhook['id']]);
    }
}

