<?php
/**
 * Common Utility Functions
 */

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/base_path.php';
require_once __DIR__ . '/webhook_helper.php';

/**
 * Sanitize output to prevent XSS
 */
function escape($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Format currency
 */
function formatCurrency($amount) {
    return '$' . number_format($amount, 2);
}

/**
 * Format date
 */
function formatDate($date, $format = 'Y-m-d') {
    if (empty($date)) return '';
    $dateObj = new DateTime($date);
    return $dateObj->format($format);
}

/**
 * Format datetime
 */
function formatDateTime($datetime, $format = 'Y-m-d H:i:s') {
    if (empty($datetime)) return '';
    $dateObj = new DateTime($datetime);
    return $dateObj->format($format);
}

/**
 * Generate random discount code
 */
function generateDiscountCode($length = 8, $prefix = '') {
    $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    $code = $prefix;
    for ($i = 0; $i < $length; $i++) {
        $code .= $characters[rand(0, strlen($characters) - 1)];
    }
    return $code;
}

/**
 * Check if discount code is unique
 */
function isDiscountCodeUnique($code, $excludeId = null) {
    $db = getDB();
    if ($excludeId) {
        $stmt = $db->prepare("SELECT COUNT(*) FROM discount_codes WHERE code = ? AND id != ?");
        $stmt->execute([$code, $excludeId]);
    } else {
        $stmt = $db->prepare("SELECT COUNT(*) FROM discount_codes WHERE code = ?");
        $stmt->execute([$code]);
    }
    return $stmt->fetchColumn() == 0;
}

/**
 * Get subscription tier limits
 */
function getTierLimits($tierId) {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM subscription_tiers WHERE id = ?");
    $stmt->execute([$tierId]);
    return $stmt->fetch();
}

/**
 * Check if user can create discount code based on tier
 */
function canCreateDiscountCode($userId) {
    // Check demo mode override
    $db = getDB();
    try {
        $stmt = $db->prepare("SELECT demo_mode_enabled FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        if ($user && isset($user['demo_mode_enabled']) && $user['demo_mode_enabled'] == 1) {
            return true; // Bypass limits in demo mode
        }
    } catch (PDOException $e) {
        // Column might not exist, continue with normal check
    }
    
    $stmt = $db->prepare("
        SELECT st.discount_codes_limit, 
               (SELECT COUNT(*) FROM discount_codes WHERE user_id = ? AND MONTH(created_at) = MONTH(NOW()) AND YEAR(created_at) = YEAR(NOW())) as current_month_count
        FROM subscriptions s
        JOIN subscription_tiers st ON s.tier_id = st.id
        WHERE s.user_id = ? AND s.status = 'active'
        LIMIT 1
    ");
    $stmt->execute([$userId, $userId]);
    $result = $stmt->fetch();
    
    if (!$result) return false;
    
    // NULL limit means unlimited
    if ($result['discount_codes_limit'] === null) return true;
    
    return $result['current_month_count'] < $result['discount_codes_limit'];
}

/**
 * Check if user can send email campaign based on tier
 */
function canSendEmailCampaign($userId, $contactCount) {
    // Check demo mode override
    $db = getDB();
    try {
        $stmt = $db->prepare("SELECT demo_mode_enabled FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        if ($user && isset($user['demo_mode_enabled']) && $user['demo_mode_enabled'] == 1) {
            return true; // Bypass limits in demo mode
        }
    } catch (PDOException $e) {
        // Column might not exist, continue with normal check
    }
    
    $stmt = $db->prepare("
        SELECT st.email_contacts_limit, st.email_campaigns_limit,
               (SELECT COUNT(*) FROM email_subscribers WHERE user_id = ? AND status = 'subscribed') as subscriber_count,
               (SELECT COUNT(*) FROM email_campaigns WHERE user_id = ? AND MONTH(created_at) = MONTH(NOW()) AND YEAR(created_at) = YEAR(NOW())) as current_month_campaigns
        FROM subscriptions s
        JOIN subscription_tiers st ON s.tier_id = st.id
        WHERE s.user_id = ? AND s.status = 'active'
        LIMIT 1
    ");
    $stmt->execute([$userId, $userId, $userId]);
    $result = $stmt->fetch();
    
    if (!$result) return false;
    
    // Check contact limit
    if ($result['email_contacts_limit'] !== null && $contactCount > $result['email_contacts_limit']) {
        return false;
    }
    
    // Check campaign limit
    if ($result['email_campaigns_limit'] !== null && $result['current_month_campaigns'] >= $result['email_campaigns_limit']) {
        return false;
    }
    
    return true;
}

/**
 * Get user metrics summary
 */
function getUserMetricsSummary($userId, $days = 30) {
    $db = getDB();
    $stmt = $db->prepare("
        SELECT 
            COUNT(*) as total_visits,
            COUNT(DISTINCT session_id) as unique_visitors,
            COUNT(CASE WHEN conversion_type IS NOT NULL THEN 1 END) as conversions,
            SUM(conversion_value) as total_revenue
        FROM user_metrics
        WHERE user_id = ? AND visited_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
    ");
    $stmt->execute([$userId, $days]);
    return $stmt->fetch();
}

/**
 * Get campaign performance summary
 */
function getCampaignPerformanceSummary($userId, $days = 30) {
    $db = getDB();
    $stmt = $db->prepare("
        SELECT 
            ad_platform,
            SUM(impressions) as total_impressions,
            SUM(clicks) as total_clicks,
            SUM(conversions) as total_conversions,
            SUM(cost) as total_cost,
            SUM(revenue) as total_revenue,
            CASE 
                WHEN SUM(clicks) > 0 THEN (SUM(cost) / SUM(clicks))
                ELSE 0 
            END as avg_cpc,
            CASE 
                WHEN SUM(impressions) > 0 THEN (SUM(clicks) / SUM(impressions) * 100)
                ELSE 0 
            END as ctr
        FROM ad_performance
        WHERE user_id = ? AND date >= DATE_SUB(NOW(), INTERVAL ? DAY)
        GROUP BY ad_platform
    ");
    $stmt->execute([$userId, $days]);
    return $stmt->fetchAll();
}

/**
 * Log user activity/metric
 */
function logUserMetric($userId, $data = []) {
    $db = getDB();
    
    $fields = ['user_id'];
    $values = [$userId];
    $placeholders = ['?'];
    
    $allowedFields = [
        'session_id', 'page_url', 'referrer', 'utm_source', 'utm_medium', 
        'utm_campaign', 'utm_term', 'utm_content', 'ip_address', 'user_agent', 
        'device_type', 'conversion_type', 'conversion_value', 'discount_code_used'
    ];
    
    foreach ($allowedFields as $field) {
        if (isset($data[$field])) {
            $fields[] = $field;
            $values[] = $data[$field];
            $placeholders[] = '?';
        }
    }
    
    // Auto-detect device type if not provided
    if (!isset($data['device_type']) && isset($data['user_agent'])) {
        $userAgent = strtolower($data['user_agent']);
        if (preg_match('/mobile|android|iphone|ipad/', $userAgent)) {
            $deviceType = preg_match('/tablet|ipad/', $userAgent) ? 'tablet' : 'mobile';
        } else {
            $deviceType = 'desktop';
        }
        $fields[] = 'device_type';
        $values[] = $deviceType;
        $placeholders[] = '?';
    }
    
    $sql = "INSERT INTO user_metrics (" . implode(', ', $fields) . ") VALUES (" . implode(', ', $placeholders) . ")";
    $stmt = $db->prepare($sql);
    return $stmt->execute($values);
}

/**
 * Get device type from user agent
 */
function getDeviceType($userAgent) {
    if (empty($userAgent)) return 'unknown';
    
    $userAgent = strtolower($userAgent);
    if (preg_match('/tablet|ipad/', $userAgent)) return 'tablet';
    if (preg_match('/mobile|android|iphone/', $userAgent)) return 'mobile';
    return 'desktop';
}

/**
 * Validate email address
 */
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Generate pagination HTML
 */
function generatePagination($currentPage, $totalPages, $baseUrl) {
    if ($totalPages <= 1) return '';
    
    $html = '<nav aria-label="Page navigation"><ul class="pagination justify-content-center">';
    
    // Previous button
    if ($currentPage > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '?page=' . ($currentPage - 1) . '">Previous</a></li>';
    }
    
    // Page numbers
    for ($i = 1; $i <= $totalPages; $i++) {
        $active = ($i == $currentPage) ? 'active' : '';
        $html .= '<li class="page-item ' . $active . '"><a class="page-link" href="' . $baseUrl . '?page=' . $i . '">' . $i . '</a></li>';
    }
    
    // Next button
    if ($currentPage < $totalPages) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '?page=' . ($currentPage + 1) . '">Next</a></li>';
    }
    
    $html .= '</ul></nav>';
    return $html;
}

/**
 * Set flash message
 */
function setFlashMessage($type, $message) {
    $_SESSION['flash_type'] = $type;
    $_SESSION['flash_message'] = $message;
}

/**
 * Get and clear flash message
 */
function getFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $type = $_SESSION['flash_type'] ?? 'info';
        $message = $_SESSION['flash_message'];
        unset($_SESSION['flash_message'], $_SESSION['flash_type']);
        return ['type' => $type, 'message' => $message];
    }
    return null;
}

/**
 * Validate and apply discount code
 * Returns array with 'valid' => true/false and discount details or error message
 */
function validateAndUseDiscountCode($code, $orderValue, $userId = null) {
    $db = getDB();
    $code = strtoupper(trim($code));
    
    // Get discount code
    $stmt = $db->prepare("SELECT * FROM discount_codes WHERE code = ? AND status = 'active'");
    $stmt->execute([$code]);
    $discount = $stmt->fetch();
    
    if (!$discount) {
        return ['valid' => false, 'error' => 'Discount code not found or inactive'];
    }
    
    // Check if code is expired
    $now = date('Y-m-d');
    if ($discount['end_date'] && $discount['end_date'] < $now) {
        return ['valid' => false, 'error' => 'Discount code has expired'];
    }
    
    if ($discount['start_date'] && $discount['start_date'] > $now) {
        return ['valid' => false, 'error' => 'Discount code is not yet valid'];
    }
    
    // Check usage limit
    if ($discount['usage_limit'] !== null && $discount['usage_count'] >= $discount['usage_limit']) {
        return ['valid' => false, 'error' => 'Discount code has reached its usage limit'];
    }
    
    // Check minimum purchase amount
    if ($discount['min_purchase_amount'] !== null && $orderValue < $discount['min_purchase_amount']) {
        return ['valid' => false, 'error' => 'Minimum purchase amount not met'];
    }
    
    // Calculate discount amount
    if ($discount['discount_type'] === 'percentage') {
        $discountAmount = ($orderValue * $discount['discount_value']) / 100;
        if ($discount['max_discount_amount'] !== null && $discountAmount > $discount['max_discount_amount']) {
            $discountAmount = $discount['max_discount_amount'];
        }
    } else {
        $discountAmount = $discount['discount_value'];
    }
    
    return [
        'valid' => true,
        'discount_id' => $discount['id'],
        'discount_type' => $discount['discount_type'],
        'discount_value' => $discount['discount_value'],
        'discount_amount' => $discountAmount,
        'owner_user_id' => $discount['user_id']
    ];
}

/**
 * Record discount code usage
 */
function recordDiscountCodeUsage($discountCodeId, $userId, $orderId = null, $orderValue = null, $discountAmount, $conversionValue = null) {
    $db = getDB();
    
    // Get discount code owner
    $stmt = $db->prepare("SELECT user_id FROM discount_codes WHERE id = ?");
    $stmt->execute([$discountCodeId]);
    $discount = $stmt->fetch();
    
    if (!$discount) {
        return false;
    }
    
    // Update usage count
    $stmt = $db->prepare("UPDATE discount_codes SET usage_count = usage_count + 1 WHERE id = ?");
    $stmt->execute([$discountCodeId]);
    
    // Record detailed usage
    $stmt = $db->prepare("
        INSERT INTO discount_code_usage 
        (discount_code_id, user_id, order_id, order_value, discount_amount, conversion_value, ip_address, user_agent, used_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    
    $ipAddress = $_SERVER['REMOTE_ADDR'] ?? null;
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? null;
    
    $stmt->execute([
        $discountCodeId,
        $userId,
        $orderId,
        $orderValue,
        $discountAmount,
        $conversionValue,
        $ipAddress,
        $userAgent
    ]);
    
    // Trigger webhook for discount usage (use owner's user_id for webhook context)
    $originalUserId = $_SESSION['user_id'] ?? null;
    $_SESSION['user_id'] = $discount['user_id'];
    triggerWebhookEvent('discount.used', [
        'discount_code_id' => $discountCodeId,
        'owner_user_id' => $discount['user_id'],
        'used_by_user_id' => $userId,
        'order_id' => $orderId,
        'order_value' => $orderValue,
        'discount_amount' => $discountAmount,
        'conversion_value' => $conversionValue,
        'timestamp' => date('c')
    ]);
    if ($originalUserId !== null) {
        $_SESSION['user_id'] = $originalUserId;
    } else {
        unset($_SESSION['user_id']);
    }
    
    return true;
}

