<?php
/**
 * Database Connection Handler
 * Provides PDO connection with error handling
 */

require_once __DIR__ . '/../config/database.php';

// Load production config if exists (for shared hosting optimization)
if (file_exists(__DIR__ . '/../config/production.php')) {
    require_once __DIR__ . '/../config/production.php';
}

class Database {
    private static $instance = null;
    private $connection;

    private function __construct() {
        try {
            // Handle port in host string if provided (format: host:port)
            $host = DB_HOST;
            $port = '';
            if (strpos($host, ':') !== false) {
                list($host, $port) = explode(':', $host, 2);
                $dsn = "mysql:host=$host;port=$port;dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            } else {
                $dsn = "mysql:host=" . $host . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            }
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ];
            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            // More helpful error message for setup
            if (basename($_SERVER['PHP_SELF']) === 'setup.php' || !isset($_SERVER['PHP_SELF'])) {
                throw $e; // Let setup.php handle the error
            } else {
                die("Database connection failed. Please ensure MySQL is running and check config/database.php. Error: " . $e->getMessage());
            }
        }
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function getConnection() {
        return $this->connection;
    }

    // Prevent cloning
    private function __clone() {}

    // Prevent unserialization
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }
}

// Helper function for easy access
function getDB() {
    return Database::getInstance()->getConnection();
}

