<?php
/**
 * Credential Helper Functions
 * Functions to securely retrieve and decrypt API credentials
 */

require_once __DIR__ . '/database.php';

/**
 * Get encrypted credentials for a user and service
 */
function getEncryptedCredentials($userId, $serviceName) {
    $db = getDB();
    $stmt = $db->prepare("
        SELECT credential_value, is_encrypted
        FROM api_credentials
        WHERE user_id = ? AND service_name = ? AND status = 'active'
    ");
    $stmt->execute([$userId, $serviceName]);
    return $stmt->fetch();
}

/**
 * Decrypt credentials
 */
function decryptCredentials($encryptedValue) {
    $encryptionKey = getenv('ENCRYPTION_KEY') ?: 'default_key_change_in_production';
    
    try {
        $decrypted = openssl_decrypt(
            base64_decode($encryptedValue),
            'AES-256-CBC',
            hash('sha256', $encryptionKey),
            0,
            substr(hash('sha256', $encryptionKey), 0, 16)
        );
        
        if ($decrypted === false) {
            return null;
        }
        
        return json_decode($decrypted, true);
    } catch (Exception $e) {
        error_log("Credential decryption error: " . $e->getMessage());
        return null;
    }
}

/**
 * Get user's API credentials for a service (decrypted)
 */
function getUserCredentials($userId, $serviceName) {
    $encrypted = getEncryptedCredentials($userId, $serviceName);
    
    if (!$encrypted) {
        return null;
    }
    
    if ($encrypted['is_encrypted']) {
        return decryptCredentials($encrypted['credential_value']);
    }
    
    // If not encrypted (legacy), try to decode JSON
    return json_decode($encrypted['credential_value'], true);
}

/**
 * Check if user has configured a service
 */
function hasServiceConfigured($userId, $serviceName) {
    $db = getDB();
    $stmt = $db->prepare("
        SELECT COUNT(*) as count
        FROM api_credentials
        WHERE user_id = ? AND service_name = ? AND status = 'active'
    ");
    $stmt->execute([$userId, $serviceName]);
    $result = $stmt->fetch();
    return $result['count'] > 0;
}

/**
 * Delete user's credentials for a service
 */
function deleteUserCredentials($userId, $serviceName) {
    $db = getDB();
    $stmt = $db->prepare("
        UPDATE api_credentials 
        SET status = 'inactive'
        WHERE user_id = ? AND service_name = ?
    ");
    return $stmt->execute([$userId, $serviceName]);
}

