<?php
/**
 * Base Path Helper
 * Handles subdirectory installations and domain configuration
 */

// Load domain configuration
if (!defined('MAIN_PATH')) {
    require_once __DIR__ . '/../config/domain_config.php';
}

/**
 * Get the base URL path for the application
 * Detects if running in a subdirectory
 */
if (!function_exists('getBasePath')) {
    function getBasePath() {
        if (defined('BASE_PATH')) {
            return BASE_PATH;
        }
        
        // Get the actual application root directory (where includes/ is located - parent directory)
        $appRootFs = dirname(__DIR__);
        
        // Get the document root
        $docRoot = $_SERVER['DOCUMENT_ROOT'] ?? '';
        $docRoot = str_replace('\\', '/', rtrim($docRoot, '/'));
        $appRootFs = str_replace('\\', '/', rtrim($appRootFs, '/'));
        
        // Calculate relative path from document root to app root
        $basePath = '/';
        
        if ($docRoot && strpos($appRootFs, $docRoot) === 0) {
            // App root is inside document root
            $relativePath = substr($appRootFs, strlen($docRoot));
            $relativePath = trim($relativePath, '/');
            $basePath = $relativePath ? '/' . $relativePath . '/' : '/';
        } else {
            // Fallback: extract from SCRIPT_NAME, removing subdirectories (admin, client, api)
            $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
            $parts = array_filter(explode('/', trim($scriptName, '/')));
            
            // Remove known subdirectories from the beginning
            while (!empty($parts) && in_array($parts[0], ['admin', 'client', 'api'])) {
                array_shift($parts);
            }
            
            if (!empty($parts)) {
                // First remaining part is the app root directory
                $basePath = '/' . $parts[0] . '/';
            }
        }
        
        // Ensure base path starts with /
        if (strpos($basePath, '/') !== 0) {
            $basePath = '/' . $basePath;
        }
        
        // Ensure trailing slash (unless it's root)
        if ($basePath !== '/' && substr($basePath, -1) !== '/') {
            $basePath .= '/';
        }
        
        define('BASE_PATH', $basePath);
        return BASE_PATH;
    }
}

/**
 * Get full URL path (with base path)
 */
if (!function_exists('url')) {
    function url($path = '') {
        // If path is already a full URL, return as-is
        if (strpos($path, 'http://') === 0 || strpos($path, 'https://') === 0) {
            return $path;
        }
        
        // Use domain config if available
        if (defined('MAIN_BASE_URL') && strpos($path, 'acceleratedadz') === false && strpos($path, 'acceleratedleadz') === false) {
            $basePath = getBasePath();
            $path = ltrim($path, '/');
            $basePath = rtrim($basePath, '/');
            
            if (empty($path)) {
                return $basePath . '/';
            }
            
            return $basePath . '/' . $path;
        }
        
        // Fallback to original behavior
        $basePath = getBasePath();
        $path = ltrim($path, '/');
        $basePath = rtrim($basePath, '/');
        
        if (empty($path)) {
            return $basePath . '/';
        }
        
        return $basePath . '/' . $path;
    }
}
