<?php
/**
 * Authentication System
 * Handles login, logout, session management, and role-based access control
 */

session_start();

require_once __DIR__ . '/database.php';

class Auth {
    private $db;

    public function __construct() {
        $this->db = getDB();
    }

    /**
     * Login user with email and password
     */
    public function login($email, $password) {
        $stmt = $this->db->prepare("SELECT id, email, password_hash, business_name, role, status FROM users WHERE email = ? AND status = 'active'");
        $stmt->execute([$email]);
        $user = $stmt->fetch();

        if ($user && password_verify($password, $user['password_hash'])) {
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['user_role'] = $user['role'];
            $_SESSION['business_name'] = $user['business_name'];
            
            // Update last login
            $updateStmt = $this->db->prepare("UPDATE users SET updated_at = NOW() WHERE id = ?");
            $updateStmt->execute([$user['id']]);
            
            return true;
        }
        
        return false;
    }

    /**
     * Logout current user
     */
    public function logout() {
        $_SESSION = array();
        
        if (isset($_COOKIE[session_name()])) {
            setcookie(session_name(), '', time() - 3600, '/');
        }
        
        session_destroy();
    }

    /**
     * Check if user is logged in
     */
    public function isLoggedIn() {
        return isset($_SESSION['user_id']);
    }

    /**
     * Check if user has specific role
     */
    public function hasRole($role) {
        return isset($_SESSION['user_role']) && $_SESSION['user_role'] === $role;
    }

    /**
     * Require authentication
     */
    public function requireAuth() {
        if (!$this->isLoggedIn()) {
            require_once __DIR__ . '/functions.php';
            header('Location: ' . url('login.php'));
            exit;
        }
    }

    /**
     * Require specific role
     */
    public function requireRole($role) {
        $this->requireAuth();
        
        if (!$this->hasRole($role)) {
            require_once __DIR__ . '/functions.php';
            header('Location: ' . url('unauthorized.php'));
            exit;
        }
    }

    /**
     * Get current user data
     */
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }

        $stmt = $this->db->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        return $stmt->fetch();
    }

    /**
     * Get current user's subscription
     */
    public function getCurrentSubscription() {
        if (!$this->isLoggedIn()) {
            return null;
        }

        $stmt = $this->db->prepare("
            SELECT s.*, st.tier_name, st.tier_level, st.monthly_price, st.google_ads_budget, 
                   st.facebook_ads_budget, st.email_contacts_limit, st.email_campaigns_limit, 
                   st.discount_codes_limit, st.direct_mail_quantity, st.direct_mail_frequency, st.features
            FROM subscriptions s
            JOIN subscription_tiers st ON s.tier_id = st.id
            WHERE s.user_id = ? AND s.status = 'active'
            ORDER BY s.created_at DESC
            LIMIT 1
        ");
        $stmt->execute([$_SESSION['user_id']]);
        return $stmt->fetch();
    }

    /**
     * Generate CSRF token
     */
    public function generateCSRFToken() {
        if (!isset($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }

    /**
     * Verify CSRF token
     */
    public function verifyCSRFToken($token) {
        return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }
}

// Helper functions
function isLoggedIn() {
    $auth = new Auth();
    return $auth->isLoggedIn();
}

function requireAuth() {
    $auth = new Auth();
    $auth->requireAuth();
}

function requireAdmin() {
    $auth = new Auth();
    $auth->requireRole('admin');
}

function requireClient() {
    requireAuth();
    
    // Allow admins who are viewing as a client
    if ($_SESSION['user_role'] === 'admin' && isset($_SESSION['viewing_as_client_id'])) {
        // Verify admin has access to this client
        if (adminHasClientAccess($_SESSION['user_id'], $_SESSION['viewing_as_client_id'])) {
            return; // Allow access
        }
    }
    
    // Otherwise require client role
    if ($_SESSION['user_role'] !== 'client') {
        require_once __DIR__ . '/functions.php';
        header('Location: ' . url('unauthorized.php'));
        exit;
    }
}

function getCurrentUser() {
    $auth = new Auth();
    return $auth->getCurrentUser();
}

function getCurrentSubscription() {
    $auth = new Auth();
    return $auth->getCurrentSubscription();
}

/**
 * Check if admin has access to a specific client
 */
function adminHasClientAccess($adminId, $clientId) {
    $db = getDB();
    $stmt = $db->prepare("SELECT COUNT(*) FROM admin_client_assignments WHERE admin_id = ? AND client_id = ?");
    $stmt->execute([$adminId, $clientId]);
    return $stmt->fetchColumn() > 0;
}

/**
 * Get all client IDs assigned to an admin
 */
function getAdminClients($adminId) {
    $db = getDB();
    $stmt = $db->prepare("
        SELECT client_id 
        FROM admin_client_assignments 
        WHERE admin_id = ?
    ");
    $stmt->execute([$adminId]);
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

/**
 * Assign a client to an admin
 */
function assignClientToAdmin($adminId, $clientId, $assignedBy = null) {
    $db = getDB();
    try {
        $stmt = $db->prepare("
            INSERT INTO admin_client_assignments (admin_id, client_id, assigned_by)
            VALUES (?, ?, ?)
            ON DUPLICATE KEY UPDATE assigned_at = CURRENT_TIMESTAMP
        ");
        $stmt->execute([$adminId, $clientId, $assignedBy ?: $adminId]);
        return true;
    } catch (PDOException $e) {
        error_log("Error assigning client to admin: " . $e->getMessage());
        return false;
    }
}

/**
 * Unassign a client from an admin
 */
function unassignClientFromAdmin($adminId, $clientId) {
    $db = getDB();
    try {
        $stmt = $db->prepare("DELETE FROM admin_client_assignments WHERE admin_id = ? AND client_id = ?");
        $stmt->execute([$adminId, $clientId]);
        return true;
    } catch (PDOException $e) {
        error_log("Error unassigning client from admin: " . $e->getMessage());
        return false;
    }
}

/**
 * Get effective user ID for API/resource access
 * If admin is accessing, check if viewing_as_client is set
 * Otherwise return their own ID
 */
function getEffectiveUserId() {
    if (!isset($_SESSION['user_id'])) {
        return null;
    }
    
    // If admin is viewing as a client, use that client's ID
    if ($_SESSION['user_role'] === 'admin' && isset($_SESSION['viewing_as_client_id'])) {
        // Verify admin has access to this client
        if (adminHasClientAccess($_SESSION['user_id'], $_SESSION['viewing_as_client_id'])) {
            return $_SESSION['viewing_as_client_id'];
        }
    }
    
    return $_SESSION['user_id'];
}

/**
 * Require access to a client resource (works for clients accessing their own, or admins accessing assigned clients)
 */
function requireClientAccess($clientId) {
    requireAuth();
    $userId = $_SESSION['user_id'];
    $userRole = $_SESSION['user_role'];
    
    // Clients can only access their own resources
    if ($userRole === 'client' && $userId != $clientId) {
        require_once __DIR__ . '/functions.php';
        header('Location: ' . url('unauthorized.php'));
        exit;
    }
    
    // Admins can access their assigned clients
    if ($userRole === 'admin' && !adminHasClientAccess($userId, $clientId)) {
        require_once __DIR__ . '/functions.php';
        setFlashMessage('danger', 'You do not have access to this client.');
        header('Location: ' . url('admin/clients.php'));
        exit;
    }
}

