<?php
/**
 * Activity Logging Helper
 * Logs admin actions for audit trail
 */

require_once __DIR__ . '/database.php';

/**
 * Log admin activity
 */
function logActivity($actionType, $entityType, $entityId = null, $description = null, $oldValues = null, $newValues = null, $affectedCount = 1) {
    if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_role']) || $_SESSION['user_role'] !== 'admin') {
        return false;
    }
    
    $db = getDB();
    $adminId = $_SESSION['user_id'];
    $ipAddress = $_SERVER['REMOTE_ADDR'] ?? null;
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? null;
    
    // Convert arrays to JSON if needed
    if (is_array($oldValues)) {
        $oldValues = json_encode($oldValues);
    }
    if (is_array($newValues)) {
        $newValues = json_encode($newValues);
    }
    
    try {
        $stmt = $db->prepare("
            INSERT INTO admin_activity_log 
            (admin_id, action_type, entity_type, entity_id, description, old_values, new_values, affected_count, ip_address, user_agent)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([
            $adminId,
            $actionType,
            $entityType,
            $entityId,
            $description,
            $oldValues,
            $newValues,
            $affectedCount,
            $ipAddress,
            $userAgent
        ]);
    } catch (PDOException $e) {
        error_log("Failed to log activity: " . $e->getMessage());
        return false;
    }
}

/**
 * Get activity logs
 */
function getActivityLogs($limit = 50, $offset = 0, $filters = []) {
    $db = getDB();
    
    $whereClauses = [];
    $params = [];
    
    if (isset($filters['admin_id'])) {
        $whereClauses[] = "admin_id = ?";
        $params[] = $filters['admin_id'];
    }
    
    if (isset($filters['action_type'])) {
        $whereClauses[] = "action_type = ?";
        $params[] = $filters['action_type'];
    }
    
    if (isset($filters['entity_type'])) {
        $whereClauses[] = "entity_type = ?";
        $params[] = $filters['entity_type'];
    }
    
    $whereClause = !empty($whereClauses) ? 'WHERE ' . implode(' AND ', $whereClauses) : '';
    
    $query = "
        SELECT aal.*, u.email as admin_email, u.business_name as admin_name
        FROM admin_activity_log aal
        JOIN users u ON aal.admin_id = u.id
        $whereClause
        ORDER BY aal.created_at DESC
        LIMIT ? OFFSET ?
    ";
    
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    return $stmt->fetchAll();
}

/**
 * Get activity log count
 */
function getActivityLogCount($filters = []) {
    $db = getDB();
    
    $whereClauses = [];
    $params = [];
    
    if (isset($filters['admin_id'])) {
        $whereClauses[] = "admin_id = ?";
        $params[] = $filters['admin_id'];
    }
    
    if (isset($filters['action_type'])) {
        $whereClauses[] = "action_type = ?";
        $params[] = $filters['action_type'];
    }
    
    if (isset($filters['entity_type'])) {
        $whereClauses[] = "entity_type = ?";
        $params[] = $filters['entity_type'];
    }
    
    $whereClause = !empty($whereClauses) ? 'WHERE ' . implode(' AND ', $whereClauses) : '';
    
    $query = "SELECT COUNT(*) FROM admin_activity_log $whereClause";
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    return $stmt->fetchColumn();
}

