<?php
/**
 * Referral Processor
 * Processes referrals and creates lead records from affiliate clicks
 */

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/AffiliateTracker.php';

class ReferralProcessor {
    private $db;
    private $tracker;
    
    public function __construct() {
        $this->db = getDB();
        $this->tracker = new AffiliateTracker();
    }
    
    /**
     * Process referral and create lead
     */
    public function processReferral($email, $name = null, $phone = null, $companyName = null, $notes = null) {
        // Get current affiliate from session
        $affiliate = $this->tracker->getCurrentAffiliate();
        
        if (!$affiliate) {
            return ['success' => false, 'error' => 'No affiliate referral found'];
        }
        
        // Get latest click
        $click = $this->tracker->getLatestClick();
        $clickId = $click ? $click['id'] : null;
        
        // Create lead record
        $stmt = $this->db->prepare("
            INSERT INTO client_leads 
            (affiliate_click_id, partner_id, partner_code, lead_email, lead_name, lead_phone, 
             company_name, lead_source, status, notes, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, 'affiliate', 'new', ?, NOW())
        ");
        
        $stmt->execute([
            $clickId,
            $affiliate['partner_id'],
            $affiliate['partner_code'],
            $email,
            $name,
            $phone,
            $companyName,
            $notes
        ]);
        
        $leadId = $this->db->lastInsertId();
        
        return [
            'success' => true,
            'lead_id' => $leadId,
            'partner_id' => $affiliate['partner_id'],
            'partner_code' => $affiliate['partner_code']
        ];
    }
    
    /**
     * Convert lead to client (when user signs up)
     */
    public function convertLeadToClient($leadId, $userId, $orderId = null) {
        // Get lead details
        $stmt = $this->db->prepare("SELECT * FROM client_leads WHERE id = ?");
        $stmt->execute([$leadId]);
        $lead = $stmt->fetch();
        
        if (!$lead || $lead['status'] === 'converted') {
            return ['success' => false, 'error' => 'Lead not found or already converted'];
        }
        
        // Update lead status
        $updateStmt = $this->db->prepare("
            UPDATE client_leads 
            SET status = 'converted', converted_at = NOW(), converted_to_user_id = ?
            WHERE id = ?
        ");
        $updateStmt->execute([$userId, $leadId]);
        
        // Get order value if order ID provided
        $orderValue = 0;
        if ($orderId) {
            $orderStmt = $this->db->prepare("SELECT order_amount FROM client_orders WHERE id = ?");
            $orderStmt->execute([$orderId]);
            $order = $orderStmt->fetch();
            $orderValue = $order ? floatval($order['order_amount']) : 0;
        }
        
        // Record conversion if we have a partner
        if ($lead['partner_id']) {
            require_once __DIR__ . '/CommissionCalculator.php';
            $calculator = new CommissionCalculator();
            
            $clickId = $lead['affiliate_click_id'];
            $conversionType = $orderId ? 'subscription' : 'trial_signup';
            $conversionValue = $orderValue ?: 0;
            
            // If no order value, use a default for trial signups
            if ($conversionValue == 0 && $conversionType === 'trial_signup') {
                $conversionValue = 10; // Default trial value
            }
            
            $calculator->recordConversion(
                $lead['partner_id'],
                $clickId,
                $leadId,
                $orderId,
                $userId,
                $conversionType,
                $conversionValue
            );
        }
        
        return ['success' => true, 'lead_id' => $leadId, 'user_id' => $userId];
    }
    
    /**
     * Process order conversion (when client places order/subscribes)
     */
    public function processOrderConversion($orderId, $userId) {
        // Check if user was referred by affiliate
        $stmt = $this->db->prepare("
            SELECT id, partner_id, affiliate_click_id 
            FROM client_leads 
            WHERE converted_to_user_id = ? AND status = 'converted' AND partner_id IS NOT NULL
            ORDER BY converted_at DESC
            LIMIT 1
        ");
        $stmt->execute([$userId]);
        $lead = $stmt->fetch();
        
        if (!$lead || !$lead['partner_id']) {
            return ['success' => false, 'message' => 'No affiliate referral found'];
        }
        
        // Get order details
        $orderStmt = $this->db->prepare("SELECT * FROM client_orders WHERE id = ?");
        $orderStmt->execute([$orderId]);
        $order = $orderStmt->fetch();
        
        if (!$order) {
            return ['success' => false, 'error' => 'Order not found'];
        }
        
        // Check if conversion already recorded for this order
        $checkStmt = $this->db->prepare("
            SELECT id FROM affiliate_conversions WHERE order_id = ?
        ");
        $checkStmt->execute([$orderId]);
        if ($checkStmt->fetch()) {
            return ['success' => false, 'message' => 'Conversion already recorded'];
        }
        
        // Record conversion
        require_once __DIR__ . '/CommissionCalculator.php';
        $calculator = new CommissionCalculator();
        
        $conversionType = $order['order_type'] === 'subscription' ? 'subscription' : 'one_time';
        $conversionValue = floatval($order['order_amount']);
        
        $conversionId = $calculator->recordConversion(
            $lead['partner_id'],
            $lead['affiliate_click_id'],
            $lead['id'],
            $orderId,
            $userId,
            $conversionType,
            $conversionValue
        );
        
        return [
            'success' => true,
            'conversion_id' => $conversionId,
            'partner_id' => $lead['partner_id']
        ];
    }
}

