<?php
/**
 * Partner Dashboard Data Provider
 * Provides statistics and data for affiliate partner dashboard
 */

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/CommissionCalculator.php';
require_once __DIR__ . '/AffiliateTracker.php';

class PartnerDashboard {
    private $db;
    private $calculator;
    private $tracker;
    
    public function __construct() {
        $this->db = getDB();
        $this->calculator = new CommissionCalculator();
        $this->tracker = new AffiliateTracker();
    }
    
    /**
     * Get dashboard stats for a partner
     */
    public function getDashboardStats($partnerId, $dateRange = 30) {
        $startDate = date('Y-m-d', strtotime("-$dateRange days"));
        $endDate = date('Y-m-d');
        
        // Get clicks
        $clickStats = $this->tracker->getClickStats($partnerId, $startDate, $endDate);
        
        // Get conversions
        $conversionStmt = $this->db->prepare("
            SELECT 
                COUNT(*) as total_conversions,
                SUM(conversion_value) as total_revenue,
                SUM(commission_amount) as total_commission
            FROM affiliate_conversions
            WHERE partner_id = ? AND converted_at >= ? AND converted_at <= ?
        ");
        $conversionStmt->execute([$partnerId, $startDate, $endDate]);
        $conversionStats = $conversionStmt->fetch();
        
        // Get pending commissions
        $pendingCommission = $this->calculator->getPendingCommissions($partnerId);
        
        // Get paid commissions
        $paidCommission = $this->calculator->getPaidCommissions($partnerId);
        
        // Get conversion rate
        $conversionRate = 0;
        if ($clickStats['total_clicks'] > 0) {
            $conversionRate = ($conversionStats['total_conversions'] / $clickStats['total_clicks']) * 100;
        }
        
        return [
            'clicks' => [
                'total' => intval($clickStats['total_clicks']),
                'unique_sessions' => intval($clickStats['unique_sessions']),
                'unique_visitors' => intval($clickStats['unique_visitors'])
            ],
            'conversions' => [
                'total' => intval($conversionStats['total_conversions']),
                'rate' => round($conversionRate, 2),
                'revenue' => floatval($conversionStats['total_revenue'] ?? 0),
                'commission' => floatval($conversionStats['total_commission'] ?? 0)
            ],
            'commissions' => [
                'pending' => $pendingCommission,
                'paid' => $paidCommission,
                'total' => $pendingCommission + $paidCommission
            ]
        ];
    }
    
    /**
     * Get recent conversions
     */
    public function getRecentConversions($partnerId, $limit = 10) {
        $stmt = $this->db->prepare("
            SELECT 
                ac.*,
                u.business_name,
                u.email as client_email,
                co.order_amount,
                co.order_type
            FROM affiliate_conversions ac
            LEFT JOIN users u ON ac.user_id = u.id
            LEFT JOIN client_orders co ON ac.order_id = co.id
            WHERE ac.partner_id = ?
            ORDER BY ac.converted_at DESC
            LIMIT ?
        ");
        $stmt->execute([$partnerId, $limit]);
        return $stmt->fetchAll();
    }
    
    /**
     * Get payout history
     */
    public function getPayoutHistory($partnerId, $limit = 10) {
        $stmt = $this->db->prepare("
            SELECT * FROM affiliate_payouts
            WHERE partner_id = ?
            ORDER BY created_at DESC
            LIMIT ?
        ");
        $stmt->execute([$partnerId, $limit]);
        return $stmt->fetchAll();
    }
}

