<?php
/**
 * Partner Authentication & Authorization
 * Handles affiliate partner login and session management
 */

require_once __DIR__ . '/database.php';

class PartnerAuth {
    private $db;
    
    public function __construct() {
        $this->db = getDB();
    }
    
    /**
     * Authenticate partner by partner code and email
     */
    public function login($partnerCode, $email) {
        $stmt = $this->db->prepare("
            SELECT * FROM affiliate_partners 
            WHERE partner_code = ? AND contact_email = ? AND status = 'active'
        ");
        $stmt->execute([$partnerCode, $email]);
        $partner = $stmt->fetch();
        
        if ($partner) {
            $_SESSION['partner_id'] = $partner['id'];
            $_SESSION['partner_code'] = $partner['partner_code'];
            $_SESSION['partner_name'] = $partner['partner_name'];
            $_SESSION['partner_email'] = $partner['contact_email'];
            $_SESSION['user_role'] = 'partner';
            
            // Update last login
            $updateStmt = $this->db->prepare("UPDATE affiliate_partners SET updated_at = NOW() WHERE id = ?");
            $updateStmt->execute([$partner['id']]);
            
            return true;
        }
        
        return false;
    }
    
    /**
     * Check if user is logged in as partner
     */
    public function isPartnerLoggedIn() {
        return isset($_SESSION['partner_id']) && isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'partner';
    }
    
    /**
     * Get current partner data
     */
    public function getCurrentPartner() {
        if (!$this->isPartnerLoggedIn()) {
            return null;
        }
        
        $stmt = $this->db->prepare("SELECT * FROM affiliate_partners WHERE id = ?");
        $stmt->execute([$_SESSION['partner_id']]);
        return $stmt->fetch();
    }
    
    /**
     * Generate unique partner code
     */
    public function generatePartnerCode($partnerName) {
        $base = strtoupper(substr(preg_replace('/[^a-zA-Z0-9]/', '', $partnerName), 0, 5));
        $base = $base ?: 'AFF';
        $code = $base . rand(1000, 9999);
        
        // Ensure uniqueness
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM affiliate_partners WHERE partner_code = ?");
        $stmt->execute([$code]);
        $exists = $stmt->fetchColumn() > 0;
        
        if ($exists) {
            // Try with different number
            $code = $base . rand(10000, 99999);
        }
        
        return $code;
    }
    
    /**
     * Logout partner
     */
    public function logout() {
        unset($_SESSION['partner_id']);
        unset($_SESSION['partner_code']);
        unset($_SESSION['partner_name']);
        unset($_SESSION['partner_email']);
        if (isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'partner') {
            unset($_SESSION['user_role']);
        }
    }
    
    /**
     * Verify partner code is valid
     */
    public function verifyPartnerCode($partnerCode) {
        $stmt = $this->db->prepare("SELECT id FROM affiliate_partners WHERE partner_code = ? AND status = 'active'");
        $stmt->execute([$partnerCode]);
        $partner = $stmt->fetch();
        return $partner ? $partner['id'] : false;
    }
}

// Helper functions
function requirePartner() {
    $auth = new PartnerAuth();
    if (!$auth->isPartnerLoggedIn()) {
        require_once __DIR__ . '/functions.php';
        header('Location: ' . url('partner/login.php'));
        exit;
    }
}

function getCurrentPartner() {
    $auth = new PartnerAuth();
    return $auth->getCurrentPartner();
}

