<?php
/**
 * New Client Model
 * Handles client creation and integration with affiliate system
 */

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/ReferralProcessor.php';

class NewClientModel {
    private $db;
    private $processor;
    
    public function __construct() {
        $this->db = getDB();
        $this->processor = new ReferralProcessor();
    }
    
    /**
     * Create new client with affiliate attribution
     */
    public function createClient($userData, $subscriptionData = null, $orderData = null) {
        try {
            $this->db->beginTransaction();
            
            // Check if there's a lead for this email
            $leadStmt = $this->db->prepare("
                SELECT id FROM client_leads 
                WHERE lead_email = ? AND status != 'converted'
                ORDER BY created_at DESC
                LIMIT 1
            ");
            $leadStmt->execute([$userData['email']]);
            $lead = $leadStmt->fetch();
            $leadId = $lead ? $lead['id'] : null;
            
            // Create user
            $passwordHash = password_hash($userData['password'], PASSWORD_DEFAULT);
            $userStmt = $this->db->prepare("
                INSERT INTO users 
                (email, password_hash, business_name, contact_name, phone, address, city, state, zip_code, role, status)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'client', 'active')
            ");
            
            $userStmt->execute([
                $userData['email'],
                $passwordHash,
                $userData['business_name'] ?? $userData['company_name'] ?? '',
                $userData['contact_name'] ?? $userData['name'] ?? '',
                $userData['phone'] ?? null,
                $userData['address'] ?? null,
                $userData['city'] ?? null,
                $userData['state'] ?? null,
                $userData['zip_code'] ?? null
            ]);
            
            $userId = $this->db->lastInsertId();
            
            // Convert lead if exists
            if ($leadId) {
                $this->processor->convertLeadToClient($leadId, $userId, null);
            }
            
            // Create subscription if provided
            $subscriptionId = null;
            if ($subscriptionData) {
                $subStmt = $this->db->prepare("
                    INSERT INTO subscriptions 
                    (user_id, tier_id, start_date, renewal_date, status, auto_renew, payment_status)
                    VALUES (?, ?, CURDATE(), DATE_ADD(CURDATE(), INTERVAL 1 MONTH), 'active', TRUE, 'paid')
                ");
                $subStmt->execute([
                    $userId,
                    $subscriptionData['tier_id']
                ]);
                $subscriptionId = $this->db->lastInsertId();
            }
            
            // Create order if provided
            $orderId = null;
            if ($orderData && $subscriptionId) {
                $orderStmt = $this->db->prepare("
                    INSERT INTO client_orders 
                    (user_id, lead_id, order_type, subscription_id, order_amount, currency, order_status, order_date, completed_at)
                    VALUES (?, ?, 'subscription', ?, ?, 'USD', 'completed', NOW(), NOW())
                ");
                $orderStmt->execute([
                    $userId,
                    $leadId,
                    $subscriptionId,
                    $orderData['amount']
                ]);
                $orderId = $this->db->lastInsertId();
                
                // Process order conversion for affiliate
                if ($orderId) {
                    $this->processor->processOrderConversion($orderId, $userId);
                }
            }
            
            $this->db->commit();
            
            return [
                'success' => true,
                'user_id' => $userId,
                'subscription_id' => $subscriptionId,
                'order_id' => $orderId,
                'lead_id' => $leadId
            ];
            
        } catch (Exception $e) {
            $this->db->rollBack();
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}

