<?php
/**
 * PHPMailer-based SMTP Mailer
 */

require_once __DIR__ . '/../config/email_config.php';

// Load Composer autoload if available
if (file_exists(__DIR__ . '/../vendor/autoload.php')) {
    require_once __DIR__ . '/../vendor/autoload.php';
}

class Mailer
{
    private ?\PHPMailer\PHPMailer\PHPMailer $mailer = null;
    private array $lastError = [];

    public function __construct(array $overrides = [])
    {
        // Try to load PHPMailer directly if autoload didn't work
        if (!class_exists('PHPMailer\\PHPMailer\\PHPMailer')) {
            $phpmailerPath = __DIR__ . '/../vendor/phpmailer/phpmailer/src/PHPMailer.php';
            if (file_exists($phpmailerPath)) {
                require_once $phpmailerPath;
                require_once __DIR__ . '/../vendor/phpmailer/phpmailer/src/SMTP.php';
                require_once __DIR__ . '/../vendor/phpmailer/phpmailer/src/Exception.php';
            }
        }
        
        if (!class_exists('PHPMailer\\PHPMailer\\PHPMailer')) {
            $this->lastError = ['error' => 'PHPMailer not installed'];
            return;
        }

        $this->mailer = new \PHPMailer\PHPMailer\PHPMailer(true);

        // Use IMAP settings as default if SMTP settings are not provided
        // Convert IMAP host to SMTP (e.g., imap.gmail.com -> smtp.gmail.com)
        $imapHost = defined('IMAP_HOST') ? IMAP_HOST : '';
        $smtpHostFromImap = '';
        if (!empty($imapHost) && empty(SMTP_HOST)) {
            // Replace common IMAP prefixes with SMTP
            $smtpHostFromImap = preg_replace('/^imaps?\./', 'smtp.', $imapHost);
            // If no replacement occurred, try to derive from domain (fallback)
            if ($smtpHostFromImap === $imapHost && strpos($imapHost, '.') !== false) {
                $parts = explode('.', $imapHost, 2);
                $smtpHostFromImap = 'smtp.' . (count($parts) > 1 ? $parts[1] : $parts[0]);
            }
        }
        
        $host = $overrides['host'] ?? (!empty(SMTP_HOST) ? SMTP_HOST : $smtpHostFromImap);
        $port = (int)($overrides['port'] ?? (!empty(SMTP_PORT) ? SMTP_PORT : 587));
        $username = $overrides['username'] ?? (!empty(SMTP_USER) ? SMTP_USER : (defined('IMAP_USER') && !empty(IMAP_USER) ? IMAP_USER : ''));
        $password = $overrides['password'] ?? (!empty(SMTP_PASS) ? SMTP_PASS : (defined('IMAP_PASS') && !empty(IMAP_PASS) ? IMAP_PASS : ''));
        $secure = $overrides['secure'] ?? (!empty(SMTP_SECURE) ? SMTP_SECURE : ((defined('IMAP_ENCRYPTION') && IMAP_ENCRYPTION === 'ssl') ? 'ssl' : 'tls'));
        $fromEmail = $overrides['from_email'] ?? (!empty(SMTP_FROM_EMAIL) ? SMTP_FROM_EMAIL : (defined('IMAP_USER') && !empty(IMAP_USER) ? IMAP_USER : ''));
        $fromName = $overrides['from_name'] ?? (!empty(SMTP_FROM_NAME) ? SMTP_FROM_NAME : 'Marketing Platform');

        $this->mailer->isSMTP();
        $this->mailer->Host = $host;
        $this->mailer->Port = $port ?: 587;
        $this->mailer->SMTPAuth = !empty($username);
        if (!empty($secure)) {
            $this->mailer->SMTPSecure = $secure; // 'tls' or 'ssl'
        }
        if (!empty($username)) {
            $this->mailer->Username = $username;
            $this->mailer->Password = $password;
        }

        if (!empty($fromEmail)) {
            $this->mailer->setFrom($fromEmail, $fromName ?: '');
        }
    }

    public function send(array $message): array
    {
        if (!$this->mailer) {
            return $this->lastError ?: ['error' => 'Mailer not initialized'];
        }

        try {
            $this->mailer->clearAllRecipients();
            $this->mailer->Subject = $message['subject'] ?? '';
            $html = $message['html'] ?? '';
            $text = $message['text'] ?? '';

            if ($html) {
                $this->mailer->isHTML(true);
                $this->mailer->Body = $html;
                if ($text) {
                    $this->mailer->AltBody = $text;
                }
            } else {
                $this->mailer->isHTML(false);
                $this->mailer->Body = $text;
            }

            $to = $message['to'] ?? [];
            if (is_string($to)) {
                $this->mailer->addAddress($to);
            } elseif (is_array($to)) {
                foreach ($to as $recipient) {
                    if (is_string($recipient)) {
                        $this->mailer->addAddress($recipient);
                    } elseif (is_array($recipient) && !empty($recipient['email'])) {
                        $this->mailer->addAddress($recipient['email'], $recipient['name'] ?? '');
                    }
                }
            }

            if (!empty($message['reply_to'])) {
                $this->mailer->addReplyTo($message['reply_to']);
            }

            if (!empty($message['attachments']) && is_array($message['attachments'])) {
                foreach ($message['attachments'] as $filePath => $name) {
                    if (is_int($filePath)) {
                        $this->mailer->addAttachment($name);
                    } else {
                        $this->mailer->addAttachment($filePath, $name);
                    }
                }
            }

            // Prepare message first (needed to get raw MIME message)
            if (!$this->mailer->preSend()) {
                throw new \Exception($this->mailer->ErrorInfo);
            }
            
            // Get raw message BEFORE sending (so we can save it to IMAP Sent folder)
            $rawMessage = $this->mailer->getSentMIMEMessage();
            
            // Now send the email
            if (!$this->mailer->postSend()) {
                throw new \Exception($this->mailer->ErrorInfo);
            }
            
            // Save sent email to IMAP Sent folder (for Namecheap and other IMAP providers)
            if (!empty($rawMessage)) {
                $this->saveToImapSentFolder($rawMessage);
            } else {
                // Fallback: construct message after sending
                $this->saveToImapSentFolder();
            }
            
            return ['success' => true];
        } catch (\Throwable $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Save sent email to IMAP Sent folder (e.g., Namecheap)
     * This ensures sent emails appear in the Sent folder on the email server
     * 
     * @param string|null $rawMessage Pre-built raw message (from getSentMIMEMessage)
     */
    private function saveToImapSentFolder(?string $rawMessage = null): bool
    {
        // Only proceed if IMAP extension is available and IMAP settings are configured
        if (!function_exists('imap_open')) {
            return false;
        }
        
        $imapHost = defined('IMAP_HOST') ? IMAP_HOST : '';
        $imapUser = defined('IMAP_USER') ? IMAP_USER : '';
        $imapPass = defined('IMAP_PASS') ? IMAP_PASS : '';
        $imapPort = defined('IMAP_PORT') ? (int)IMAP_PORT : 993;
        $imapEncryption = defined('IMAP_ENCRYPTION') ? IMAP_ENCRYPTION : 'ssl';
        $sentFolder = defined('IMAP_SENT_FOLDER') ? IMAP_SENT_FOLDER : 'Sent';
        
        // Skip if IMAP not configured
        if (empty($imapHost) || empty($imapUser) || empty($imapPass)) {
            return false;
        }
        
        try {
            // Build IMAP connection string
            $flags = ($imapEncryption === 'ssl') ? '/ssl' : (($imapEncryption === 'tls') ? '/tls' : '');
            $mailboxPath = sprintf('{%s:%d%s}', $imapHost, $imapPort, $flags);
            
            // Connect to IMAP
            $imap = @imap_open($mailboxPath, $imapUser, $imapPass);
            if (!$imap) {
                error_log('IMAP connection failed for Sent folder: ' . imap_last_error());
                return false;
            }
            
            // Get the raw message if not provided
            if (empty($rawMessage)) {
                // Try to get from PHPMailer if available
                if (method_exists($this->mailer, 'getSentMIMEMessage')) {
                    $rawMessage = $this->mailer->getSentMIMEMessage();
                }
                
                // Fallback: construct message manually
                if (empty($rawMessage)) {
                    $rawMessage = $this->constructRawMessage();
                }
            }
            
            if (empty($rawMessage)) {
                imap_close($imap);
                error_log('Could not generate raw message for IMAP Sent folder');
                return false;
            }
            
            // Try common Sent folder names (Namecheap may use different names)
            $sentFolders = [
                $sentFolder,
                'INBOX.' . $sentFolder,
                'Sent',
                'INBOX.Sent',
                'Sent Messages',
                'INBOX.Sent Messages'
            ];
            
            $saved = false;
            foreach ($sentFolders as $folder) {
                // Try to append to folder (will fail if folder doesn't exist, but won't throw error)
                $result = @imap_append($imap, $mailboxPath . $folder, $rawMessage);
                if ($result) {
                    $saved = true;
                    error_log("Sent email saved to IMAP folder: {$folder}");
                    break;
                }
            }
            
            if (!$saved) {
                // Try to create Sent folder if it doesn't exist
                foreach ($sentFolders as $folder) {
                    $folderPath = $mailboxPath . $folder;
                    $mailboxes = @imap_list($imap, $mailboxPath, '*');
                    $folderExists = false;
                    
                    if ($mailboxes) {
                        foreach ($mailboxes as $mailbox) {
                            $decoded = mb_convert_encoding($mailbox, 'UTF-8', 'UTF7-IMAP');
                            if ($decoded === $folderPath || strpos($decoded, $folder) !== false) {
                                $folderExists = true;
                                break;
                            }
                        }
                    }
                    
                    if (!$folderExists) {
                        // Try to create folder (some servers don't allow this)
                        @imap_createmailbox($imap, imap_utf7_encode($folderPath));
                    }
                    
                    // Try again to append
                    $result = @imap_append($imap, $folderPath, $rawMessage);
                    if ($result) {
                        $saved = true;
                        error_log("Sent email saved to IMAP folder: {$folder}");
                        break;
                    }
                }
            }
            
            imap_close($imap);
            
            if (!$saved) {
                error_log('Failed to save sent email to IMAP Sent folder. Check folder name configuration.');
            }
            
            return $saved;
        } catch (\Throwable $e) {
            error_log('Error saving to IMAP Sent folder: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Construct raw MIME message from PHPMailer object
     */
    private function constructRawMessage(): string
    {
        try {
            // PHPMailer has a method to get the raw message
            // Use it if available, otherwise construct manually
            if (method_exists($this->mailer, 'getSentMIMEMessage')) {
                return $this->mailer->getSentMIMEMessage();
            }
            
            // Manual construction as fallback
            $boundary = '----=_Part_' . md5(time() . rand());
            $message = '';
            
            // Headers
            $message .= 'From: ' . $this->mailer->encodeHeader($this->mailer->FromName) . ' <' . $this->mailer->From . ">\r\n";
            $message .= 'To: ';
            $toAddresses = [];
            foreach ($this->mailer->getToAddresses() as $addr) {
                $toAddresses[] = $addr[0];
            }
            $message .= implode(', ', $toAddresses) . "\r\n";
            
            if (!empty($this->mailer->getCcAddresses())) {
                $message .= 'Cc: ' . implode(', ', array_column($this->mailer->getCcAddresses(), 0)) . "\r\n";
            }
            if (!empty($this->mailer->getBccAddresses())) {
                $message .= 'Bcc: ' . implode(', ', array_column($this->mailer->getBccAddresses(), 0)) . "\r\n";
            }
            
            $message .= 'Subject: ' . $this->mailer->encodeHeader($this->mailer->Subject) . "\r\n";
            $message .= 'Date: ' . date('r') . "\r\n";
            $message .= 'MIME-Version: 1.0' . "\r\n";
            
            if ($this->mailer->isHTML()) {
                $message .= 'Content-Type: multipart/alternative; boundary="' . $boundary . '"' . "\r\n";
                $message .= "\r\n";
                $message .= '--' . $boundary . "\r\n";
                $message .= 'Content-Type: text/plain; charset=' . $this->mailer->CharSet . "\r\n";
                $message .= 'Content-Transfer-Encoding: ' . $this->mailer->Encoding . "\r\n";
                $message .= "\r\n";
                $message .= $this->mailer->AltBody . "\r\n";
                $message .= '--' . $boundary . "\r\n";
                $message .= 'Content-Type: text/html; charset=' . $this->mailer->CharSet . "\r\n";
                $message .= 'Content-Transfer-Encoding: ' . $this->mailer->Encoding . "\r\n";
                $message .= "\r\n";
                $message .= $this->mailer->Body . "\r\n";
                $message .= '--' . $boundary . '--' . "\r\n";
            } else {
                $message .= 'Content-Type: text/plain; charset=' . $this->mailer->CharSet . "\r\n";
                $message .= 'Content-Transfer-Encoding: ' . $this->mailer->Encoding . "\r\n";
                $message .= "\r\n";
                $message .= $this->mailer->Body . "\r\n";
            }
            
            return $message;
        } catch (\Throwable $e) {
            error_log('Error constructing raw message: ' . $e->getMessage());
            return '';
        }
    }
}


