<?php
/**
 * Lead Capture
 * Captures leads from forms and associates them with affiliate referrals
 */

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/ReferralProcessor.php';

class LeadCapture {
    private $db;
    private $processor;
    
    public function __construct() {
        $this->db = getDB();
        $this->processor = new ReferralProcessor();
    }
    
    /**
     * Capture lead from form submission
     */
    public function captureLead($data) {
        $email = filter_var($data['email'] ?? '', FILTER_SANITIZE_EMAIL);
        $name = trim($data['name'] ?? '');
        $phone = trim($data['phone'] ?? '');
        $companyName = trim($data['company_name'] ?? '');
        $notes = trim($data['notes'] ?? '');
        
        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['success' => false, 'error' => 'Valid email is required'];
        }
        
        // Process referral if affiliate session exists
        $result = $this->processor->processReferral($email, $name, $phone, $companyName, $notes);
        
        if (!$result['success']) {
            // Still create lead even without affiliate
            $stmt = $this->db->prepare("
                INSERT INTO client_leads 
                (lead_email, lead_name, lead_phone, company_name, lead_source, status, notes, created_at)
                VALUES (?, ?, ?, ?, 'direct', 'new', ?, NOW())
            ");
            
            $stmt->execute([$email, $name, $phone, $companyName, $notes]);
            $leadId = $this->db->lastInsertId();
            
            return [
                'success' => true,
                'lead_id' => $leadId,
                'affiliate' => false
            ];
        }
        
        return $result;
    }
    
    /**
     * Get lead by ID
     */
    public function getLead($leadId) {
        $stmt = $this->db->prepare("SELECT * FROM client_leads WHERE id = ?");
        $stmt->execute([$leadId]);
        return $stmt->fetch();
    }
    
    /**
     * Update lead status
     */
    public function updateLeadStatus($leadId, $status, $notes = null) {
        $validStatuses = ['new', 'contacted', 'qualified', 'converted', 'lost'];
        if (!in_array($status, $validStatuses)) {
            return ['success' => false, 'error' => 'Invalid status'];
        }
        
        $stmt = $this->db->prepare("
            UPDATE client_leads 
            SET status = ?, notes = COALESCE(?, notes), updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$status, $notes, $leadId]);
        
        return ['success' => true];
    }
}

