<?php
/**
 * Commission Calculator
 * Calculates commissions for affiliate partners based on conversion values
 */

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/webhook_helper.php';

class CommissionCalculator {
    private $db;
    
    public function __construct() {
        $this->db = getDB();
    }
    
    /**
     * Calculate commission amount based on partner settings and order value
     */
    public function calculateCommission($partnerId, $orderValue, $conversionType = 'subscription') {
        // Get partner details
        $stmt = $this->db->prepare("SELECT * FROM affiliate_partners WHERE id = ?");
        $stmt->execute([$partnerId]);
        $partner = $stmt->fetch();
        
        if (!$partner || $partner['status'] !== 'active') {
            return 0;
        }
        
        $commissionRate = floatval($partner['commission_rate']);
        $commissionType = $partner['commission_type'];
        
        // Apply commission based on type
        switch ($commissionType) {
            case 'percentage':
                $commission = ($orderValue * $commissionRate) / 100;
                break;
                
            case 'fixed':
                $commission = $commissionRate; // In this case, rate is the fixed amount
                break;
                
            case 'tiered':
                $commission = $this->calculateTieredCommission($partner, $orderValue);
                break;
                
            default:
                $commission = ($orderValue * $commissionRate) / 100;
        }
        
        // Apply type-specific multipliers if needed
        $commission = $this->applyConversionTypeMultiplier($commission, $conversionType, $partner);
        
        return round($commission, 2);
    }
    
    /**
     * Calculate tiered commission based on order value
     */
    private function calculateTieredCommission($partner, $orderValue) {
        // Get tier structure (stored in notes or separate table)
        // For now, simple tier logic
        $tierLevel = intval($partner['tier_level']);
        $baseRate = floatval($partner['commission_rate']);
        
        // Tier 1: 0-1000 = base rate
        // Tier 2: 1000-5000 = base rate * 1.2
        // Tier 3: 5000+ = base rate * 1.5
        
        if ($orderValue >= 5000) {
            return ($orderValue * ($baseRate * 1.5)) / 100;
        } elseif ($orderValue >= 1000) {
            return ($orderValue * ($baseRate * 1.2)) / 100;
        } else {
            return ($orderValue * $baseRate) / 100;
        }
    }
    
    /**
     * Apply conversion type multiplier
     */
    private function applyConversionTypeMultiplier($commission, $conversionType, $partner) {
        // You can customize multipliers per conversion type
        $multipliers = [
            'lead' => 0.1, // 10% of normal commission for leads
            'trial_signup' => 0.5, // 50% for trials
            'subscription' => 1.0, // Full commission
            'renewal' => 0.8, // 80% for renewals
            'upgrade' => 0.5 // 50% for upgrades (only on upgrade amount)
        ];
        
        $multiplier = $multipliers[$conversionType] ?? 1.0;
        return $commission * $multiplier;
    }
    
    /**
     * Record a conversion and calculate commission
     */
    public function recordConversion($partnerId, $clickId, $leadId, $orderId, $userId, $conversionType, $conversionValue) {
        // Calculate commission
        $commissionRate = $this->getPartnerCommissionRate($partnerId);
        $commissionAmount = $this->calculateCommission($partnerId, $conversionValue, $conversionType);
        
        // Record conversion
        $stmt = $this->db->prepare("
            INSERT INTO affiliate_conversions 
            (partner_id, click_id, lead_id, order_id, user_id, conversion_type, conversion_value, 
             commission_rate, commission_amount, commission_status, converted_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())
        ");
        
        $stmt->execute([
            $partnerId, $clickId, $leadId, $orderId, $userId,
            $conversionType, $conversionValue, $commissionRate, $commissionAmount
        ]);
        
        $conversionId = $this->db->lastInsertId();
        
        // Update click as converted
        if ($clickId) {
            $updateClick = $this->db->prepare("
                UPDATE affiliate_clicks 
                SET converted = TRUE, conversion_id = ? 
                WHERE id = ?
            ");
            $updateClick->execute([$conversionId, $clickId]);
        }
        
        // Update lead as converted
        if ($leadId) {
            $updateLead = $this->db->prepare("
                UPDATE client_leads 
                SET status = 'converted', converted_at = NOW(), converted_to_user_id = ?
                WHERE id = ?
            ");
            $updateLead->execute([$userId, $leadId]);
        }
        
        // Trigger webhook for conversion
        if (isset($_SESSION['user_id'])) {
            triggerWebhookEvent('conversion.tracked', [
                'partner_id' => $partnerId,
                'conversion_id' => $conversionId,
                'conversion_type' => $conversionType,
                'conversion_value' => $conversionValue,
                'commission_amount' => $commissionAmount,
                'order_id' => $orderId,
                'click_id' => $clickId,
                'lead_id' => $leadId,
                'timestamp' => date('c')
            ]);
        }
        
        return $conversionId;
    }
    
    /**
     * Get partner's current commission rate
     */
    private function getPartnerCommissionRate($partnerId) {
        $stmt = $this->db->prepare("SELECT commission_rate FROM affiliate_partners WHERE id = ?");
        $stmt->execute([$partnerId]);
        $partner = $stmt->fetch();
        return $partner ? floatval($partner['commission_rate']) : 0;
    }
    
    /**
     * Get total pending commissions for a partner
     */
    public function getPendingCommissions($partnerId) {
        $stmt = $this->db->prepare("
            SELECT SUM(commission_amount) as total 
            FROM affiliate_conversions 
            WHERE partner_id = ? AND commission_status = 'pending'
        ");
        $stmt->execute([$partnerId]);
        $result = $stmt->fetch();
        return floatval($result['total'] ?? 0);
    }
    
    /**
     * Get total paid commissions for a partner
     */
    public function getPaidCommissions($partnerId) {
        $stmt = $this->db->prepare("
            SELECT SUM(commission_amount) as total 
            FROM affiliate_conversions 
            WHERE partner_id = ? AND commission_status = 'paid'
        ");
        $stmt->execute([$partnerId]);
        $result = $stmt->fetch();
        return floatval($result['total'] ?? 0);
    }
}

