<?php
/**
 * Affiliate Tracker
 * Tracks clicks, sessions, and attribution for affiliate partners
 */

require_once __DIR__ . '/database.php';
require_once __DIR__ . '/functions.php';

class AffiliateTracker {
    private $db;
    
    public function __construct() {
        $this->db = getDB();
    }
    
    /**
     * Track an affiliate click
     */
    public function trackClick($partnerCode, $referrerUrl = null, $landingPage = null) {
        // Verify partner code
        $stmt = $this->db->prepare("SELECT id FROM affiliate_partners WHERE partner_code = ? AND status = 'active'");
        $stmt->execute([$partnerCode]);
        $partner = $stmt->fetch();
        
        if (!$partner) {
            return false;
        }
        
        // Get or create session ID
        if (!isset($_SESSION['affiliate_session_id'])) {
            $_SESSION['affiliate_session_id'] = bin2hex(random_bytes(16));
        }
        
        // Get device type
        $deviceType = $this->getDeviceType($_SERVER['HTTP_USER_AGENT'] ?? '');
        
        // Extract UTM parameters
        $utmSource = $_GET['utm_source'] ?? null;
        $utmMedium = $_GET['utm_medium'] ?? null;
        $utmCampaign = $_GET['utm_campaign'] ?? null;
        $utmTerm = $_GET['utm_term'] ?? null;
        $utmContent = $_GET['utm_content'] ?? null;
        
        // Store partner code in session
        $_SESSION['affiliate_code'] = $partnerCode;
        $_SESSION['affiliate_partner_id'] = $partner['id'];
        
        // Record click
        $stmt = $this->db->prepare("
            INSERT INTO affiliate_clicks 
            (partner_id, partner_code, visitor_ip, visitor_user_agent, referrer_url, landing_page,
             utm_source, utm_medium, utm_campaign, utm_term, utm_content, session_id, device_type)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $partner['id'],
            $partnerCode,
            $_SERVER['REMOTE_ADDR'] ?? null,
            $_SERVER['HTTP_USER_AGENT'] ?? null,
            $referrerUrl ?: ($_SERVER['HTTP_REFERER'] ?? null),
            $landingPage ?: ($_SERVER['REQUEST_URI'] ?? null),
            $utmSource,
            $utmMedium,
            $utmCampaign,
            $utmTerm,
            $utmContent,
            $_SESSION['affiliate_session_id'],
            $deviceType
        ]);
        
        return $this->db->lastInsertId();
    }
    
    /**
     * Get current affiliate from session
     */
    public function getCurrentAffiliate() {
        if (isset($_SESSION['affiliate_code']) && isset($_SESSION['affiliate_partner_id'])) {
            return [
                'partner_code' => $_SESSION['affiliate_code'],
                'partner_id' => $_SESSION['affiliate_partner_id']
            ];
        }
        return null;
    }
    
    /**
     * Get device type from user agent
     */
    private function getDeviceType($userAgent) {
        if (empty($userAgent)) {
            return 'unknown';
        }
        
        $mobilePattern = '/(android|iphone|ipad|ipod|mobile)/i';
        $tabletPattern = '/(tablet|ipad|playbook|silk)/i';
        
        if (preg_match($tabletPattern, $userAgent)) {
            return 'tablet';
        } elseif (preg_match($mobilePattern, $userAgent)) {
            return 'mobile';
        } else {
            return 'desktop';
        }
    }
    
    /**
     * Get latest click for current session
     */
    public function getLatestClick() {
        if (!isset($_SESSION['affiliate_session_id'])) {
            return null;
        }
        
        $stmt = $this->db->prepare("
            SELECT * FROM affiliate_clicks 
            WHERE session_id = ? 
            ORDER BY clicked_at DESC 
            LIMIT 1
        ");
        $stmt->execute([$_SESSION['affiliate_session_id']]);
        return $stmt->fetch();
    }
    
    /**
     * Get click statistics for a partner
     */
    public function getClickStats($partnerId, $startDate = null, $endDate = null) {
        $whereClause = "partner_id = ?";
        $params = [$partnerId];
        
        if ($startDate) {
            $whereClause .= " AND clicked_at >= ?";
            $params[] = $startDate;
        }
        
        if ($endDate) {
            $whereClause .= " AND clicked_at <= ?";
            $params[] = $endDate;
        }
        
        $stmt = $this->db->prepare("
            SELECT 
                COUNT(*) as total_clicks,
                COUNT(DISTINCT session_id) as unique_sessions,
                COUNT(DISTINCT visitor_ip) as unique_visitors,
                SUM(CASE WHEN converted = TRUE THEN 1 ELSE 0 END) as conversions
            FROM affiliate_clicks 
            WHERE $whereClause
        ");
        $stmt->execute($params);
        return $stmt->fetch();
    }
}

