# Shared Database Setup Guide

## Overview

All sites (ai-itinc.com/public_html and affiliate-it.com) use the **same database** located in `affiliate-it.com/market/config/database.php`. This ensures seamless data sharing across all platforms.

## Directory Structure

```
htdocs/1.2/
├── affiliate-it.com/
│   └── market/
│       ├── config/
│       │   └── database.php          ← MAIN DATABASE CONFIG
│       └── includes/
│           ├── database.php          ← Database connection class
│           ├── functions.php        ← Shared functions
│           ├── base_path.php        ← Path helpers
│           └── AffiliateTracker.php  ← Affiliate tracking class
│
└── public_html/                      ← ai-itinc.com public sites
    ├── acceleratedadz/
    │   └── includes/
    │       ├── base_path.php         → Points to affiliate-it.com/market
    │       ├── database.php          → Uses affiliate-it.com database
    │       └── functions.php        → Uses affiliate-it.com functions
    │
    └── acceleratedleadz/
        └── includes/
            ├── base_path.php         → Points to affiliate-it.com/market
            ├── database.php          → Uses affiliate-it.com database
            ├── functions.php         → Uses affiliate-it.com functions
            └── AffiliateTracker.php  → Uses affiliate-it.com class
```

## Database Configuration

### Main Configuration File

**Location**: `affiliate-it.com/market/config/database.php`

```php
<?php
// Database Configuration
define('DB_HOST', 'localhost');
define('DB_NAME', 'mindsybq_market');
define('DB_USER', 'mindsybq_appuser');
define('DB_PASS', 'M@rket-2025$');
define('DB_CHARSET', 'utf8mb4');
```

### How It Works

1. **affiliate-it.com sites** directly use `affiliate-it.com/market/config/database.php`

2. **public_html sites** include files that:
   - Automatically detect the path to `affiliate-it.com/market/`
   - Load the shared database configuration
   - Use the same database connection class

## Path Detection

The public_html includes automatically detect the affiliate-it.com path using multiple fallback methods:

1. **Relative Path**: `../../../affiliate-it.com/market/`
2. **Alternative Relative**: `../../affiliate-it.com/market/`
3. **Document Root Based**: Calculates from `$_SERVER['DOCUMENT_ROOT']`

## Database Connection

### In affiliate-it.com

```php
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/includes/database.php';

$db = getDB(); // Uses shared connection
```

### In public_html/acceleratedadz or acceleratedleadz

```php
require_once __DIR__ . '/includes/database.php';

$db = getDB(); // Automatically uses affiliate-it.com database config
```

The `database.php` file in public_html includes:
- Automatically finds `affiliate-it.com/market/config/database.php`
- Loads the shared database configuration
- Uses the same `Database` class and `getDB()` function

## Shared Classes

### AffiliateTracker

Both sites use the same `AffiliateTracker` class:

- **Location**: `affiliate-it.com/market/includes/AffiliateTracker.php`
- **Public HTML Reference**: `public_html/acceleratedleadz/includes/AffiliateTracker.php`
  - Automatically loads the shared class
  - Falls back to minimal implementation if not found

### Functions

Shared utility functions in `affiliate-it.com/market/includes/functions.php` are available to all sites:

- `escape()` - XSS protection
- `url()` - URL generation
- `formatCurrency()` - Currency formatting
- `formatDate()` - Date formatting
- And more...

## Database Schema

All sites share these tables:

### Ad Management
- `acceleratedadz_campaigns`
- `acceleratedadz_ad_groups`
- `acceleratedadz_ads`
- `acceleratedadz_performance`

### Lead Management
- `email_subscribers`
- `client_leads`
- `accelerated_leadz_email_queue`
- `accelerated_leadz_campaigns`

### Affiliate System
- `affiliate_partners`
- `affiliate_clicks`
- `affiliate_conversions`

### User Management
- `users`
- `campaigns`
- `campaign_ads`
- `saved_ads`
- `api_keys`

## Verification

### Test Database Connection

Create a test file: `public_html/test_db.php`

```php
<?php
require_once __DIR__ . '/acceleratedadz/includes/database.php';

try {
    $db = getDB();
    $stmt = $db->query("SELECT COUNT(*) as count FROM users");
    $result = $stmt->fetch();
    echo "Database connected! Users table has {$result['count']} records.";
} catch (Exception $e) {
    echo "Error: " . $e->getMessage();
}
```

### Verify Shared Schema

Run this in both locations to confirm they see the same data:

```php
// In affiliate-it.com/market/test.php
$db = getDB();
$stmt = $db->query("SELECT DATABASE() as db");
echo $stmt->fetch()['db']; // Should show: mindsybq_market

// In public_html/acceleratedadz/test.php
require_once __DIR__ . '/includes/database.php';
$db = getDB();
$stmt = $db->query("SELECT DATABASE() as db");
echo $stmt->fetch()['db']; // Should show: mindsybq_market
```

## Troubleshooting

### Issue: "Database connection failed"

**Solutions**:
1. Verify database credentials in `affiliate-it.com/market/config/database.php`
2. Check MySQL service is running
3. Verify database exists: `SHOW DATABASES LIKE 'mindsybq_market';`
4. Check user permissions: `GRANT ALL ON mindsybq_market.* TO 'mindsybq_appuser'@'localhost';`

### Issue: "Cannot find affiliate-it.com/market path"

**Solutions**:
1. Verify directory structure matches expected layout
2. Check file permissions on include files
3. Verify `__DIR__` resolves correctly
4. Add debug output to see detected path:
   ```php
   echo "Detected path: " . $marketPath;
   ```

### Issue: "Class AffiliateTracker not found"

**Solutions**:
1. Verify `affiliate-it.com/market/includes/AffiliateTracker.php` exists
2. Check path detection in `public_html/acceleratedleadz/includes/AffiliateTracker.php`
3. Ensure database is loaded before AffiliateTracker
4. Check PHP error logs for exact path issues

## Benefits of Shared Database

✅ **Single Source of Truth**: All data in one place  
✅ **Consistency**: Same schema across all sites  
✅ **Simplified Updates**: Update schema once, affects all sites  
✅ **Real-time Sync**: Changes immediately visible everywhere  
✅ **Easier Backups**: One database to backup  
✅ **Better Performance**: No cross-database joins needed  

## Environment Configuration

For local development, ensure `.env` files (if used) point to the same database:

**affiliate-it.com/market/.env**:
```
DB_HOST=localhost
DB_NAME=mindsybq_market
DB_USER=mindsybq_appuser
DB_PASS=M@rket-2025$
```

**public_html/.env** (if separate, otherwise uses affiliate-it.com config):
```
# Will use affiliate-it.com/market/config/database.php
```

## Migration Notes

If you have existing separate databases:

1. Export data from old databases
2. Import into `mindsybq_market`
3. Update all public_html includes to use shared configuration
4. Test all functionality
5. Remove old database configurations

