# Production Deployment Guide

## Overview

This guide covers deploying the marketing platform where **ai-itinc.com** is the main hosting site and **affiliate-it.com** points to the same server/directory.

## Domain Architecture

### Production Setup
- **Main Host:** `ai-itinc.com` (primary domain)
- **Affiliate Domain:** `affiliate-it.com` (points to ai-itinc.com)
- **Both domains:** Serve from same `/mk/market/` directory
- **Public endpoints:**
  - Ads: `https://ai-itinc.com/acceleratedadz.php`
  - Leads: `https://ai-itinc.com/acceleratedleadz.php`

## Pre-Deployment Checklist

- [ ] Database schema imported
- [ ] Configuration files updated
- [ ] SSL certificates installed
- [ ] Domain DNS configured
- [ ] Email service configured (IMAP/SMTP)
- [ ] Cron jobs set up
- [ ] File permissions set
- [ ] Backup strategy in place

## Server Configuration

### Apache Virtual Host Setup

**Single Virtual Host (Both Domains):**

```apache
<VirtualHost *:80>
    ServerName ai-itinc.com
    ServerAlias affiliate-it.com www.ai-itinc.com www.affiliate-it.com
    DocumentRoot /var/www/html/mk/market
    
    <Directory /var/www/html/mk/market>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
    
    # Logs
    ErrorLog ${APACHE_LOG_DIR}/ai-itinc-error.log
    CustomLog ${APACHE_LOG_DIR}/ai-itinc-access.log combined
</VirtualHost>

<VirtualHost *:443>
    ServerName ai-itinc.com
    ServerAlias affiliate-it.com www.ai-itinc.com www.affiliate-it.com
    DocumentRoot /var/www/html/mk/market
    
    # SSL Configuration
    SSLEngine on
    SSLCertificateFile /etc/ssl/certs/ai-itinc.com.crt
    SSLCertificateKeyFile /etc/ssl/private/ai-itinc.com.key
    SSLCertificateChainFile /etc/ssl/certs/ai-itinc.com-chain.crt
    
    <Directory /var/www/html/mk/market>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
    
    # Logs
    ErrorLog ${APACHE_LOG_DIR}/ai-itinc-ssl-error.log
    CustomLog ${APACHE_LOG_DIR}/ai-itinc-ssl-access.log combined
</VirtualHost>
```

### Nginx Configuration

```nginx
server {
    listen 80;
    listen [::]:80;
    server_name ai-itinc.com affiliate-it.com www.ai-itinc.com www.affiliate-it.com;
    
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    listen [::]:443 ssl http2;
    server_name ai-itinc.com affiliate-it.com www.ai-itinc.com www.affiliate-it.com;
    
    root /var/www/html/mk/market;
    index index.php;
    
    # SSL Configuration
    ssl_certificate /etc/ssl/certs/ai-itinc.com.crt;
    ssl_certificate_key /etc/ssl/private/ai-itinc.com.key;
    ssl_trusted_certificate /etc/ssl/certs/ai-itinc.com-chain.crt;
    
    # Security headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;
    
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }
    
    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
    }
    
    # Deny access to hidden files
    location ~ /\. {
        deny all;
    }
}
```

## DNS Configuration

### Point Both Domains to Same Server

**A Records:**
```
ai-itinc.com          → YOUR_SERVER_IP
affiliate-it.com      → YOUR_SERVER_IP
www.ai-itinc.com      → YOUR_SERVER_IP
www.affiliate-it.com  → YOUR_SERVER_IP
```

**CNAME Alternative:**
```
affiliate-it.com      → ai-itinc.com (CNAME)
www.affiliate-it.com  → www.ai-itinc.com (CNAME)
```

## Configuration Files

### 1. Database Configuration

Edit `config/database.php`:
```php
define('DB_HOST', 'localhost');
define('DB_NAME', 'mindsybq_market');
define('DB_USER', 'your_db_user');
define('DB_PASS', 'your_secure_password');
define('DB_CHARSET', 'utf8mb4');
```

### 2. Domain Configuration

The system auto-detects production based on domain. No changes needed to `config/domain_config.php` if domains are correctly configured.

### 3. Email Configuration

Edit `config/email_config.php`:
```php
// IMAP Settings (Roundbox)
define('IMAP_HOST', 'imap.roundbox.com');
define('IMAP_PORT', 993);
define('IMAP_ENCRYPTION', 'ssl');
define('IMAP_USER', 'your-email@domain.com');
define('IMAP_PASS', 'your-password');

// SMTP Settings
define('SMTP_HOST', 'smtp.roundbox.com');
define('SMTP_PORT', 587);
define('SMTP_SECURE', 'tls');
define('SMTP_USER', 'your-email@domain.com');
define('SMTP_PASS', 'your-password');
```

## Cron Jobs Setup

### Email Queue Worker (Every Minute)

```bash
* * * * * /usr/bin/php /var/www/html/mk/market/api/acceleratedleadz_worker.php >> /var/log/acceleratedleadz_worker.log 2>&1
```

### IMAP Processing (Every 5 Minutes)

```bash
*/5 * * * * /usr/bin/php /var/www/html/mk/market/api/acceleratedleadz.php?action=process_imap >> /var/log/imap_processor.log 2>&1
```

## File Permissions

```bash
# Set proper ownership
sudo chown -R www-data:www-data /var/www/html/mk/market

# Set directory permissions
find /var/www/html/mk/market -type d -exec chmod 755 {} \;

# Set file permissions
find /var/www/html/mk/market -type f -exec chmod 644 {} \;

# Writable directories
chmod -R 775 /var/www/html/mk/market/uploads
chmod -R 775 /var/www/html/mk/market/assets
```

## Security Checklist

- [ ] SSL certificates installed and valid
- [ ] Database credentials secured
- [ ] `.env` file excluded from version control
- [ ] Error reporting disabled in production
- [ ] File upload limits configured
- [ ] CSRF protection enabled
- [ ] Rate limiting configured
- [ ] Regular backups scheduled

## Post-Deployment Testing

### 1. Test Main Platform
```
https://ai-itinc.com/mk/market/login.php
https://affiliate-it.com/mk/market/login.php (should work)
```

### 2. Test AcceleratedAdz
```
https://ai-itinc.com/acceleratedadz.php?id=1
```

### 3. Test AcceleratedLeadz
```
https://ai-itinc.com/acceleratedleadz.php?ref=TEST123
```

### 4. Test API Endpoints
```bash
curl https://ai-itinc.com/mk/market/api/acceleratedadz.php?action=get_campaigns
```

### 5. Test Database Connection
- Login to platform
- Create test campaign
- Verify data appears in database

## Monitoring

### Log Files to Monitor
- Apache/Nginx error logs
- PHP error log: `/var/log/php-errors.log`
- Application logs: `error_log` in project root
- Cron logs: `/var/log/acceleratedleadz_worker.log`

### Key Metrics
- Database connection errors
- API response times
- Email queue length
- Failed email sends
- Server resource usage

## Backup Strategy

### Database Backups
```bash
# Daily backup
0 2 * * * mysqldump -u user -p password mindsybq_market > /backups/db_$(date +\%Y\%m\%d).sql
```

### File Backups
```bash
# Weekly backup
0 3 * * 0 tar -czf /backups/files_$(date +\%Y\%m\%d).tar.gz /var/www/html/mk/market
```

## Troubleshooting

### Issue: Domains not resolving
- Check DNS propagation: `dig ai-itinc.com`
- Verify virtual host configuration
- Check server firewall rules

### Issue: SSL certificate errors
- Verify certificate is valid
- Check certificate chain
- Ensure intermediate certificates installed

### Issue: Database connection failed
- Verify MySQL is running
- Check credentials in `config/database.php`
- Test connection: `mysql -u user -p`

### Issue: Cron jobs not running
- Check cron service: `systemctl status cron`
- Verify file paths are absolute
- Check permissions on PHP executable

## Support

For deployment issues:
- Email: support@ai-itinc.com
- Check logs for detailed error messages
- Review `LOCAL_SETUP.md` for testing locally first

