# JWT Integration Guide

## Overview

This guide explains how to use JWT (JSON Web Tokens) for secure API authentication between **ai-itinc.com** (public_html) and **affiliate-it.com** (CRM platform).

## Architecture

### JWT Flow

```
┌─────────────────┐         ┌──────────────────┐         ┌─────────────────┐
│  ai-itinc.com  │──────▶│  JWT Generator   │────────▶│ affiliate-it.com│
│  (public_html)  │       │   /jwt/api/      │         │  (API endpoint)  │
└─────────────────┘       └──────────────────┘         └─────────────────┘
      │                          │                                │
      │  1. Generate Token       │                                │
      │─────────────────────────▶│                                │
      │                          │                                │
      │                          │  2. Return JWT                 │
      │◀─────────────────────────│                                │
      │                          │                                │
      │  3. API Call with JWT     │                                │
      │──────────────────────────────────────────────────────────▶│
      │                          │                                │
      │                          │  4. Verify Token               │
      │                          │     Extract User Context      │
      │                          │                                │
      │  5. API Response          │                                │
      │◀──────────────────────────────────────────────────────────│
```

## Setup

### 1. Configure JWT Secret Key

Set a shared secret key for both domains. This should be a strong, random string.

**For ai-itinc.com (public_html):**
Create or update `.env` file:
```bash
JWT_SECRET_KEY=your-very-secure-random-secret-key-here
```

**For affiliate-it.com (CRM):**
Create or update `affiliate-it.com/market/.env`:
```bash
JWT_SECRET_KEY=your-very-secure-random-secret-key-here
```

**Important:** Use the **same secret key** on both domains!

### 2. JWT Generator Location

The JWT generator is located at:
- `public_html/jwt/` - For ai-itinc.com
- `jwt/` - Root level (alternative location)

Both use the same `JWTGenerator.php` class.

## Usage Examples

### Generating JWT Tokens

#### Method 1: Using JWTGenerator Class (Recommended)

```php
<?php
require_once __DIR__ . '/jwt/src/JWTGenerator.php';

use JWTGenerator\JWTGenerator;

$secret = $_ENV['JWT_SECRET_KEY'] ?? 'your-secret-key';
$generator = new JWTGenerator($secret);

$payload = [
    'user_id' => 123,
    'api_access' => true,
    'source' => 'acceleratedleadz'
];

$token = $generator->generateToken($payload, 3600); // 1 hour expiration
echo $token;
```

#### Method 2: Using API Endpoint

```php
<?php
$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL => 'https://ai-itinc.com/jwt/api/generate.php',
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => json_encode([
        'payload' => [
            'user_id' => 123,
            'api_access' => true
        ],
        'secret_key' => 'your-secret-key',
        'expiration' => 3600
    ]),
    CURLOPT_HTTPHEADER => [
        'Content-Type: application/json'
    ]
]);

$response = curl_exec($ch);
$result = json_decode($response, true);
$token = $result['token'];
curl_close($ch);
```

### Using JWT in API Calls

#### Sending Lead Data from AcceleratedLeadz

```php
<?php
require_once __DIR__ . '/includes/LeadAPI.php';

$api = new LeadAPI([
    'api_base_url' => 'https://affiliate-it.com/market/api',
    'jwt_secret' => $_ENV['JWT_SECRET_KEY'] ?? 'your-secret-key'
]);

$leadData = [
    'email' => 'user@example.com',
    'name' => 'John Doe',
    'phone' => '555-1234',
    'company_name' => 'Example Corp',
    'notes' => 'Interested in product',
    'user_id' => 123,
    'campaign_id' => 5,
    'affiliate_code' => 'PARTNER123'
];

$result = $api->captureLead($leadData);

if ($result['success']) {
    echo "Lead captured successfully!";
} else {
    echo "Error: " . $result['error'];
}
```

#### Fetching Ads from AcceleratedAdz

```php
<?php
require_once __DIR__ . '/includes/AffiliateAPI.php';
require_once __DIR__ . '/includes/JWTClient.php';

$config = [
    'api' => [
        'base_url' => 'https://affiliate-it.com/market/api',
        'use_jwt' => true,
        'jwt_secret' => $_ENV['JWT_SECRET_KEY'] ?? 'your-secret-key',
        'timeout' => 30
    ]
];

$api = new AffiliateAPI($config);
$ads = $api->getAds($clientId);
```

### Verifying JWT Tokens on API Endpoints

#### In affiliate-it.com API endpoints:

```php
<?php
// affiliate-it.com/market/api/affiliate_tracking.php

require_once __DIR__ . '/../includes/database.php';

// Extract JWT from Authorization header
$authHeader = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
$token = null;

if (preg_match('/Bearer (.+)/', $authHeader, $matches)) {
    $token = $matches[1];
}

if (!$token) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'JWT token required']);
    exit;
}

// Verify JWT token
require_once __DIR__ . '/../../jwt/src/JWTGenerator.php';
use JWTGenerator\JWTGenerator;

$jwtSecret = $_ENV['JWT_SECRET_KEY'] ?? 'your-secret-key';
$generator = new JWTGenerator($jwtSecret);

try {
    $payload = $generator->decodeToken($token);
    
    // Extract user context
    $userId = $payload['user_id'] ?? 0;
    $apiAccess = $payload['api_access'] ?? false;
    
    if (!$apiAccess) {
        throw new Exception('Invalid token permissions');
    }
    
    // Token is valid, proceed with API logic
    // ...
    
} catch (Exception $e) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Invalid token: ' . $e->getMessage()]);
    exit;
}
```

## Helper Classes

### LeadAPI Class

Located at: `public_html/acceleratedleadz/includes/LeadAPI.php`

**Features:**
- Automatic JWT generation
- Lead data submission
- Click tracking
- Error handling

**Usage:**
```php
$api = new LeadAPI(['api_base_url' => 'https://affiliate-it.com/market/api']);
$result = $api->captureLead($leadData);
```

### JWTClient Class

Located at: `public_html/acceleratedadz/includes/JWTClient.php`

**Features:**
- Simple JWT token generation
- Token verification
- Fallback to API endpoint if local generator unavailable

**Usage:**
```php
$jwt = new JWTClient($secret);
$token = $jwt->generateToken(['user_id' => 123]);
```

## Security Best Practices

1. **Use Strong Secrets**: Generate a random 32+ character secret key
   ```bash
   openssl rand -base64 32
   ```

2. **Set Token Expiration**: Don't use tokens that last forever
   ```php
   $token = $generator->generateToken($payload, 3600); // 1 hour
   ```

3. **Validate Token on Every Request**: Always verify JWT on API endpoints

4. **Use HTTPS**: Always use HTTPS in production for token transmission

5. **Don't Log Tokens**: Never log JWT tokens in error logs or debug output

6. **Rotate Secrets**: Periodically rotate JWT secret keys (requires coordination)

## Testing

### Test JWT Generation

```bash
curl -X POST https://ai-itinc.com/jwt/api/generate.php \
  -H "Content-Type: application/json" \
  -d '{
    "payload": {"test": true},
    "secret_key": "your-secret-key",
    "expiration": 3600
  }'
```

### Test JWT Verification

```bash
curl -X POST https://ai-itinc.com/jwt/api/verify.php \
  -H "Content-Type: application/json" \
  -d '{
    "token": "your-jwt-token-here",
    "secret_key": "your-secret-key"
  }'
```

### Test API with JWT

```bash
# Generate token first, then use in API call
TOKEN=$(curl -s -X POST https://ai-itinc.com/jwt/api/generate.php \
  -H "Content-Type: application/json" \
  -d '{"payload": {"api_access": true}, "secret_key": "your-secret-key"}' \
  | jq -r '.token')

# Use token in API call
curl -X POST https://affiliate-it.com/market/api/affiliate_tracking.php \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/x-www-form-urlencoded" \
  -d "action=capture_lead&email=test@example.com"
```

## Troubleshooting

### Token Generation Fails

**Problem:** JWT generation returns null

**Solutions:**
1. Check JWT_SECRET_KEY is set in environment
2. Verify JWTGenerator.php file exists
3. Check composer dependencies (firebase/jwt-php)
4. Check PHP error logs

### Token Verification Fails

**Problem:** API rejects token with "Invalid token" error

**Solutions:**
1. Verify both domains use the same secret key
2. Check token hasn't expired
3. Verify algorithm matches (HS256 default)
4. Check token format (should be three parts separated by dots)

### CORS Issues

**Problem:** Browser blocks API requests

**Solutions:**
1. Ensure API endpoints set CORS headers:
   ```php
   header('Access-Control-Allow-Origin: https://ai-itinc.com');
   header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
   header('Access-Control-Allow-Headers: Authorization, Content-Type');
   ```

2. Handle OPTIONS preflight requests

## Migration from API Keys

If currently using API keys, you can migrate to JWT:

1. Keep API key support as fallback
2. Generate JWT tokens using existing API keys
3. Gradually migrate endpoints to JWT-only
4. Remove API key support once migration complete

## Related Documentation

- `ARCHITECTURE_OVERVIEW.md` - System architecture
- `DOMAIN_CONNECTIONS.md` - Domain integration guide
- `API_DOCUMENTATION.md` - API reference (if exists)

