# Marketing Service Platform - Complete Installation Guide

## Overview

This comprehensive marketing service platform provides a complete solution for managing multi-channel marketing campaigns with a 3-tier subscription model. The platform includes Google Ads, Facebook Ads, email marketing, direct mail, discount code management, and advanced analytics.

## ✅ Completed Features

### Core Platform
- ✅ **Database Schema** - Complete MySQL schema with all required tables
- ✅ **Authentication System** - Role-based access control (admin/client)
- ✅ **Subscription Management** - 3-tier pricing system with restrictions
- ✅ **Settings Dashboard** - Secure API credential management

### Client Features
- ✅ **Client Dashboard** - Campaign metrics and analytics overview
- ✅ **Campaign Management** - Create and manage marketing campaigns
- ✅ **Discount Code Generator** - Advanced rule engine with usage tracking
- ✅ **Analytics Dashboard** - User metrics and conversion tracking
- ✅ **Reports System** - Exportable reports (CSV/PDF) with custom date ranges
- ✅ **Direct Mail Management** - Address validation and cost estimation
- ✅ **Tracking Code Generator** - Website integration for metrics

### Admin Features
- ✅ **Admin Dashboard** - System-wide metrics and client management
- ✅ **Client Management** - CRUD operations for client accounts
- ✅ **Subscription Management** - Upgrade/downgrade functionality
- ✅ **Campaign Oversight** - Monitor all client campaigns
- ✅ **Revenue Tracking** - Financial metrics and reporting

### API Integrations
- ✅ **Google Ads API** - Campaign creation, management, and performance tracking
- ✅ **Facebook Marketing API** - Ad management and audience targeting
- ✅ **Email Services** - SendGrid and Mailchimp integration
- ✅ **Metrics Tracking** - UTM parameters and conversion tracking

## Installation Requirements

### Server Requirements
- **PHP**: 8.1 or higher
- **MySQL**: 8.0 or higher
- **Web Server**: Apache or Nginx
- **SSL Certificate**: Required for production
- **Memory**: Minimum 512MB RAM
- **Storage**: 1GB minimum

### PHP Extensions Required
```bash
php-mysql
php-curl
php-json
php-mbstring
php-openssl
php-gd
php-zip
```

### Optional Libraries (for enhanced features)
```bash
# Google Ads API
composer require googleads/google-ads-php

# Facebook Marketing API
composer require facebook/php-business-sdk

# Email Services
composer require sendgrid/sendgrid
composer require mailchimp/marketing

# PDF Generation (for reports)
composer require tecnickcom/tcpdf
```

## Installation Steps

### 1. Database Setup

```sql
-- Create database
CREATE DATABASE marketing_platform CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

-- Create user
CREATE USER 'marketing_user'@'localhost' IDENTIFIED BY 'secure_password';
GRANT ALL PRIVILEGES ON marketing_platform.* TO 'marketing_user'@'localhost';
FLUSH PRIVILEGES;
```

### 2. Import Database Schema

```bash
mysql -u marketing_user -p marketing_platform < database/schema.sql
```

### 3. Configure Environment

Create `.env` file in project root:

```env
# Database Configuration
DB_HOST=localhost
DB_NAME=marketing_platform
DB_USER=marketing_user
DB_PASS=secure_password
DB_CHARSET=utf8mb4

# Security
ENCRYPTION_KEY=your_32_character_encryption_key_here
SESSION_SECRET=your_session_secret_key_here

# Google Ads API (Optional)
GOOGLE_ADS_CLIENT_ID=your_client_id
GOOGLE_ADS_CLIENT_SECRET=your_client_secret
GOOGLE_ADS_DEVELOPER_TOKEN=your_developer_token
GOOGLE_ADS_REFRESH_TOKEN=your_refresh_token
GOOGLE_ADS_API_VERSION=v14

# Facebook Marketing API (Optional)
FACEBOOK_APP_ID=your_app_id
FACEBOOK_APP_SECRET=your_app_secret
FACEBOOK_ACCESS_TOKEN=your_access_token
FACEBOOK_API_VERSION=v18.0

# Email Services (Optional)
SENDGRID_API_KEY=your_sendgrid_api_key
SENDGRID_FROM_EMAIL=noreply@yourdomain.com
SENDGRID_FROM_NAME=Your Business Name

MAILCHIMP_API_KEY=your_mailchimp_api_key
MAILCHIMP_SERVER_PREFIX=us1
```

### 4. Web Server Configuration

#### Apache (.htaccess)
```apache
RewriteEngine On
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^(.*)$ index.php [QSA,L]

# Security headers
Header always set X-Content-Type-Options nosniff
Header always set X-Frame-Options DENY
Header always set X-XSS-Protection "1; mode=block"
```

#### Nginx
```nginx
server {
    listen 80;
    server_name yourdomain.com;
    root /var/www/marketing-platform;
    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
    }
}
```

### 5. File Permissions

```bash
# Set proper permissions
chmod 755 /var/www/marketing-platform
chmod 644 /var/www/marketing-platform/*.php
chmod 644 /var/www/marketing-platform/includes/*.php
chmod 644 /var/www/marketing-platform/api/*.php
chmod 644 /var/www/marketing-platform/client/*.php
chmod 644 /var/www/marketing-platform/admin/*.php

# Make writable directories
chmod 777 /var/www/marketing-platform/uploads
chmod 777 /var/www/marketing-platform/logs
```

### 6. SSL Certificate (Production)

```bash
# Using Let's Encrypt
sudo apt install certbot python3-certbot-apache
sudo certbot --apache -d yourdomain.com
```

## Default Login Credentials

### Admin Account
- **Email**: admin@marketingplatform.com
- **Password**: admin123
- **Role**: Administrator

⚠️ **Important**: Change the default admin password immediately after installation!

## Subscription Tiers

### Starter Plan - $299/month
- Google Ads: $200/month budget
- Facebook Ads: $150/month budget
- Email: 1,000 contacts, 4 campaigns/month
- Discount Codes: 5 codes/month
- Direct Mail: 500 postcards quarterly

### Professional Plan - $599/month
- Google Ads: $500/month budget
- Facebook Ads: $400/month budget
- Email: 5,000 contacts, unlimited campaigns
- Discount Codes: Unlimited with advanced rules
- Direct Mail: 1,500 postcards monthly
- A/B Testing and optimization

### Enterprise Plan - $999/month
- Google Ads: $1,200/month budget
- Facebook Ads: $800/month budget
- Email: Unlimited contacts and campaigns
- Premium discount system
- Direct Mail: 5,000 postcards monthly
- Full API access and custom integrations

## API Integration Setup

### Google Ads API
1. Create Google Cloud Project
2. Enable Google Ads API
3. Create OAuth 2.0 credentials
4. Generate refresh token
5. Apply for developer token

### Facebook Marketing API
1. Create Facebook App
2. Add Marketing API product
3. Generate access token
4. Get ad account ID

### Email Services
1. **SendGrid**: Create account, generate API key
2. **Mailchimp**: Create account, get API key and server prefix

## Cron Jobs

Set up these cron jobs for automated tasks:

```bash
# Sync ad performance data (every hour)
0 * * * * /usr/bin/php /var/www/marketing-platform/api/sync_performance.php

# Send scheduled email campaigns (every 15 minutes)
*/15 * * * * /usr/bin/php /var/www/marketing-platform/api/send_scheduled_emails.php

# Generate daily reports (daily at 6 AM)
0 6 * * * /usr/bin/php /var/www/marketing-platform/api/generate_daily_reports.php
```

## Security Checklist

- [ ] Change default admin password
- [ ] Set strong encryption keys
- [ ] Enable HTTPS
- [ ] Configure firewall
- [ ] Regular database backups
- [ ] Update PHP and MySQL regularly
- [ ] Monitor error logs
- [ ] Implement rate limiting

## Troubleshooting

### Common Issues

1. **Database Connection Error**
   - Check database credentials in `.env`
   - Verify MySQL service is running
   - Ensure user has proper permissions

2. **API Integration Not Working**
   - Verify API credentials in settings
   - Check if required libraries are installed
   - Review error logs

3. **Permission Denied Errors**
   - Check file permissions
   - Verify web server user ownership
   - Ensure writable directories exist

### Log Files
- **PHP Errors**: `/var/log/php_errors.log`
- **Application Logs**: `/var/www/marketing-platform/logs/`
- **Web Server Logs**: `/var/log/apache2/error.log` or `/var/log/nginx/error.log`

## Support

For technical support or questions:
- **Documentation**: Check this installation guide
- **Error Logs**: Review application and server logs
- **Database**: Verify schema and data integrity

## License

This marketing platform is proprietary software. All rights reserved.

---

**Installation Complete!** 🎉

Your marketing service platform is now ready for use. Access the admin panel to configure clients and start managing marketing campaigns.
