# Developer Guide - Marketing Platform

## Project Structure

```
accelerated-market/
├── admin/              # Admin panel pages
├── api/                # API endpoints
├── assets/             # CSS, JavaScript, images
├── client/             # Client dashboard pages
├── config/             # Configuration files
├── database/           # SQL schema files
├── docs/               # Documentation
├── includes/           # PHP includes and utilities
├── tests/              # Test files
├── index.php           # Landing page
├── login.php           # Login page
├── setup.php           # Setup wizard
└── composer.json       # PHP dependencies
```

## Database Schema

### Core Tables

- **users**: Client and admin accounts
- **subscriptions**: Active subscriptions
- **subscription_tiers**: Pricing tier definitions
- **campaigns**: Marketing campaigns
- **ad_performance**: Campaign performance metrics
- **email_campaigns**: Email marketing campaigns
- **discount_codes**: Discount/promo codes
- **direct_mail_campaigns**: Direct mail orders
- **user_metrics**: Visitor tracking and analytics
- **api_credentials**: Stored API keys (encrypted)
- **transactions**: Payment history

## Code Conventions

### PHP Standards
- Use PSR-4 autoloading
- Follow PSR-12 coding style
- Use prepared statements for all database queries
- Always escape output with `escape()` function
- Validate all user input

### Security Best Practices
- Always use prepared statements
- Hash passwords with `password_hash()`
- Escape all output with `htmlspecialchars()`
- Validate email addresses
- Use CSRF tokens on forms
- Check user permissions

### Database Access
```php
// Get database connection
$db = getDB();

// Query with prepared statement
$stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$userId]);
$user = $stmt->fetch();
```

### Helper Functions

```php
// Escape output
escape($string)

// Format currency
formatCurrency($amount)

// Format date
formatDate($date, $format = 'Y-m-d')

// Validate email
isValidEmail($email)

// Set flash message
setFlashMessage('success', 'Message')

// Get flash message
getFlashMessage()

// Generate CSRF token field
csrfField()

// Validate CSRF token
requireCSRFToken()
```

## Adding New Features

### Creating a New Admin Page

1. Create file in `admin/` directory
2. Require authentication:
```php
require_once __DIR__ . '/../includes/auth.php';
requireAdmin();
```
3. Use database helpers:
```php
$db = getDB();
```
4. Include header/footer:
```php
require_once __DIR__ . '/../includes/header.php';
// ... page content ...
require_once __DIR__ . '/../includes/footer.php';
```

### Creating a New API Endpoint

1. Create file in `api/` directory
2. Set JSON headers:
```php
header('Content-Type: application/json');
```
3. Handle authentication
4. Return JSON response:
```php
echo json_encode(['success' => true, 'data' => $data]);
```

## Testing

### Running Tests
```bash
vendor/bin/phpunit
```

### Test Structure
- `tests/Unit/`: Unit tests
- `tests/Integration/`: Integration tests
- `tests/bootstrap.php`: Test setup

## Deployment

### Production Checklist
- [ ] Change default admin password
- [ ] Update database credentials
- [ ] Enable HTTPS
- [ ] Set proper file permissions
- [ ] Configure error logging
- [ ] Set up database backups
- [ ] Review security settings
- [ ] Test all functionality
- [ ] Remove setup.php or protect it

### File Permissions
```bash
chmod 755 -R .
chmod 644 config/*.php
chmod 600 config/database.php  # More secure
```

## Troubleshooting

### Common Issues

#### Database Connection Failed
- Check MySQL is running
- Verify credentials in `config/database.php`
- Check database exists
- Verify user permissions

#### Session Issues
- Check session directory is writable
- Verify session.save_path in php.ini
- Check session cookies are enabled

#### Permission Errors
- Check file permissions
- Verify web server user has access
- Check directory permissions

## Contributing

### Code Style
- Follow PSR-12 standards
- Comment complex logic
- Use descriptive variable names
- Keep functions focused and small

### Version Control
- Create feature branches
- Write meaningful commit messages
- Test before committing
- Update documentation

