# Deployment Guide - Marketing Platform

## Prerequisites

- PHP 7.4 or higher
- MySQL 5.7+ or MariaDB 10.3+
- Apache/Nginx web server (recommended) or PHP built-in server (development only)
- Composer (for dependencies)
- Git (for version control)

## Production Deployment Steps

### 1. Server Setup

#### For Apache:
```apache
<VirtualHost *:80>
    ServerName your-domain.com
    DocumentRoot /var/www/marketing-platform
    
    <Directory /var/www/marketing-platform>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
    
    ErrorLog ${APACHE_LOG_DIR}/marketing-platform-error.log
    CustomLog ${APACHE_LOG_DIR}/marketing-platform-access.log combined
</VirtualHost>
```

#### For Nginx:
```nginx
server {
    listen 80;
    server_name your-domain.com;
    root /var/www/marketing-platform;
    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php7.4-fpm.sock;
        fastcgi_index index.php;
        include fastcgi_params;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
    }

    location ~ /\. {
        deny all;
    }
}
```

### 2. File Permissions

```bash
# Set ownership
sudo chown -R www-data:www-data /var/www/marketing-platform

# Set directory permissions
find /var/www/marketing-platform -type d -exec chmod 755 {} \;

# Set file permissions
find /var/www/marketing-platform -type f -exec chmod 644 {} \;

# Protect sensitive files
chmod 600 config/database.php
chmod 600 .env  # If using environment variables
```

### 3. Database Configuration

1. Create production database:
```sql
CREATE DATABASE marketing_platform_prod CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'marketing_user'@'localhost' IDENTIFIED BY 'STRONG_PASSWORD_HERE';
GRANT ALL PRIVILEGES ON marketing_platform_prod.* TO 'marketing_user'@'localhost';
FLUSH PRIVILEGES;
```

2. Update `config/database.php`:
```php
define('DB_HOST', 'localhost');
define('DB_NAME', 'marketing_platform_prod');
define('DB_USER', 'marketing_user');
define('DB_PASS', 'STRONG_PASSWORD_HERE');
define('DB_CHARSET', 'utf8mb4');
```

3. Import schema:
```bash
mysql -u marketing_user -p marketing_platform_prod < database/schema.sql
```

4. Apply performance indexes:
```bash
mysql -u marketing_user -p marketing_platform_prod < database/add_performance_indexes.sql
```

### 4. Security Configuration

1. **Environment Variables** (Recommended):
   - Create `.env` file (if implementing .env support)
   - Move sensitive data out of `config/database.php`
   - Add `.env` to `.gitignore`

2. **HTTPS/SSL**:
   - Install SSL certificate (Let's Encrypt recommended)
   - Force HTTPS redirects
   - Update `index.php` and other redirects to use HTTPS

3. **Error Reporting**:
   ```php
   // In php.ini or .htaccess
   display_errors = Off
   log_errors = On
   error_log = /var/log/php-errors.log
   ```

4. **Disable Setup**:
   ```bash
   # Remove or protect setup.php
   rm setup.php
   # OR
   chmod 600 setup.php
   # OR add .htaccess protection
   ```

### 5. Application Setup

1. Install dependencies:
```bash
composer install --no-dev --optimize-autoloader
```

2. Run setup (if still needed):
   - Visit `http://your-domain.com/setup.php`
   - Complete initial configuration
   - Remove `.setup_complete` if you want to re-run setup

3. Change default admin password:
   - Log in with default credentials
   - Change password immediately

### 6. Performance Optimization

1. **OPcache** (PHP):
   ```ini
   ; In php.ini
   opcache.enable=1
   opcache.memory_consumption=128
   opcache.max_accelerated_files=10000
   ```

2. **Database Indexes**:
   ```bash
   mysql -u marketing_user -p marketing_platform_prod < database/add_performance_indexes.sql
   ```

3. **Caching** (Future):
   - Implement Redis/Memcached for session storage
   - Add query result caching
   - Implement page caching

### 7. Backup Strategy

#### Database Backup:
```bash
# Daily backup script
mysqldump -u marketing_user -p marketing_platform_prod > /backups/db_$(date +%Y%m%d).sql

# Keep last 30 days
find /backups -name "db_*.sql" -mtime +30 -delete
```

#### Application Backup:
```bash
# Backup application files
tar -czf /backups/app_$(date +%Y%m%d).tar.gz /var/www/marketing-platform
```

### 8. Monitoring

1. **Log Files**:
   - Apache/Nginx access logs
   - PHP error logs
   - Application-specific logs

2. **Health Checks**:
   - Create `healthcheck.php` endpoint
   - Monitor database connectivity
   - Monitor disk space

3. **Uptime Monitoring**:
   - Use services like UptimeRobot or Pingdom
   - Set up alerts for downtime

## Post-Deployment Checklist

- [ ] Database is properly configured and secured
- [ ] All sensitive files have correct permissions (600)
- [ ] HTTPS is enabled and working
- [ ] Error reporting is disabled in production
- [ ] Setup.php is protected or removed
- [ ] Default admin password has been changed
- [ ] Database backups are configured
- [ ] Monitoring is set up
- [ ] SSL certificate is valid
- [ ] All environment variables are set
- [ ] File permissions are correct
- [ ] Performance indexes are applied
- [ ] OPcache is enabled
- [ ] Log rotation is configured

## Troubleshooting

### Database Connection Issues
- Verify MySQL is running: `systemctl status mysql`
- Check credentials in `config/database.php`
- Verify user permissions: `SHOW GRANTS FOR 'marketing_user'@'localhost';`
- Check firewall rules

### Permission Errors
- Verify file ownership: `ls -la`
- Check directory permissions: `stat directory_name`
- Ensure web server user has read access

### Performance Issues
- Check if indexes are applied: `SHOW INDEX FROM users;`
- Enable slow query log
- Review PHP-FPM/Apache worker settings
- Monitor database queries

### Security Issues
- Verify `display_errors = Off`
- Check file permissions
- Review access logs for suspicious activity
- Ensure HTTPS is forced

## Maintenance

### Regular Tasks
- **Daily**: Review error logs
- **Weekly**: Check disk space and database size
- **Monthly**: Review and update dependencies
- **Quarterly**: Security audit and updates

### Updates
1. Backup database and files
2. Pull latest code
3. Run database migrations (if any)
4. Update dependencies: `composer update`
5. Clear caches (if implemented)
6. Test functionality
7. Monitor error logs

## Support

For deployment issues, refer to:
- Developer Guide: `docs/DEVELOPER_GUIDE.md`
- Troubleshooting section in User Guide
- Check application logs: `/var/log/`

---

**Last Updated:** October 29, 2025

