# System Architecture Overview

## Domain Architecture

### 1. **ai-itinc.com (Public Domain - public_html)**
Hosts the public-facing advertising and lead capture systems:

#### **AcceleratedAdz** (`/acceleratedadz.php`)
- **Purpose**: Public advertising wall that displays rotating ads from clients/affiliates
- **Location**: `public_html/acceleratedadz/`
- **Functionality**:
  - Fetches active ads from affiliate-it.com CRM via API
  - Rotates through client ads automatically
  - Tracks ad impressions and clicks
  - Redirects users to AcceleratedLeadz landing pages
- **Integration**: 
  - Uses `AffiliateAPI.php` class to make cURL requests to affiliate-it.com
  - API endpoint: `https://affiliate-it.com/api/acceleratedadz.php`
  - Authentication: JWT tokens or API keys via Bearer token

#### **AcceleratedLeadz** (`/acceleratedleadz.php`)
- **Purpose**: Lead capture forms that receive traffic from ad clicks
- **Location**: `public_html/acceleratedleadz/`
- **Functionality**:
  - Displays public lead capture forms
  - Tracks affiliate codes from URL parameters
  - Captures lead data (name, email, phone, company, notes)
  - Sends lead data to affiliate-it.com CRM via API
  - Supports campaign-specific forms and email automation
- **Integration**:
  - Makes cURL POST requests to `https://affiliate-it.com/api/affiliate_tracking.php`
  - Or directly writes to shared database (if on same server)
  - Uses JWT tokens for authentication

#### **JWT Generator** (`/jwt/api/`)
- **Purpose**: Generate and verify JWT tokens for cross-domain API authentication
- **Location**: `public_html/jwt/` and `jwt/`
- **Functionality**:
  - Generates JWT tokens with configurable payload
  - Verifies JWT tokens for API authentication
  - Used for secure communication between domains

### 2. **affiliate-it.com (CRM Platform)**
Hosts the unified control plane for clients/affiliates:

#### **CRM System**
- **Location**: `affiliate-it.com/market/client/crm.php`
- **Features**:
  - Unified lead management
  - Client database management
  - Contact management
  - Activity tracking
  - Lead attribution tracking

#### **Ad Generator**
- **Location**: `affiliate-it.com/market/client/acceleratedadz.php`
- **Features**:
  - Visual ad builder interface
  - Campaign management
  - Ad group creation
  - Ad creation and editing
  - Performance analytics
  - Ad approval workflow

#### **Client Dashboard** (Unified Control Plane)
- **Location**: `affiliate-it.com/market/client/dashboard.php`
- **Features**:
  - Single dashboard for all client operations
  - Campaign overview
  - Lead intake monitoring
  - CRM data access
  - Analytics dashboard
  - Ad performance metrics
  - Lead source attribution

#### **Lead Import Configuration**
- **Location**: `affiliate-it.com/market/client/acceleratedleadz.php`
- **Features**:
  - Configure lead capture forms
  - Set up email automation campaigns
  - Configure lead import sources
  - Map lead data fields
  - Set up affiliate tracking

#### **Analytics**
- **Location**: `affiliate-it.com/market/client/analytics.php`
- **Features**:
  - Campaign performance metrics
  - Lead conversion tracking
  - ROI calculations
  - Traffic source analysis
  - A/B testing results

## Data Flow

### Complete User Journey

1. **Client/Affiliate Signs Up**
   ```
   https://affiliate-it.com/market/register.php
   → Creates account in CRM
   → Gets access to dashboard
   ```

2. **Client Creates Ad Campaign**
   ```
   https://affiliate-it.com/market/client/acceleratedadz.php
   → Creates campaign via ad builder
   → Sets destination URL: https://ai-itinc.com/acceleratedleadz?ref=PARTNER_CODE&campaign=5
   → Ad saved to acceleratedadz_campaigns/acceleratedadz_ads tables
   ```

3. **Ad Published to Public Wall**
   ```
   Ad becomes active
   → Available at: https://ai-itinc.com/acceleratedadz?id=123
   → Or via campaign: https://ai-itinc.com/acceleratedadz?campaign=5
   → Ad rotates through advertising wall
   ```

4. **User Clicks Ad**
   ```
   Click tracked in acceleratedadz_performance table
   → Redirects to: https://ai-itinc.com/acceleratedleadz?ref=PARTNER_CODE&campaign=5
   → Affiliate click tracked in affiliate_clicks table
   ```

5. **User Submits Lead Form**
   ```
   Form submission at ai-itinc.com/acceleratedleadz
   → Lead data sent via API to affiliate-it.com
   → Or directly written to shared database
   → Stored in:
     - email_subscribers (CRM)
     - client_leads (with affiliate attribution)
   ```

6. **Lead Appears in CRM**
   ```
   https://affiliate-it.com/market/client/crm.php
   → Lead visible with affiliate tracking
   → Campaign association visible
   → Ready for follow-up
   ```

## API Communication

### Authentication Methods

1. **JWT Token Authentication** (Recommended)
   ```
   Generate JWT: POST https://ai-itinc.com/jwt/api/generate.php
   {
     "payload": {
       "user_id": 123,
       "api_access": true
     },
     "secret_key": "shared_secret",
     "expiration": 3600
   }
   
   Use in API calls:
   Authorization: Bearer <jwt_token>
   ```

2. **API Key Authentication** (Alternative)
   ```
   Authorization: Bearer <api_key>
   Or: X-API-Key: <api_key>
   ```

### API Endpoints

#### AcceleratedAdz API
- **Base URL**: `https://affiliate-it.com/market/api/acceleratedadz.php`
- **Actions**:
  - `create_campaign` - Create new ad campaign
  - `create_ad_group` - Create ad group
  - `create_ad` - Create individual ad
  - `get_campaigns` - List user's campaigns
  - `track_click` - Track ad click (public)
  - `update_performance` - Update performance metrics
  - `get_performance` - Get campaign performance data

#### AcceleratedLeadz API
- **Base URL**: `https://affiliate-it.com/market/api/acceleratedleadz.php`
- **Actions**:
  - `process_imap` - Process IMAP emails
  - `queue_email` - Queue email for sending
  - `process_queue` - Process email queue (cron)
  - `create_campaign` - Create email campaign

#### Affiliate Tracking API
- **Base URL**: `https://affiliate-it.com/market/api/affiliate_tracking.php`
- **Actions**:
  - `track_click` - Track affiliate click
  - `capture_lead` - Capture and store lead
  - `get_affiliate_info` - Get affiliate information
  - `get_stats` - Get affiliate statistics

### Example API Integration

#### Fetching Ads from ai-itinc.com
```php
// In public_html/acceleratedadz/index.php or acceleratedadz.php
$config = include 'config.php';
$api = new AffiliateAPI($config);

// Get JWT token (optional, can use API key)
$jwtToken = generateJWT(['api_access' => true]);

// Fetch ads
$ads = $api->getAds($clientId);

// If using JWT
$api->setAuthToken($jwtToken);
$ads = $api->getAds($clientId);
```

#### Sending Lead Data from ai-itinc.com
```php
// In public_html/acceleratedleadz/acceleratedleadz.php
$leadData = [
    'action' => 'capture_lead',
    'email' => $email,
    'name' => $fullName,
    'phone' => $phone,
    'company_name' => $companyName,
    'notes' => $notes,
    'user_id' => $userId,
    'campaign_id' => $campaignId,
    'affiliate_code' => $affiliateCode
];

// Generate JWT token
$jwtToken = generateJWT([
    'user_id' => $userId,
    'api_access' => true,
    'timestamp' => time()
]);

// Send via cURL
$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL => 'https://affiliate-it.com/market/api/affiliate_tracking.php',
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => http_build_query($leadData),
    CURLOPT_HTTPHEADER => [
        'Authorization: Bearer ' . $jwtToken,
        'Content-Type: application/x-www-form-urlencoded'
    ]
]);

$response = curl_exec($ch);
curl_close($ch);
```

## Database Schema

### Shared Tables

Both domains share the same database for seamless data access:

**Ad Management**:
- `acceleratedadz_campaigns` - Ad campaigns
- `acceleratedadz_ad_groups` - Ad groups
- `acceleratedadz_ads` - Individual ads
- `acceleratedadz_performance` - Performance metrics

**Lead Management**:
- `email_subscribers` - CRM subscriber list
- `client_leads` - Captured leads with attribution
- `accelerated_leadz_email_queue` - Email automation queue
- `accelerated_leadz_campaigns` - Email campaigns

**Affiliate System**:
- `affiliate_partners` - Affiliate partner accounts
- `affiliate_clicks` - Click tracking
- `affiliate_conversions` - Conversion tracking

**User Management**:
- `users` - Client/affiliate accounts
- `campaigns` - General campaigns
- `campaign_ads` - Campaign-specific ads
- `saved_ads` - Saved ad templates

## Security & Authentication

### JWT Implementation
- **Location**: `public_html/jwt/` and `jwt/`
- **Usage**: Cross-domain API authentication
- **Flow**:
  1. Generate JWT token with payload
  2. Include in API request headers
  3. Verify on affiliate-it.com API endpoints
  4. Extract user context from token

### API Key System
- **Location**: `api_keys` table in database
- **Usage**: Alternative authentication method
- **Features**:
  - User-specific API keys
  - Expiration dates
  - Usage tracking
  - Active/inactive status

## Configuration

### Domain Configuration
- **File**: `affiliate-it.com/market/config/domain_config.php`
- **Settings**:
  - Domain detection (local vs production)
  - Base URLs for each domain
  - API endpoint URLs
  - CORS configuration

### API Configuration
- **File**: `public_html/acceleratedadz/config.php`
- **Settings**:
  - API base URL: `https://affiliate-it.com/api`
  - API key (if using key-based auth)
  - Timeout settings
  - Ad rotation settings

## Integration Patterns

### Option 1: Shared Database (Current)
- Both domains access same database
- Faster performance
- Simpler setup
- Requires same server or remote DB access

### Option 2: API-Only Communication (Recommended for separate servers)
- ai-itinc.com makes API calls to affiliate-it.com
- Uses JWT for authentication
- More secure for cross-server setup
- Better separation of concerns

### Option 3: Hybrid (Best Practice)
- Use API for writes (leads, tracking)
- Use direct DB access for reads (if on same server)
- Fallback to API if DB connection fails

## Troubleshooting

### Common Issues

1. **Ads Not Displaying**
   - Check API authentication
   - Verify ad status is 'active'
   - Check database connection
   - Verify API endpoint URLs

2. **Leads Not Capturing**
   - Verify affiliate code in URL
   - Check API authentication
   - Verify database permissions
   - Check API endpoint connectivity

3. **JWT Token Errors**
   - Verify shared secret key
   - Check token expiration
   - Verify token payload format
   - Check JWT generator configuration

4. **Cross-Domain CORS Issues**
   - Verify CORS headers in API responses
   - Check domain configuration
   - Verify allowed origins list

## Support & Documentation

- **Domain Connections**: `docs/DOMAIN_CONNECTIONS.md`
- **Public Endpoints**: `docs/PUBLIC_ENDPOINTS.md`
- **System Architecture**: `docs/SYSTEM_ARCHITECTURE.md`
- **API Documentation**: `docs/API_DOCUMENTATION.md` (if exists)

