# Application Flow Documentation

This document describes the complete flow of the marketing platform, focusing on the email-first campaign workflow.

## Overview

The platform follows an **email-first campaign workflow** where users:
1. Create email campaigns (default campaign type)
2. Create ads in the Ad Builder
3. Add ads to campaigns
4. Create email content with ad insertion
5. Send to CRM customers with rate limiting (1 per minute, configurable daily limit)
6. Ads are displayed on `public_html/acceleratedadz/acceleratedadz.php` for tracking

---

## 1. Campaign Creation Flow

**File**: `market/client/campaigns.php`

### Process:
1. User clicks "Create Campaign"
2. Modal opens with **Email Marketing pre-selected** as default
3. User enters campaign name, type, dates, budget
4. Campaign saved to `campaigns` table with `status='draft'`
5. Webhook triggered: `campaign.created`

### Database:
- **Table**: `campaigns`
- **Fields**: `user_id`, `campaign_name`, `campaign_type` (default: 'email'), `status`, `budget`, `start_date`, `end_date`
- **Default Type**: `'email'` (Email Marketing)

### Campaign Deletion:
- User can delete campaigns via "Delete" button
- Cascades to `campaign_ads` (removes linked ads)
- `email_campaigns` with `campaign_id` set to NULL (SET NULL)
- Webhook triggered: `campaign.deleted`

---

## 2. Ad Creation & Linking Flow

**Files**: 
- `market/client/ad_builder.php` (UI)
- `market/api/ad_builder.php` (API)

### Process:
1. User creates visual ad in Ad Builder
2. Ad saved to `saved_ads` table with `ad_data` (JSON: headline, description, productImage, cta, colors, etc.)
3. User clicks "Add to Campaign"
4. Selects campaign (includes 'email' type campaigns)
5. Ad linked via `campaign_ads` table:
   - `campaign_id` → links to campaign
   - `saved_ad_id` → links to saved ad data
   - `status='active'`

### Database:
- **Table**: `saved_ads` (ad design data)
- **Table**: `campaign_ads` (links ads to campaigns)
- **Relationship**: `campaign_ads.saved_ad_id` → `saved_ads.id`

---

## 3. Email Campaign Flow

**Files**:
- `market/client/email_campaigns.php` (UI)
- `market/api/email_service.php` (API, creation handler)

### Creation Process:
1. User clicks "Create Email Campaign"
2. Modal opens with:
   - Campaign name, subject, HTML content
   - **Recipient Source** radio buttons:
     - Manual entry (comma-separated emails)
     - **Send to CRM Customers** (from `email_subscribers` where `status='subscribed'`)
   - Daily limit field (for CRM sending)
   - Link to parent campaign (optional)
3. If CRM source selected:
   - Fetches all `email_subscribers` for user
   - Creates entries in `email_campaign_queue` table
   - Sets `daily_limit` in `email_campaigns`
4. Email campaign saved to `email_campaigns` table

### Ad Insertion Process:
1. User clicks "Insert Ad" button in email content editor
2. Selects campaign from dropdown (shows campaigns with ads)
3. Modal shows ads from selected campaign
4. User clicks "Insert This Ad"
5. Ad HTML generated with:
   - Ad data from `saved_ads.ad_data`
   - Tracking link: `acceleratedadz.php?campaign_ad={campaign_ad_id}&source=campaign&redirect={destinationUrl}`
6. HTML inserted into email content textarea
7. User can preview complete email with ads

### Database:
- **Table**: `email_campaigns`
  - `campaign_id` → links to parent `campaigns` (optional)
  - `email_content` → HTML with inserted ads
  - `daily_limit` → max emails per day (for CRM sending)
  - `daily_sent_count` → current day's sent count
  - `last_sent_date` → date of last send (for daily reset)
- **Table**: `email_campaign_queue` (for rate-limited sending)
  - `email_campaign_id` → links to email campaign
  - `email`, `first_name`, `last_name` → recipient data
  - `status` → 'pending', 'sending', 'sent', 'failed'

---

## 4. Email Sending with Rate Limiting

**File**: `market/api/email_queue_processor.php`

### Rate Limiting:
- **Rate**: 1 email per minute (60 seconds)
- **Daily Limit**: Configurable (e.g., send 100 per day out of 500 total CRM customers)
- **IMAP Integration**: Uses IMAP credentials from `config/email_config.php` (IMAP_HOST, IMAP_USER, IMAP_PASS)

### Process:
1. **Campaign Creation**:
   - If CRM source + daily limit: All CRM customers queued in `email_campaign_queue` with `status='pending'`

2. **Queue Processing** (run via cron every minute):
   - Process one email per campaign per run
   - Check daily limit (reset if new day)
   - If under limit:
     - Get next `pending` email from queue
     - Mark as `sending`
     - Send via SMTP (using IMAP credentials)
     - Mark as `sent` or `failed`
     - Update `daily_sent_count` in `email_campaigns`
   - If limit reached: Skip until next day

3. **Cron Setup**:
   ```bash
   * * * * * php /path/to/market/api/email_queue_processor.php
   ```

---

## 5. AcceleratedAdz Display Flow

**File**: `public_html/acceleratedadz/acceleratedadz.php`

### Public Ad Display:
1. Email contains ad link: `acceleratedadz.php?campaign_ad={id}&source=campaign&redirect={url}`
2. User clicks link in email
3. `acceleratedadz.php` receives `campaign_ad` parameter
4. Looks up `campaign_ads` table:
   ```sql
   SELECT ca.*, sa.ad_data, c.user_id
   FROM campaign_ads ca
   LEFT JOIN saved_ads sa ON ca.saved_ad_id = sa.id
   JOIN campaigns c ON ca.campaign_id = c.id
   WHERE ca.id = ? AND ca.status = 'active'
   ```
5. Displays ad with:
   - Headline, description, image from `ad_data`
   - CTA button with click tracking
6. On click:
   - Tracks click (logs or updates `ad_performance`)
   - Redirects to destination URL

### Click Tracking:
- Click link: `acceleratedadz.php?click={ad_id}&redirect={url}&source=campaign`
- Tracks click event
- Redirects to final destination

---

## 6. CRM Integration Flow

**File**: `market/client/crm.php`

### Customer Data:
- CRM customers stored in `email_subscribers` table
- Fields: `email`, `first_name`, `last_name`, `status` ('subscribed', 'unsubscribed', 'bounced')
- Users can import customers via CSV
- Customers managed in CRM interface

### Email Sending Integration:
- Email campaigns can select "Send to CRM Customers"
- Queries: `SELECT * FROM email_subscribers WHERE user_id = ? AND status = 'subscribed'`
- All subscribed customers queued for sending
- Rate limiting applied: 1 per minute, daily limit enforced

---

## 7. Subscription Override (Demo Mode)

**File**: `market/client/settings.php`

### Demo Mode:
- User toggle: "Enable Demo Mode (Override Subscription Limits)"
- Stored in `users.demo_mode_enabled` column (TINYINT(1))
- When enabled:
  - `canCreateDiscountCode()` returns `true` (bypasses limits)
  - `canSendEmailCampaign()` returns `true` (bypasses limits)
  - All subscription tier limits ignored

### Limit Functions:
- `market/includes/functions.php`:
  - `canCreateDiscountCode()` - checks `demo_mode_enabled`
  - `canSendEmailCampaign()` - checks `demo_mode_enabled`

---

## 8. Testing Dashboard Flow

**File**: `market/client/testing_dashboard.php`

### Features:
- Step-by-step flow testing interface
- Status overview (campaigns, ads, email campaigns, CRM customers)
- Quick links to all relevant pages
- Testing data display (latest campaigns, email campaigns)
- Visual flow with accordion steps:
  1. Create Email Campaign
  2. Create Ad in Ad Builder
  3. Add Ad to Campaign
  4. Create Email Content
  5. Insert Ads from Campaign
  6. Preview Email
  7. Send to CRM Customers
  8. View AcceleratedAdz

---

## Data Flow Diagram

```
User Action                    Database                    Result
─────────────────────────────────────────────────────────────────────
1. Create Campaign    →   campaigns table          →   Campaign created (type='email')
2. Create Ad          →   saved_ads table          →   Ad design saved
3. Add to Campaign    →   campaign_ads table       →   Ad linked to campaign
4. Create Email       →   email_campaigns table     →   Email campaign created
   Campaign
5. Insert Ad          →   email_content (HTML)     →   Ad HTML with tracking link
6. Select CRM         →   email_campaign_queue     →   Customers queued
   Customers
7. Queue Processor    →   email_campaign_queue     →   1 email sent per minute
   (every minute)        email_campaigns          →   Daily count updated
8. User clicks ad     →   acceleratedadz.php      →   Ad displayed
   in email                                        →   Click tracked
                                                    →   Redirect to destination
```

---

## Key Integration Points

### Campaigns ↔ Ad Builder:
- Ad Builder dropdown includes 'email' campaigns
- Ads saved to `saved_ads`, linked via `campaign_ads`

### Campaigns ↔ Email Campaigns:
- Email campaigns can link to parent campaign via `campaign_id`
- Email campaigns can insert ads from any campaign with active ads

### Email Campaigns ↔ CRM:
- CRM customers (`email_subscribers`) used as recipients
- Queue system for rate-limited sending
- Daily limit enforced

### Email Campaigns ↔ AcceleratedAdz:
- Ads in emails link to `acceleratedadz.php?campaign_ad={id}`
- Public page displays ad and tracks clicks
- Redirects to destination URL

### Webhooks:
- `campaign.created` - when campaign created
- `campaign.updated` - when campaign status changes
- `campaign.deleted` - when campaign deleted
- `email.sent` - when email sent
- `email.opened` - when email opened (if tracked)
- `email.clicked` - when email link clicked (if tracked)
- `conversion.tracked` - when conversion recorded
- `discount.used` - when discount code used

---

## Files Modified/Created

### Modified:
1. `market/client/campaigns.php` - Default email, delete functionality
2. `market/client/email_campaigns.php` - CRM integration, rate limiting UI
3. `market/api/email_service.php` - Campaign creation, queue setup
4. `market/client/settings.php` - Demo mode toggle
5. `market/includes/functions.php` - Demo mode override checks
6. `market/includes/header.php` - Testing dashboard link

### Created:
1. `public_html/acceleratedadz/acceleratedadz.php` - Public ad display
2. `market/client/testing_dashboard.php` - Testing interface
3. `market/api/email_queue_processor.php` - Rate-limited email processor
4. `market/database/migrations/006_email_rate_limiting.sql` - Database migration
5. `market/docs/APPLICATION_FLOW.md` - This documentation

---

## Next Steps

1. **Run Migration**: Execute `006_email_rate_limiting.sql` to add queue table and daily_limit column
2. **Setup Cron**: Configure cron to run `email_queue_processor.php` every minute
3. **Test Flow**: Use Testing Dashboard to verify complete flow
4. **Verify IMAP**: Ensure IMAP credentials are configured in `config/email_config.php`
5. **Enable Demo Mode**: Toggle demo mode in Settings for full testing

