-- ============================================================================
-- UNIFIED DATABASE SCHEMA FOR MARKETING PLATFORM
-- ============================================================================
-- MySQL 8.0+
-- 
-- This is the COMPLETE unified schema file containing:
--   - All core tables (users, subscriptions, campaigns, etc.)
--   - All extended features (templates, A/B testing, segmentation, etc.)
--   - All migrations (activity logs, admin assignments, affiliate system)
--   - AcceleratedLeadz: Email automation & CRM integration (IMAP from roundbox)
--   - AcceleratedAdz: Native ad platform (hosted on ai-itinc.com)
--   - All performance indexes
--   - All seed data
--
-- IMPORTANT: This is the ONLY schema file you need to import.
-- Other schema files (schema.sql, schema_updated.sql, schema_with_indexes.sql, 
-- and files in migrations/) are kept for reference but are now DEPRECATED.
--
-- Usage:
--   1. Import this file: mysql -u user -p database_name < database/schema_main.sql
--   2. OR use the setup wizard which will import it automatically
-- ============================================================================

CREATE DATABASE IF NOT EXISTS mindsybq_market CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE mindsybq_market;

-- Core: Users
CREATE TABLE IF NOT EXISTS users (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    business_name VARCHAR(255) NOT NULL,
    contact_name VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    address TEXT,
    city VARCHAR(100),
    state VARCHAR(50),
    zip_code VARCHAR(20),
    role ENUM('admin', 'client') DEFAULT 'client',
    status ENUM('active', 'inactive', 'suspended') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_role (role),
    INDEX idx_status (status),
    INDEX idx_role_status (role, status),
    INDEX idx_business_name (business_name),
    INDEX idx_role_business (role, business_name),
    INDEX idx_contact_name (contact_name),
    INDEX idx_created_at (created_at DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Subscription tiers
CREATE TABLE IF NOT EXISTS subscription_tiers (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    tier_name VARCHAR(50) NOT NULL UNIQUE,
    tier_level INT NOT NULL UNIQUE,
    monthly_price DECIMAL(10, 2) NOT NULL,
    google_ads_budget DECIMAL(10, 2) NOT NULL,
    facebook_ads_budget DECIMAL(10, 2) NOT NULL,
    email_contacts_limit INT DEFAULT NULL,
    email_campaigns_limit INT DEFAULT NULL,
    discount_codes_limit INT DEFAULT NULL,
    direct_mail_quantity INT NOT NULL,
    direct_mail_frequency ENUM('monthly', 'quarterly') DEFAULT 'monthly',
    features TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Subscriptions
CREATE TABLE IF NOT EXISTS subscriptions (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    tier_id INT UNSIGNED NOT NULL,
    start_date DATE NOT NULL,
    renewal_date DATE NOT NULL,
    status ENUM('active', 'cancelled', 'expired', 'pending') DEFAULT 'active',
    auto_renew BOOLEAN DEFAULT TRUE,
    payment_status ENUM('paid', 'pending', 'failed', 'refunded') DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (tier_id) REFERENCES subscription_tiers(id) ON DELETE RESTRICT,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_renewal_date (renewal_date),
    INDEX idx_user_status (user_id, status),
    INDEX idx_user_tier_status (user_id, tier_id, status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Campaigns
CREATE TABLE IF NOT EXISTS campaigns (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    campaign_name VARCHAR(255) NOT NULL,
    campaign_type ENUM('google_ads', 'facebook_ads', 'email', 'direct_mail', 'multi_channel') NOT NULL,
    status ENUM('draft', 'active', 'paused', 'completed', 'cancelled') DEFAULT 'draft',
    budget DECIMAL(10, 2),
    start_date DATE,
    end_date DATE,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_type (campaign_type),
    INDEX idx_status (status),
    INDEX idx_user_status_type (user_id, status, campaign_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Discount codes
CREATE TABLE IF NOT EXISTS discount_codes (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    code VARCHAR(50) NOT NULL,
    discount_type ENUM('percentage', 'fixed_amount') NOT NULL,
    discount_value DECIMAL(10, 2) NOT NULL,
    min_purchase_amount DECIMAL(10, 2) DEFAULT NULL,
    max_discount_amount DECIMAL(10, 2) DEFAULT NULL,
    usage_limit INT DEFAULT NULL,
    usage_count INT DEFAULT 0,
    start_date DATE,
    end_date DATE,
    status ENUM('active', 'inactive', 'expired') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_code (code),
    INDEX idx_user_id (user_id),
    INDEX idx_code (code),
    INDEX idx_status (status),
    INDEX idx_user_status_expiry (user_id, status, end_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Email campaigns
CREATE TABLE IF NOT EXISTS email_campaigns (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    campaign_id INT UNSIGNED,
    campaign_name VARCHAR(255) NOT NULL,
    subject_line VARCHAR(255) NOT NULL,
    email_content TEXT NOT NULL,
    recipient_count INT DEFAULT 0,
    sent_count INT DEFAULT 0,
    opened_count INT DEFAULT 0,
    clicked_count INT DEFAULT 0,
    bounced_count INT DEFAULT 0,
    status ENUM('draft', 'scheduled', 'sending', 'sent', 'failed') DEFAULT 'draft',
    scheduled_at DATETIME,
    sent_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (campaign_id) REFERENCES campaigns(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_user_status_created (user_id, status, created_at DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Email subscribers
CREATE TABLE IF NOT EXISTS email_subscribers (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    email VARCHAR(255) NOT NULL,
    first_name VARCHAR(100),
    last_name VARCHAR(100),
    status ENUM('subscribed', 'unsubscribed', 'bounced') DEFAULT 'subscribed',
    subscribed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    unsubscribed_at TIMESTAMP NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_email (user_id, email),
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_email (email)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Ad performance
CREATE TABLE IF NOT EXISTS ad_performance (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    campaign_id INT UNSIGNED,
    ad_platform ENUM('google_ads', 'facebook_ads') NOT NULL,
    ad_id VARCHAR(255),
    ad_name VARCHAR(255),
    impressions INT DEFAULT 0,
    clicks INT DEFAULT 0,
    conversions INT DEFAULT 0,
    cost DECIMAL(10, 2) DEFAULT 0.00,
    revenue DECIMAL(10, 2) DEFAULT 0.00,
    date DATE NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (campaign_id) REFERENCES campaigns(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_platform (ad_platform),
    INDEX idx_date (date),
    UNIQUE KEY unique_ad_date (ad_id, date),
    INDEX idx_user_date (user_id, date DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- User metrics
CREATE TABLE IF NOT EXISTS user_metrics (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    session_id VARCHAR(255),
    page_url VARCHAR(500),
    referrer VARCHAR(500),
    utm_source VARCHAR(100),
    utm_medium VARCHAR(100),
    utm_campaign VARCHAR(100),
    utm_term VARCHAR(100),
    utm_content VARCHAR(100),
    ip_address VARCHAR(45),
    user_agent TEXT,
    device_type ENUM('desktop', 'mobile', 'tablet', 'unknown') DEFAULT 'unknown',
    conversion_type VARCHAR(50),
    conversion_value DECIMAL(10, 2) DEFAULT NULL,
    discount_code_used VARCHAR(50),
    visited_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_visited_at (visited_at),
    INDEX idx_utm_source (utm_source),
    INDEX idx_utm_campaign (utm_campaign),
    INDEX idx_conversion_type (conversion_type),
    INDEX idx_user_visited (user_id, visited_at DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Direct mail campaigns (rich version)
CREATE TABLE IF NOT EXISTS direct_mail_campaigns (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    campaign_id INT UNSIGNED,
    campaign_name VARCHAR(255) NOT NULL,
    mail_type ENUM('postcard', 'letter', 'flyer') DEFAULT 'postcard',
    quantity INT NOT NULL,
    design_file VARCHAR(255),
    mailing_list_file VARCHAR(255),
    cost_per_unit DECIMAL(10, 2),
    total_cost DECIMAL(10, 2),
    postage_cost DECIMAL(10, 2),
    status ENUM('draft', 'scheduled', 'in_production', 'mailed', 'delivered', 'cancelled') DEFAULT 'draft',
    scheduled_date DATE,
    mailed_date DATE,
    delivery_date DATE,
    tracking_number VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (campaign_id) REFERENCES campaigns(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Transactions
CREATE TABLE IF NOT EXISTS transactions (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    subscription_id INT UNSIGNED,
    transaction_type ENUM('subscription', 'refund', 'adjustment') DEFAULT 'subscription',
    amount DECIMAL(10, 2) NOT NULL,
    currency VARCHAR(3) DEFAULT 'USD',
    payment_method VARCHAR(50),
    payment_status ENUM('pending', 'completed', 'failed', 'refunded') DEFAULT 'pending',
    transaction_id VARCHAR(255),
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (subscription_id) REFERENCES subscriptions(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_status (payment_status),
    INDEX idx_created_at (created_at),
    INDEX idx_user_date_status (user_id, created_at DESC, payment_status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- API credentials
CREATE TABLE IF NOT EXISTS api_credentials (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    service_name ENUM('google_ads', 'facebook_ads', 'sendgrid', 'mailchimp', 'usps') NOT NULL,
    credential_key VARCHAR(255) NOT NULL,
    credential_value TEXT NOT NULL,
    is_encrypted BOOLEAN DEFAULT TRUE,
    status ENUM('active', 'inactive') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_service (user_id, service_name),
    INDEX idx_user_id (user_id),
    INDEX idx_service (service_name)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Admin Activity Log (superset)
CREATE TABLE IF NOT EXISTS admin_activity_log (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    admin_id INT UNSIGNED NOT NULL,
    action_type ENUM('create', 'update', 'delete', 'activate', 'deactivate', 'suspend', 'export', 'login', 'logout', 'bulk_action') NOT NULL,
    entity_type VARCHAR(50) NOT NULL,
    entity_id INT UNSIGNED,
    description TEXT,
    old_values JSON,
    new_values JSON,
    affected_count INT DEFAULT 1,
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (admin_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_admin_id (admin_id),
    INDEX idx_action_type (action_type),
    INDEX idx_entity_type (entity_type),
    INDEX idx_created_at (created_at DESC),
    INDEX idx_admin_created (admin_id, created_at DESC),
    INDEX idx_entity (entity_type, entity_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Extended Features: Templates, A/B, Segmentation, Email Tracking, Reporting, API Keys, Webhooks, Files, Direct Mail Tracking
CREATE TABLE IF NOT EXISTS campaign_templates (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NULL,
    template_name VARCHAR(255) NOT NULL,
    template_type ENUM('google_ads', 'facebook_ads', 'email', 'direct_mail', 'multi_channel') NOT NULL,
    template_data JSON NOT NULL,
    description TEXT,
    is_public BOOLEAN DEFAULT FALSE,
    usage_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_template_type (template_type),
    INDEX idx_is_public (is_public)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS campaign_variants (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    campaign_id INT UNSIGNED NOT NULL,
    variant_name VARCHAR(255) NOT NULL,
    variant_data JSON NOT NULL,
    traffic_percentage DECIMAL(5, 2) DEFAULT 50.00,
    impressions INT DEFAULT 0,
    clicks INT DEFAULT 0,
    conversions INT DEFAULT 0,
    cost DECIMAL(10, 2) DEFAULT 0.00,
    revenue DECIMAL(10, 2) DEFAULT 0.00,
    is_winner BOOLEAN DEFAULT FALSE,
    status ENUM('active', 'paused', 'completed') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (campaign_id) REFERENCES campaigns(id) ON DELETE CASCADE,
    INDEX idx_campaign_id (campaign_id),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS email_templates (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NULL,
    template_name VARCHAR(255) NOT NULL,
    subject_line VARCHAR(255),
    html_content TEXT NOT NULL,
    text_content TEXT,
    template_category VARCHAR(100),
    thumbnail_url VARCHAR(500),
    is_public BOOLEAN DEFAULT FALSE,
    usage_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_template_category (template_category),
    INDEX idx_is_public (is_public)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS email_segments (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    segment_name VARCHAR(255) NOT NULL,
    segment_rules JSON NOT NULL,
    subscriber_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS email_segment_subscribers (
    segment_id INT UNSIGNED NOT NULL,
    subscriber_id INT UNSIGNED NOT NULL,
    added_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (segment_id, subscriber_id),
    FOREIGN KEY (segment_id) REFERENCES email_segments(id) ON DELETE CASCADE,
    FOREIGN KEY (subscriber_id) REFERENCES email_subscribers(id) ON DELETE CASCADE,
    INDEX idx_segment_id (segment_id),
    INDEX idx_subscriber_id (subscriber_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS email_tracking (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    email_campaign_id INT UNSIGNED NOT NULL,
    subscriber_id INT UNSIGNED,
    email_address VARCHAR(255) NOT NULL,
    event_type ENUM('sent', 'delivered', 'opened', 'clicked', 'bounced', 'unsubscribed', 'complained') NOT NULL,
    event_data JSON,
    ip_address VARCHAR(45),
    user_agent TEXT,
    link_url VARCHAR(500),
    occurred_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (email_campaign_id) REFERENCES email_campaigns(id) ON DELETE CASCADE,
    FOREIGN KEY (subscriber_id) REFERENCES email_subscribers(id) ON DELETE SET NULL,
    INDEX idx_email_campaign_id (email_campaign_id),
    INDEX idx_subscriber_id (subscriber_id),
    INDEX idx_event_type (event_type),
    INDEX idx_occurred_at (occurred_at),
    INDEX idx_email_address (email_address)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS scheduled_reports (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    report_name VARCHAR(255) NOT NULL,
    report_type VARCHAR(50) NOT NULL,
    report_config JSON NOT NULL,
    schedule_frequency ENUM('daily', 'weekly', 'monthly') NOT NULL,
    schedule_day INT DEFAULT NULL,
    schedule_time TIME DEFAULT '09:00:00',
    recipients JSON NOT NULL,
    last_generated_at DATETIME,
    next_run_at DATETIME,
    status ENUM('active', 'paused', 'cancelled') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_next_run_at (next_run_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS custom_reports (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    report_name VARCHAR(255) NOT NULL,
    report_type VARCHAR(50) NOT NULL,
    report_config JSON NOT NULL,
    saved_filters JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_report_type (report_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS discount_code_usage (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    discount_code_id INT UNSIGNED NOT NULL,
    user_id INT UNSIGNED NOT NULL,
    order_id VARCHAR(255),
    order_value DECIMAL(10, 2),
    discount_amount DECIMAL(10, 2) NOT NULL,
    conversion_value DECIMAL(10, 2),
    ip_address VARCHAR(45),
    user_agent TEXT,
    utm_source VARCHAR(100),
    utm_medium VARCHAR(100),
    utm_campaign VARCHAR(100),
    used_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (discount_code_id) REFERENCES discount_codes(id) ON DELETE CASCADE,
    INDEX idx_discount_code_id (discount_code_id),
    INDEX idx_user_id (user_id),
    INDEX idx_used_at (used_at DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS discount_code_templates (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NULL,
    template_name VARCHAR(255) NOT NULL,
    template_data JSON NOT NULL,
    description TEXT,
    is_public BOOLEAN DEFAULT FALSE,
    usage_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_is_public (is_public)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS api_keys (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    key_name VARCHAR(255) NOT NULL,
    api_key VARCHAR(255) NOT NULL UNIQUE,
    api_secret VARCHAR(255),
    permissions JSON,
    last_used_at DATETIME,
    expires_at DATETIME,
    status ENUM('active', 'revoked', 'expired') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_api_key (api_key),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS webhooks (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    webhook_name VARCHAR(255) NOT NULL,
    webhook_url VARCHAR(500) NOT NULL,
    events JSON NOT NULL,
    secret_token VARCHAR(255),
    status ENUM('active', 'paused', 'failed') DEFAULT 'active',
    last_triggered_at DATETIME,
    failure_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_events (events(255))
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS webhook_logs (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    webhook_id INT UNSIGNED NOT NULL,
    event_type VARCHAR(100) NOT NULL,
    payload JSON NOT NULL,
    response_status INT,
    response_body TEXT,
    error_message TEXT,
    triggered_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (webhook_id) REFERENCES webhooks(id) ON DELETE CASCADE,
    INDEX idx_webhook_id (webhook_id),
    INDEX idx_event_type (event_type),
    INDEX idx_triggered_at (triggered_at DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS file_uploads (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    file_name VARCHAR(255) NOT NULL,
    original_name VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    file_type VARCHAR(100) NOT NULL,
    file_size INT UNSIGNED NOT NULL,
    mime_type VARCHAR(100),
    upload_type ENUM('direct_mail_design', 'email_template', 'mailing_list', 'other') NOT NULL,
    related_id INT UNSIGNED,
    related_type VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_upload_type (upload_type),
    INDEX idx_related (related_type, related_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS direct_mail_tracking (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    direct_mail_campaign_id INT UNSIGNED NOT NULL,
    tracking_number VARCHAR(100) NOT NULL,
    recipient_address TEXT NOT NULL,
    status ENUM('sent', 'in_transit', 'out_for_delivery', 'delivered', 'failed') DEFAULT 'sent',
    status_updated_at DATETIME,
    delivery_date DATETIME,
    tracking_data JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (direct_mail_campaign_id) REFERENCES direct_mail_campaigns(id) ON DELETE CASCADE,
    INDEX idx_direct_mail_campaign_id (direct_mail_campaign_id),
    INDEX idx_tracking_number (tracking_number),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Admin to Client assignment mapping
CREATE TABLE IF NOT EXISTS admin_client_assignments (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    admin_id INT UNSIGNED NOT NULL,
    client_id INT UNSIGNED NOT NULL,
    assigned_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    assigned_by INT UNSIGNED,
    notes TEXT,
    FOREIGN KEY (admin_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (client_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (assigned_by) REFERENCES users(id) ON DELETE SET NULL,
    UNIQUE KEY unique_admin_client (admin_id, client_id),
    INDEX idx_admin_id (admin_id),
    INDEX idx_client_id (client_id),
    INDEX idx_assigned_at (assigned_at DESC),
    INDEX idx_admin_client_access (admin_id, client_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- B2B Affiliate & Client Acquisition System
CREATE TABLE IF NOT EXISTS affiliate_partners (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    partner_code VARCHAR(50) NOT NULL UNIQUE,
    partner_name VARCHAR(255) NOT NULL,
    contact_email VARCHAR(255) NOT NULL,
    contact_phone VARCHAR(20),
    company_name VARCHAR(255),
    website VARCHAR(255),
    commission_rate DECIMAL(5, 2) DEFAULT 10.00,
    commission_type ENUM('percentage', 'fixed', 'tiered') DEFAULT 'percentage',
    tier_level INT DEFAULT 1,
    status ENUM('active', 'inactive', 'suspended') DEFAULT 'active',
    payment_method ENUM('paypal', 'bank_transfer', 'check') DEFAULT 'paypal',
    payment_email VARCHAR(255),
    bank_account_details JSON,
    tax_id VARCHAR(50),
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    created_by INT UNSIGNED,
    INDEX idx_partner_code (partner_code),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS affiliate_clicks (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    partner_id INT UNSIGNED NOT NULL,
    partner_code VARCHAR(50) NOT NULL,
    visitor_ip VARCHAR(45),
    visitor_user_agent TEXT,
    referrer_url VARCHAR(500),
    landing_page VARCHAR(500),
    utm_source VARCHAR(100),
    utm_medium VARCHAR(100),
    utm_campaign VARCHAR(100),
    utm_term VARCHAR(100),
    utm_content VARCHAR(100),
    session_id VARCHAR(255),
    device_type ENUM('desktop', 'mobile', 'tablet', 'unknown') DEFAULT 'unknown',
    clicked_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    converted BOOLEAN DEFAULT FALSE,
    conversion_id INT UNSIGNED,
    FOREIGN KEY (partner_id) REFERENCES affiliate_partners(id) ON DELETE CASCADE,
    INDEX idx_partner_id (partner_id),
    INDEX idx_partner_code (partner_code),
    INDEX idx_clicked_at (clicked_at DESC),
    INDEX idx_converted (converted),
    INDEX idx_session_id (session_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS client_leads (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    affiliate_click_id BIGINT UNSIGNED,
    partner_id INT UNSIGNED,
    partner_code VARCHAR(50),
    lead_email VARCHAR(255) NOT NULL,
    lead_name VARCHAR(255),
    lead_phone VARCHAR(20),
    company_name VARCHAR(255),
    lead_source ENUM('affiliate', 'direct', 'organic', 'paid_ads') DEFAULT 'affiliate',
    status ENUM('new', 'contacted', 'qualified', 'converted', 'lost') DEFAULT 'new',
    notes TEXT,
    assigned_to INT UNSIGNED,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    converted_at TIMESTAMP NULL,
    converted_to_user_id INT UNSIGNED,
    FOREIGN KEY (partner_id) REFERENCES affiliate_partners(id) ON DELETE SET NULL,
    FOREIGN KEY (affiliate_click_id) REFERENCES affiliate_clicks(id) ON DELETE SET NULL,
    FOREIGN KEY (converted_to_user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_partner_id (partner_id),
    INDEX idx_status (status),
    INDEX idx_lead_email (lead_email),
    INDEX idx_created_at (created_at DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS client_orders (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    lead_id INT UNSIGNED,
    order_type ENUM('subscription', 'one_time', 'upgrade', 'renewal') DEFAULT 'subscription',
    subscription_id INT UNSIGNED,
    order_amount DECIMAL(10, 2) NOT NULL,
    currency VARCHAR(3) DEFAULT 'USD',
    order_status ENUM('pending', 'completed', 'refunded', 'cancelled') DEFAULT 'pending',
    transaction_id VARCHAR(255),
    payment_method VARCHAR(50),
    order_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (lead_id) REFERENCES client_leads(id) ON DELETE SET NULL,
    FOREIGN KEY (subscription_id) REFERENCES subscriptions(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_order_date (order_date DESC),
    INDEX idx_order_status (order_status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS affiliate_conversions (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    partner_id INT UNSIGNED NOT NULL,
    click_id BIGINT UNSIGNED,
    lead_id INT UNSIGNED,
    order_id INT UNSIGNED,
    user_id INT UNSIGNED,
    conversion_type ENUM('lead', 'trial_signup', 'subscription', 'renewal', 'upgrade') DEFAULT 'subscription',
    conversion_value DECIMAL(10, 2) NOT NULL,
    commission_rate DECIMAL(5, 2) NOT NULL,
    commission_amount DECIMAL(10, 2) NOT NULL,
    commission_status ENUM('pending', 'approved', 'paid', 'cancelled') DEFAULT 'pending',
    payout_id INT UNSIGNED,
    converted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    approved_at TIMESTAMP NULL,
    notes TEXT,
    FOREIGN KEY (partner_id) REFERENCES affiliate_partners(id) ON DELETE CASCADE,
    FOREIGN KEY (click_id) REFERENCES affiliate_clicks(id) ON DELETE SET NULL,
    FOREIGN KEY (lead_id) REFERENCES client_leads(id) ON DELETE SET NULL,
    FOREIGN KEY (order_id) REFERENCES client_orders(id) ON DELETE SET NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_partner_id (partner_id),
    INDEX idx_commission_status (commission_status),
    INDEX idx_converted_at (converted_at DESC),
    INDEX idx_order_id (order_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS affiliate_payouts (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    partner_id INT UNSIGNED NOT NULL,
    payout_period_start DATE NOT NULL,
    payout_period_end DATE NOT NULL,
    total_commission DECIMAL(10, 2) NOT NULL,
    conversion_count INT DEFAULT 0,
    payout_status ENUM('pending', 'processing', 'paid', 'failed', 'cancelled') DEFAULT 'pending',
    payment_method ENUM('paypal', 'bank_transfer', 'check') NOT NULL,
    payment_reference VARCHAR(255),
    payment_date DATE,
    payment_notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    processed_by INT UNSIGNED,
    FOREIGN KEY (partner_id) REFERENCES affiliate_partners(id) ON DELETE CASCADE,
    INDEX idx_partner_id (partner_id),
    INDEX idx_payout_status (payout_status),
    INDEX idx_payout_period (payout_period_start, payout_period_end)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS payout_conversions (
    payout_id INT UNSIGNED NOT NULL,
    conversion_id INT UNSIGNED NOT NULL,
    PRIMARY KEY (payout_id, conversion_id),
    FOREIGN KEY (payout_id) REFERENCES affiliate_payouts(id) ON DELETE CASCADE,
    FOREIGN KEY (conversion_id) REFERENCES affiliate_conversions(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- ACCELERATEDLEADZ TABLES
-- Email automation & CRM integration with IMAP from roundbox
-- Sends 1 email per minute like Mailchimp automation
-- ============================================================================

-- Email automation campaigns (like Mailchimp automation)
CREATE TABLE IF NOT EXISTS accelerated_leadz_campaigns (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    campaign_name VARCHAR(255) NOT NULL,
    campaign_type ENUM('imap_trigger', 'scheduled', 'drip') DEFAULT 'drip',
    status ENUM('active', 'paused', 'completed', 'draft') DEFAULT 'draft',
    imap_source VARCHAR(255) DEFAULT 'roundbox', -- IMAP source identifier
    trigger_condition TEXT, -- JSON: conditions that trigger the campaign
    send_rate INT DEFAULT 1, -- Emails per minute (default: 1 per minute like Mailchimp)
    start_date DATETIME,
    end_date DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_campaign_type (campaign_type),
    INDEX idx_user_status (user_id, status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Email queue for drip campaigns (1 email per minute automation)
CREATE TABLE IF NOT EXISTS accelerated_leadz_email_queue (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    campaign_id INT UNSIGNED NOT NULL,
    subscriber_id INT UNSIGNED NOT NULL, -- References email_subscribers
    email_subject VARCHAR(512),
    email_content TEXT,
    email_type ENUM('html', 'text') DEFAULT 'html',
    priority INT DEFAULT 0, -- Higher priority emails sent first
    scheduled_at DATETIME NOT NULL, -- When to send (allows 1 per minute scheduling)
    sent_at DATETIME NULL,
    status ENUM('pending', 'sending', 'sent', 'failed', 'cancelled') DEFAULT 'pending',
    error_message TEXT,
    attempts INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (campaign_id) REFERENCES accelerated_leadz_campaigns(id) ON DELETE CASCADE,
    INDEX idx_campaign_id (campaign_id),
    INDEX idx_status (status),
    INDEX idx_scheduled_at (scheduled_at),
    INDEX idx_subscriber_id (subscriber_id),
    INDEX idx_status_scheduled (status, scheduled_at),
    INDEX idx_pending_queue (status, scheduled_at, priority DESC) -- For worker queries
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- IMAP processed emails from roundbox
CREATE TABLE IF NOT EXISTS accelerated_leadz_imap_emails (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    message_id VARCHAR(255),
    from_email VARCHAR(255),
    from_name VARCHAR(255),
    subject VARCHAR(512),
    body_text TEXT,
    body_html TEXT,
    date_received DATETIME,
    processed BOOLEAN DEFAULT FALSE,
    processed_at DATETIME NULL,
    imap_source VARCHAR(255) DEFAULT 'roundbox',
    metadata JSON, -- Additional email metadata
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_user_id (user_id),
    INDEX idx_message_id (message_id),
    INDEX idx_processed (processed),
    INDEX idx_date_received (date_received),
    INDEX idx_user_processed (user_id, processed)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Email send logs for tracking
CREATE TABLE IF NOT EXISTS accelerated_leadz_send_logs (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    queue_id INT UNSIGNED NOT NULL,
    subscriber_id INT UNSIGNED NOT NULL,
    campaign_id INT UNSIGNED NOT NULL,
    email_address VARCHAR(255) NOT NULL,
    subject VARCHAR(512),
    sent_at DATETIME NOT NULL,
    opened_at DATETIME NULL,
    clicked_at DATETIME NULL,
    bounced BOOLEAN DEFAULT FALSE,
    unsubscribed BOOLEAN DEFAULT FALSE,
    error_message TEXT,
    FOREIGN KEY (queue_id) REFERENCES accelerated_leadz_email_queue(id) ON DELETE CASCADE,
    FOREIGN KEY (campaign_id) REFERENCES accelerated_leadz_campaigns(id) ON DELETE CASCADE,
    INDEX idx_queue_id (queue_id),
    INDEX idx_subscriber_id (subscriber_id),
    INDEX idx_campaign_id (campaign_id),
    INDEX idx_sent_at (sent_at),
    INDEX idx_email_address (email_address)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- ACCELERATEDADZ TABLES (Native Ad Platform)
-- Similar to Facebook Ads and Google Ads, hosted on ai-itinc.com domain
-- Public endpoints: ai-itinc.com/acceleratedadz?id=123
-- ============================================================================

-- AcceleratedAdz campaigns (similar to Facebook/Google Ads)
CREATE TABLE IF NOT EXISTS acceleratedadz_campaigns (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    campaign_name VARCHAR(255) NOT NULL,
    campaign_type ENUM('display', 'search', 'video', 'social', 'native') DEFAULT 'display',
    status ENUM('active', 'paused', 'archived', 'draft') DEFAULT 'draft',
    objective ENUM('traffic', 'conversions', 'awareness', 'engagement', 'leads') DEFAULT 'traffic',
    budget_type ENUM('daily', 'lifetime') DEFAULT 'daily',
    daily_budget DECIMAL(10, 2),
    lifetime_budget DECIMAL(10, 2),
    start_date DATETIME,
    end_date DATETIME,
    targeting JSON, -- Targeting criteria (demographics, interests, locations, etc.)
    optimization_goal VARCHAR(100) DEFAULT 'clicks', -- clicks, impressions, conversions, etc.
    bid_strategy ENUM('cpc', 'cpm', 'cpa', 'cpl') DEFAULT 'cpc',
    default_bid DECIMAL(10, 2) DEFAULT 1.00,
    domain VARCHAR(255) DEFAULT 'ai-itinc.com', -- Domain for ads
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_campaign_type (campaign_type),
    INDEX idx_user_status (user_id, status),
    INDEX idx_start_date (start_date),
    INDEX idx_end_date (end_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- AcceleratedAdz ad groups
CREATE TABLE IF NOT EXISTS acceleratedadz_ad_groups (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    campaign_id INT UNSIGNED NOT NULL,
    ad_group_name VARCHAR(255) NOT NULL,
    status ENUM('active', 'paused', 'archived') DEFAULT 'active',
    daily_budget DECIMAL(10, 2),
    bid_amount DECIMAL(10, 2),
    targeting JSON, -- Ad group specific targeting
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (campaign_id) REFERENCES acceleratedadz_campaigns(id) ON DELETE CASCADE,
    INDEX idx_campaign_id (campaign_id),
    INDEX idx_status (status),
    INDEX idx_campaign_status (campaign_id, status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- AcceleratedAdz ads (similar to Facebook/Google ad creative)
CREATE TABLE IF NOT EXISTS acceleratedadz_ads (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    ad_group_id INT UNSIGNED NOT NULL,
    campaign_id INT UNSIGNED NOT NULL, -- Denormalized for easier queries
    ad_name VARCHAR(255) NOT NULL,
    ad_type ENUM('image', 'video', 'carousel', 'text', 'html5') DEFAULT 'image',
    status ENUM('active', 'paused', 'disapproved', 'pending_review') DEFAULT 'pending_review',
    headline VARCHAR(255),
    description TEXT,
    display_url VARCHAR(255),
    destination_url TEXT, -- Landing page URL
    image_url VARCHAR(512), -- For image/video ads
    video_url VARCHAR(512), -- For video ads
    ad_data JSON, -- Full ad creative data (similar to saved_ads table)
    call_to_action VARCHAR(100), -- CTA button text
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (ad_group_id) REFERENCES acceleratedadz_ad_groups(id) ON DELETE CASCADE,
    FOREIGN KEY (campaign_id) REFERENCES acceleratedadz_campaigns(id) ON DELETE CASCADE,
    INDEX idx_ad_group_id (ad_group_id),
    INDEX idx_campaign_id (campaign_id),
    INDEX idx_status (status),
    INDEX idx_ad_type (ad_type),
    INDEX idx_campaign_status (campaign_id, status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- AcceleratedAdz performance tracking
CREATE TABLE IF NOT EXISTS acceleratedadz_performance (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    ad_id INT UNSIGNED NOT NULL,
    ad_group_id INT UNSIGNED NOT NULL,
    campaign_id INT UNSIGNED NOT NULL,
    date DATE NOT NULL,
    impressions INT UNSIGNED DEFAULT 0,
    clicks INT UNSIGNED DEFAULT 0,
    conversions INT UNSIGNED DEFAULT 0,
    spend DECIMAL(10, 2) DEFAULT 0.00,
    revenue DECIMAL(10, 2) DEFAULT 0.00,
    ctr DECIMAL(5, 4) DEFAULT 0.0000, -- Click-through rate
    cpc DECIMAL(10, 2) DEFAULT 0.00, -- Cost per click
    cpm DECIMAL(10, 2) DEFAULT 0.00, -- Cost per 1000 impressions
    cpa DECIMAL(10, 2) DEFAULT 0.00, -- Cost per acquisition
    roas DECIMAL(10, 2) DEFAULT 0.00, -- Return on ad spend
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (ad_id) REFERENCES acceleratedadz_ads(id) ON DELETE CASCADE,
    FOREIGN KEY (ad_group_id) REFERENCES acceleratedadz_ad_groups(id) ON DELETE CASCADE,
    FOREIGN KEY (campaign_id) REFERENCES acceleratedadz_campaigns(id) ON DELETE CASCADE,
    UNIQUE KEY unique_ad_date (ad_id, date),
    INDEX idx_campaign_id (campaign_id),
    INDEX idx_ad_group_id (ad_group_id),
    INDEX idx_ad_id (ad_id),
    INDEX idx_date (date),
    INDEX idx_campaign_date (campaign_id, date),
    INDEX idx_performance_metrics (campaign_id, date, impressions, clicks)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Seed data (idempotent)
INSERT INTO subscription_tiers (tier_name, tier_level, monthly_price, google_ads_budget, facebook_ads_budget, email_contacts_limit, email_campaigns_limit, discount_codes_limit, direct_mail_quantity, direct_mail_frequency, features) VALUES
('Starter', 1, 299.00, 200.00, 150.00, 1000, 4, 5, 500, 'quarterly', 'Basic marketing tools, Email support'),
('Professional', 2, 599.00, 500.00, 400.00, 5000, NULL, NULL, 1500, 'monthly', 'Advanced features, Priority support, A/B testing'),
('Enterprise', 3, 999.00, 1200.00, 800.00, NULL, NULL, NULL, 5000, 'monthly', 'All features, Dedicated support, API access, Custom integrations')
ON DUPLICATE KEY UPDATE tier_name = VALUES(tier_name);

-- Default admin (password to be set by setup process)
INSERT INTO users (email, password_hash, business_name, contact_name, role, status) VALUES
('admin@marketingplatform.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Marketing Platform Admin', 'System Administrator', 'admin', 'active')
ON DUPLICATE KEY UPDATE email = VALUES(email);


