-- ============================================================================
-- Migration 005: AcceleratedLeadz & AcceleratedAdz Tables
-- ============================================================================
-- Created for AI-IT Inc Marketing Platform
-- Adds tables for:
--   1. AcceleratedLeadz: IMAP email automation & CRM drip campaigns
--   2. AcceleratedAdz: Native ad platform similar to Facebook/Google Ads
-- ============================================================================

-- ============================================================================
-- ACCELERATEDLEADZ TABLES
-- ============================================================================

-- Email automation campaigns (like Mailchimp automation)
CREATE TABLE IF NOT EXISTS accelerated_leadz_campaigns (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    campaign_name VARCHAR(255) NOT NULL,
    campaign_type ENUM('imap_trigger', 'scheduled', 'drip') DEFAULT 'drip',
    status ENUM('active', 'paused', 'completed', 'draft') DEFAULT 'draft',
    imap_source VARCHAR(255) DEFAULT 'roundbox', -- IMAP source identifier
    trigger_condition TEXT, -- JSON: conditions that trigger the campaign
    send_rate INT DEFAULT 1, -- Emails per minute (default: 1 per minute like Mailchimp)
    start_date DATETIME,
    end_date DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_campaign_type (campaign_type),
    INDEX idx_user_status (user_id, status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Email queue for drip campaigns (1 email per minute automation)
CREATE TABLE IF NOT EXISTS accelerated_leadz_email_queue (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    campaign_id INT UNSIGNED NOT NULL,
    subscriber_id INT UNSIGNED NOT NULL, -- References email_subscribers
    email_subject VARCHAR(512),
    email_content TEXT,
    email_type ENUM('html', 'text') DEFAULT 'html',
    priority INT DEFAULT 0, -- Higher priority emails sent first
    scheduled_at DATETIME NOT NULL, -- When to send (allows 1 per minute scheduling)
    sent_at DATETIME NULL,
    status ENUM('pending', 'sending', 'sent', 'failed', 'cancelled') DEFAULT 'pending',
    error_message TEXT,
    attempts INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (campaign_id) REFERENCES accelerated_leadz_campaigns(id) ON DELETE CASCADE,
    INDEX idx_campaign_id (campaign_id),
    INDEX idx_status (status),
    INDEX idx_scheduled_at (scheduled_at),
    INDEX idx_subscriber_id (subscriber_id),
    INDEX idx_status_scheduled (status, scheduled_at),
    INDEX idx_pending_queue (status, scheduled_at, priority DESC) -- For worker queries
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- IMAP processed emails from roundbox
CREATE TABLE IF NOT EXISTS accelerated_leadz_imap_emails (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    message_id VARCHAR(255),
    from_email VARCHAR(255),
    from_name VARCHAR(255),
    subject VARCHAR(512),
    body_text TEXT,
    body_html TEXT,
    date_received DATETIME,
    processed BOOLEAN DEFAULT FALSE,
    processed_at DATETIME NULL,
    imap_source VARCHAR(255) DEFAULT 'roundbox',
    metadata JSON, -- Additional email metadata
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_user_id (user_id),
    INDEX idx_message_id (message_id),
    INDEX idx_processed (processed),
    INDEX idx_date_received (date_received),
    INDEX idx_user_processed (user_id, processed)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Email send logs for tracking
CREATE TABLE IF NOT EXISTS accelerated_leadz_send_logs (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    queue_id INT UNSIGNED NOT NULL,
    subscriber_id INT UNSIGNED NOT NULL,
    campaign_id INT UNSIGNED NOT NULL,
    email_address VARCHAR(255) NOT NULL,
    subject VARCHAR(512),
    sent_at DATETIME NOT NULL,
    opened_at DATETIME NULL,
    clicked_at DATETIME NULL,
    bounced BOOLEAN DEFAULT FALSE,
    unsubscribed BOOLEAN DEFAULT FALSE,
    error_message TEXT,
    FOREIGN KEY (queue_id) REFERENCES accelerated_leadz_email_queue(id) ON DELETE CASCADE,
    FOREIGN KEY (campaign_id) REFERENCES accelerated_leadz_campaigns(id) ON DELETE CASCADE,
    INDEX idx_queue_id (queue_id),
    INDEX idx_subscriber_id (subscriber_id),
    INDEX idx_campaign_id (campaign_id),
    INDEX idx_sent_at (sent_at),
    INDEX idx_email_address (email_address)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- ACCELERATEDADZ TABLES (Native Ad Platform)
-- ============================================================================

-- AcceleratedAdz campaigns (similar to Facebook/Google Ads)
CREATE TABLE IF NOT EXISTS acceleratedadz_campaigns (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL,
    campaign_name VARCHAR(255) NOT NULL,
    campaign_type ENUM('display', 'search', 'video', 'social', 'native') DEFAULT 'display',
    status ENUM('active', 'paused', 'archived', 'draft') DEFAULT 'draft',
    objective ENUM('traffic', 'conversions', 'awareness', 'engagement', 'leads') DEFAULT 'traffic',
    budget_type ENUM('daily', 'lifetime') DEFAULT 'daily',
    daily_budget DECIMAL(10, 2),
    lifetime_budget DECIMAL(10, 2),
    start_date DATETIME,
    end_date DATETIME,
    targeting JSON, -- Targeting criteria (demographics, interests, locations, etc.)
    optimization_goal VARCHAR(100) DEFAULT 'clicks', -- clicks, impressions, conversions, etc.
    bid_strategy ENUM('cpc', 'cpm', 'cpa', 'cpl') DEFAULT 'cpc',
    default_bid DECIMAL(10, 2) DEFAULT 1.00,
    domain VARCHAR(255) DEFAULT 'ai-itinc.com', -- Domain for ads
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_status (status),
    INDEX idx_campaign_type (campaign_type),
    INDEX idx_user_status (user_id, status),
    INDEX idx_start_date (start_date),
    INDEX idx_end_date (end_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- AcceleratedAdz ad groups
CREATE TABLE IF NOT EXISTS acceleratedadz_ad_groups (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    campaign_id INT UNSIGNED NOT NULL,
    ad_group_name VARCHAR(255) NOT NULL,
    status ENUM('active', 'paused', 'archived') DEFAULT 'active',
    daily_budget DECIMAL(10, 2),
    bid_amount DECIMAL(10, 2),
    targeting JSON, -- Ad group specific targeting
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (campaign_id) REFERENCES acceleratedadz_campaigns(id) ON DELETE CASCADE,
    INDEX idx_campaign_id (campaign_id),
    INDEX idx_status (status),
    INDEX idx_campaign_status (campaign_id, status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- AcceleratedAdz ads (similar to Facebook/Google ad creative)
CREATE TABLE IF NOT EXISTS acceleratedadz_ads (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    ad_group_id INT UNSIGNED NOT NULL,
    campaign_id INT UNSIGNED NOT NULL, -- Denormalized for easier queries
    ad_name VARCHAR(255) NOT NULL,
    ad_type ENUM('image', 'video', 'carousel', 'text', 'html5') DEFAULT 'image',
    status ENUM('active', 'paused', 'disapproved', 'pending_review') DEFAULT 'pending_review',
    headline VARCHAR(255),
    description TEXT,
    display_url VARCHAR(255),
    destination_url TEXT, -- Landing page URL
    image_url VARCHAR(512), -- For image/video ads
    video_url VARCHAR(512), -- For video ads
    ad_data JSON, -- Full ad creative data (similar to saved_ads table)
    call_to_action VARCHAR(100), -- CTA button text
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (ad_group_id) REFERENCES acceleratedadz_ad_groups(id) ON DELETE CASCADE,
    FOREIGN KEY (campaign_id) REFERENCES acceleratedadz_campaigns(id) ON DELETE CASCADE,
    INDEX idx_ad_group_id (ad_group_id),
    INDEX idx_campaign_id (campaign_id),
    INDEX idx_status (status),
    INDEX idx_ad_type (ad_type),
    INDEX idx_campaign_status (campaign_id, status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- AcceleratedAdz performance tracking
CREATE TABLE IF NOT EXISTS acceleratedadz_performance (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    ad_id INT UNSIGNED NOT NULL,
    ad_group_id INT UNSIGNED NOT NULL,
    campaign_id INT UNSIGNED NOT NULL,
    date DATE NOT NULL,
    impressions INT UNSIGNED DEFAULT 0,
    clicks INT UNSIGNED DEFAULT 0,
    conversions INT UNSIGNED DEFAULT 0,
    spend DECIMAL(10, 2) DEFAULT 0.00,
    revenue DECIMAL(10, 2) DEFAULT 0.00,
    ctr DECIMAL(5, 4) DEFAULT 0.0000, -- Click-through rate
    cpc DECIMAL(10, 2) DEFAULT 0.00, -- Cost per click
    cpm DECIMAL(10, 2) DEFAULT 0.00, -- Cost per 1000 impressions
    cpa DECIMAL(10, 2) DEFAULT 0.00, -- Cost per acquisition
    roas DECIMAL(10, 2) DEFAULT 0.00, -- Return on ad spend
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (ad_id) REFERENCES acceleratedadz_ads(id) ON DELETE CASCADE,
    FOREIGN KEY (ad_group_id) REFERENCES acceleratedadz_ad_groups(id) ON DELETE CASCADE,
    FOREIGN KEY (campaign_id) REFERENCES acceleratedadz_campaigns(id) ON DELETE CASCADE,
    UNIQUE KEY unique_ad_date (ad_id, date),
    INDEX idx_campaign_id (campaign_id),
    INDEX idx_ad_group_id (ad_group_id),
    INDEX idx_ad_id (ad_id),
    INDEX idx_date (date),
    INDEX idx_campaign_date (campaign_id, date),
    INDEX idx_performance_metrics (campaign_id, date, impressions, clicks)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

