-- Migration: B2B Affiliate & Client Acquisition System
-- Creates tables for affiliate partner management, tracking, and client acquisition

-- Affiliate Partners
CREATE TABLE IF NOT EXISTS affiliate_partners (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    partner_code VARCHAR(50) NOT NULL UNIQUE, -- Unique referral code
    partner_name VARCHAR(255) NOT NULL,
    contact_email VARCHAR(255) NOT NULL,
    contact_phone VARCHAR(20),
    company_name VARCHAR(255),
    website VARCHAR(255),
    commission_rate DECIMAL(5, 2) DEFAULT 10.00, -- Percentage commission
    commission_type ENUM('percentage', 'fixed', 'tiered') DEFAULT 'percentage',
    tier_level INT DEFAULT 1, -- For tiered commission structures
    status ENUM('active', 'inactive', 'suspended') DEFAULT 'active',
    payment_method ENUM('paypal', 'bank_transfer', 'check') DEFAULT 'paypal',
    payment_email VARCHAR(255), -- PayPal email or account details
    bank_account_details JSON, -- For bank transfers
    tax_id VARCHAR(50), -- W9/EIN for US partners
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    created_by INT UNSIGNED, -- Admin who created this partner
    INDEX idx_partner_code (partner_code),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Affiliate Clicks (Click Tracking)
CREATE TABLE IF NOT EXISTS affiliate_clicks (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    partner_id INT UNSIGNED NOT NULL,
    partner_code VARCHAR(50) NOT NULL,
    visitor_ip VARCHAR(45),
    visitor_user_agent TEXT,
    referrer_url VARCHAR(500),
    landing_page VARCHAR(500),
    utm_source VARCHAR(100),
    utm_medium VARCHAR(100),
    utm_campaign VARCHAR(100),
    utm_term VARCHAR(100),
    utm_content VARCHAR(100),
    session_id VARCHAR(255),
    device_type ENUM('desktop', 'mobile', 'tablet', 'unknown') DEFAULT 'unknown',
    clicked_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    converted BOOLEAN DEFAULT FALSE, -- Updated to TRUE when conversion happens
    conversion_id INT UNSIGNED, -- Link to affiliate_conversions
    FOREIGN KEY (partner_id) REFERENCES affiliate_partners(id) ON DELETE CASCADE,
    INDEX idx_partner_id (partner_id),
    INDEX idx_partner_code (partner_code),
    INDEX idx_clicked_at (clicked_at DESC),
    INDEX idx_converted (converted),
    INDEX idx_session_id (session_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Client Leads (Lead Capture)
CREATE TABLE IF NOT EXISTS client_leads (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    affiliate_click_id BIGINT UNSIGNED, -- Link to the click that generated this lead
    partner_id INT UNSIGNED, -- Affiliate partner who referred
    partner_code VARCHAR(50),
    lead_email VARCHAR(255) NOT NULL,
    lead_name VARCHAR(255),
    lead_phone VARCHAR(20),
    company_name VARCHAR(255),
    lead_source ENUM('affiliate', 'direct', 'organic', 'paid_ads') DEFAULT 'affiliate',
    status ENUM('new', 'contacted', 'qualified', 'converted', 'lost') DEFAULT 'new',
    notes TEXT,
    assigned_to INT UNSIGNED, -- Admin/Sales rep assigned to follow up
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    converted_at TIMESTAMP NULL, -- When lead became a client
    converted_to_user_id INT UNSIGNED, -- Link to users table when converted
    FOREIGN KEY (partner_id) REFERENCES affiliate_partners(id) ON DELETE SET NULL,
    FOREIGN KEY (affiliate_click_id) REFERENCES affiliate_clicks(id) ON DELETE SET NULL,
    FOREIGN KEY (converted_to_user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_partner_id (partner_id),
    INDEX idx_status (status),
    INDEX idx_lead_email (lead_email),
    INDEX idx_created_at (created_at DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Client Orders (Subscription/Service Orders)
CREATE TABLE IF NOT EXISTS client_orders (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id INT UNSIGNED NOT NULL, -- The client who placed the order
    lead_id INT UNSIGNED, -- Link to client_leads if converted from lead
    order_type ENUM('subscription', 'one_time', 'upgrade', 'renewal') DEFAULT 'subscription',
    subscription_id INT UNSIGNED, -- Link to subscriptions table
    order_amount DECIMAL(10, 2) NOT NULL,
    currency VARCHAR(3) DEFAULT 'USD',
    order_status ENUM('pending', 'completed', 'refunded', 'cancelled') DEFAULT 'pending',
    transaction_id VARCHAR(255),
    payment_method VARCHAR(50),
    order_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (lead_id) REFERENCES client_leads(id) ON DELETE SET NULL,
    FOREIGN KEY (subscription_id) REFERENCES subscriptions(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_order_date (order_date DESC),
    INDEX idx_order_status (order_status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Affiliate Conversions (Commission Tracking)
CREATE TABLE IF NOT EXISTS affiliate_conversions (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    partner_id INT UNSIGNED NOT NULL,
    click_id BIGINT UNSIGNED, -- Original click that led to conversion
    lead_id INT UNSIGNED, -- Lead that converted
    order_id INT UNSIGNED, -- Order that triggered the commission
    user_id INT UNSIGNED, -- Client who converted
    conversion_type ENUM('lead', 'trial_signup', 'subscription', 'renewal', 'upgrade') DEFAULT 'subscription',
    conversion_value DECIMAL(10, 2) NOT NULL, -- Order amount/value
    commission_rate DECIMAL(5, 2) NOT NULL, -- Commission rate at time of conversion
    commission_amount DECIMAL(10, 2) NOT NULL, -- Calculated commission
    commission_status ENUM('pending', 'approved', 'paid', 'cancelled') DEFAULT 'pending',
    payout_id INT UNSIGNED, -- Link to affiliate_payouts when paid
    converted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    approved_at TIMESTAMP NULL,
    notes TEXT,
    FOREIGN KEY (partner_id) REFERENCES affiliate_partners(id) ON DELETE CASCADE,
    FOREIGN KEY (click_id) REFERENCES affiliate_clicks(id) ON DELETE SET NULL,
    FOREIGN KEY (lead_id) REFERENCES client_leads(id) ON DELETE SET NULL,
    FOREIGN KEY (order_id) REFERENCES client_orders(id) ON DELETE SET NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_partner_id (partner_id),
    INDEX idx_commission_status (commission_status),
    INDEX idx_converted_at (converted_at DESC),
    INDEX idx_order_id (order_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Affiliate Payouts (Payment Tracking)
CREATE TABLE IF NOT EXISTS affiliate_payouts (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    partner_id INT UNSIGNED NOT NULL,
    payout_period_start DATE NOT NULL,
    payout_period_end DATE NOT NULL,
    total_commission DECIMAL(10, 2) NOT NULL,
    conversion_count INT DEFAULT 0,
    payout_status ENUM('pending', 'processing', 'paid', 'failed', 'cancelled') DEFAULT 'pending',
    payment_method ENUM('paypal', 'bank_transfer', 'check') NOT NULL,
    payment_reference VARCHAR(255), -- Transaction ID, check number, etc.
    payment_date DATE,
    payment_notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    processed_by INT UNSIGNED, -- Admin who processed the payout
    FOREIGN KEY (partner_id) REFERENCES affiliate_partners(id) ON DELETE CASCADE,
    INDEX idx_partner_id (partner_id),
    INDEX idx_payout_status (payout_status),
    INDEX idx_payout_period (payout_period_start, payout_period_end)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Payout Conversion Link Table (Many-to-Many)
CREATE TABLE IF NOT EXISTS payout_conversions (
    payout_id INT UNSIGNED NOT NULL,
    conversion_id INT UNSIGNED NOT NULL,
    PRIMARY KEY (payout_id, conversion_id),
    FOREIGN KEY (payout_id) REFERENCES affiliate_payouts(id) ON DELETE CASCADE,
    FOREIGN KEY (conversion_id) REFERENCES affiliate_conversions(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

