<?php
/**
 * Database Connection Test
 * Use this file to test your database connection after deployment
 * DELETE THIS FILE after successful deployment for security!
 */

// Prevent access in production if APP_ENV is set
if (defined('APP_ENV') && APP_ENV === 'production' && !isset($_GET['test'])) {
    die('Access denied. This file is disabled in production.');
}

require_once __DIR__ . '/database.php';

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html>
<head>
    <title>Database Connection Test</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 40px; }
        .success { color: green; padding: 10px; background: #d4edda; border: 1px solid #c3e6cb; border-radius: 4px; }
        .error { color: red; padding: 10px; background: #f8d7da; border: 1px solid #f5c6cb; border-radius: 4px; }
        .info { color: #004085; padding: 10px; background: #d1ecf1; border: 1px solid #bee5eb; border-radius: 4px; }
        pre { background: #f4f4f4; padding: 15px; border-radius: 4px; overflow-x: auto; }
    </style>
</head>
<body>
    <h1>Database Connection Test</h1>
    
    <?php
    $errors = [];
    $success = [];
    
    // Test 1: Check if constants are defined
    echo "<h2>1. Configuration Check</h2>";
    if (defined('DB_HOST')) {
        echo "<div class='success'>✓ DB_HOST is defined: " . DB_HOST . "</div>";
    } else {
        echo "<div class='error'>✗ DB_HOST is not defined</div>";
        $errors[] = 'DB_HOST';
    }
    
    if (defined('DB_NAME')) {
        echo "<div class='success'>✓ DB_NAME is defined: " . DB_NAME . "</div>";
    } else {
        echo "<div class='error'>✗ DB_NAME is not defined</div>";
        $errors[] = 'DB_NAME';
    }
    
    if (defined('DB_USER')) {
        echo "<div class='success'>✓ DB_USER is defined: " . DB_USER . "</div>";
    } else {
        echo "<div class='error'>✗ DB_USER is not defined</div>";
        $errors[] = 'DB_USER';
    }
    
    if (defined('DB_PASS')) {
        echo "<div class='success'>✓ DB_PASS is defined (hidden for security)</div>";
    } else {
        echo "<div class='error'>✗ DB_PASS is not defined</div>";
        $errors[] = 'DB_PASS';
    }
    
    // Test 2: Try database connection
    echo "<h2>2. Database Connection</h2>";
    try {
        require_once __DIR__ . '/../includes/database.php';
        $db = getDB();
        echo "<div class='success'>✓ Database connection successful!</div>";
        $success[] = 'Connection';
        
        // Test 3: Query test
        echo "<h2>3. Database Query Test</h2>";
        $stmt = $db->query("SELECT COUNT(*) as count FROM users");
        $result = $stmt->fetch();
        echo "<div class='success'>✓ Query executed successfully</div>";
        echo "<div class='info'>Found " . $result['count'] . " user(s) in database</div>";
        $success[] = 'Query';
        
        // Test 4: Check critical tables
        echo "<h2>4. Table Existence Check</h2>";
        $criticalTables = ['users', 'campaigns', 'acceleratedadz_campaigns', 'client_leads', 'email_subscribers'];
        $stmt = $db->query("SHOW TABLES");
        $existingTables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        foreach ($criticalTables as $table) {
            if (in_array($table, $existingTables)) {
                echo "<div class='success'>✓ Table '$table' exists</div>";
            } else {
                echo "<div class='error'>✗ Table '$table' is missing</div>";
                $errors[] = "Table: $table";
            }
        }
        
        // Test 5: PHP Extensions
        echo "<h2>5. PHP Extensions</h2>";
        $requiredExtensions = ['pdo', 'pdo_mysql', 'mbstring', 'openssl', 'json', 'curl'];
        foreach ($requiredExtensions as $ext) {
            if (extension_loaded($ext)) {
                echo "<div class='success'>✓ Extension '$ext' is loaded</div>";
            } else {
                echo "<div class='error'>✗ Extension '$ext' is missing</div>";
                $errors[] = "Extension: $ext";
            }
        }
        
        // Test 6: PHP Version
        echo "<h2>6. PHP Version</h2>";
        $phpVersion = PHP_VERSION;
        if (version_compare($phpVersion, '7.4.0', '>=')) {
            echo "<div class='success'>✓ PHP Version: $phpVersion (Compatible)</div>";
        } else {
            echo "<div class='error'>✗ PHP Version: $phpVersion (Requires PHP 7.4+)</div>";
            $errors[] = "PHP Version";
        }
        
        // Test 7: File Permissions
        echo "<h2>7. Directory Permissions</h2>";
        $directories = [
            __DIR__ . '/../uploads' => 'Uploads directory',
            __DIR__ . '/../tmp' => 'Temp directory',
            __DIR__ . '/../logs' => 'Logs directory'
        ];
        
        foreach ($directories as $dir => $name) {
            if (!is_dir($dir)) {
                if (@mkdir($dir, 0755, true)) {
                    echo "<div class='info'>ℹ Created directory: $name</div>";
                } else {
                    echo "<div class='error'>✗ Cannot create directory: $name</div>";
                    $errors[] = "Directory: $name";
                }
            }
            
            if (is_dir($dir) && is_writable($dir)) {
                echo "<div class='success'>✓ $name is writable</div>";
            } else {
                echo "<div class='error'>✗ $name is not writable (chmod 755 required)</div>";
                $errors[] = "Permission: $name";
            }
        }
        
    } catch (PDOException $e) {
        echo "<div class='error'>✗ Database connection failed!</div>";
        echo "<div class='error'>Error: " . htmlspecialchars($e->getMessage()) . "</div>";
        echo "<pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre>";
        $errors[] = 'Connection Error';
    }
    
    // Summary
    echo "<h2>Summary</h2>";
    if (empty($errors)) {
        echo "<div class='success'><strong>✓ All tests passed! Your database is configured correctly.</strong></div>";
        echo "<div class='info'><strong>⚠️ IMPORTANT:</strong> Delete this file (test_connection.php) for security!</div>";
    } else {
        echo "<div class='error'><strong>✗ Some tests failed. Please fix the following issues:</strong></div>";
        echo "<ul>";
        foreach ($errors as $error) {
            echo "<li>" . htmlspecialchars($error) . "</li>";
        }
        echo "</ul>";
    }
    ?>
    
    <hr>
    <p><a href="../login.php">→ Go to Login Page</a></p>
    <p><small>Delete this file after testing for security.</small></p>
</body>
</html>

