﻿<?php
/**
 * Marketing Platform Setup Script
 * 
 * This script helps you set up the Marketing Platform by:
 * - Checking system requirements
 * - Testing database connection
 * - Creating database if needed
 * - Importing schema
 * - Creating default admin user
 * - Verifying installation
 * 
 * Usage: Access via browser http://localhost:8000/setup.php
 *        Or run: php setup.php
 */

// Prevent direct access in production
if (file_exists(__DIR__ . '/.setup_complete')) {
    die('Setup already completed. Delete .setup_complete file to run setup again.');
}

error_reporting(E_ALL);
ini_set('display_errors', '1');

$setupComplete = false;
$errors = [];
$warnings = [];
$info = [];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['run_setup'])) {
    try {
        $dbHost = $_POST['db_host'] ?? 'localhost';
        $dbName = $_POST['db_name'] ?? 'mindsybq_market';
        $dbUser = $_POST['db_user'] ?? 'mindsybq_user';
        $dbPass = $_POST['db_pass'] ?? 'M@rket-2025';
        $adminEmail = $_POST['admin_email'] ?? 'admin@marketingplatform.com';
        $adminPassword = $_POST['admin_password'] ?? 'admin123';
        
        // Update config file
        $configContent = "<?php\n/**\n * Database Configuration\n * Auto-generated by setup script\n */\n\n";
        $configContent .= "define('DB_HOST', '" . addslashes($dbHost) . "');\n";
        $configContent .= "define('DB_NAME', '" . addslashes($dbName) . "');\n";
        $configContent .= "define('DB_USER', '" . addslashes($dbUser) . "');\n";
        $configContent .= "define('DB_PASS', '" . addslashes($dbPass) . "');\n";
        $configContent .= "define('DB_CHARSET', 'utf8mb4');\n";
        
        file_put_contents(__DIR__ . '/config/database.php', $configContent);
        
        // Test database connection without database
        try {
            $pdo = new PDO("mysql:host=$dbHost;charset=utf8mb4", $dbUser, $dbPass);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $info[] = "✓ Database server connection successful";
        } catch (PDOException $e) {
            throw new Exception("Database connection failed: " . $e->getMessage());
        }
        
        // Check if database exists, create if not
        try {
            $pdo->exec("CREATE DATABASE IF NOT EXISTS `$dbName` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
            $info[] = "✓ Database '$dbName' checked/created";
        } catch (PDOException $e) {
            throw new Exception("Failed to create database: " . $e->getMessage());
        }
        
        // Connect to the database
        $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8mb4", $dbUser, $dbPass);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // Check if tables exist
        $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
        
        if (empty($tables)) {
            // Import unified schema_main.sql
            $schemaFile = __DIR__ . '/../database/schema_main.sql';
            
            if (!file_exists($schemaFile)) {
                throw new Exception("Schema file not found. Please ensure database/schema_main.sql exists.");
            }
            
            $schema = file_get_contents($schemaFile);
            // Remove CREATE DATABASE and USE statements
            $schema = preg_replace('/CREATE DATABASE[^;]+;/i', '', $schema);
            $schema = preg_replace('/USE[^;]+;/i', '', $schema);
            $info[] = "✓ Found unified schema file: schema_main.sql";
            
            // Execute schema
            $pdo->beginTransaction();
            try {
                // Remove comments and phpMyAdmin specific statements
                $schema = preg_replace('/--.*$/m', '', $schema);
                $schema = preg_replace('/\/\*.*?\*\//s', '', $schema);
                $schema = preg_replace('/SET SQL_MODE[^;]*;/i', '', $schema);
                $schema = preg_replace('/START TRANSACTION[^;]*;/i', '', $schema);
                $schema = preg_replace('/COMMIT[^;]*;/i', '', $schema);
                $schema = preg_replace('/SET time_zone[^;]*;/i', '', $schema);
                $schema = preg_replace('/SET @OLD_[^;]*;/i', '', $schema);
                $schema = preg_replace('/\/\*![^}]*\*\//', '', $schema);
                
                // Split by semicolons and execute
                $statements = array_filter(explode(';', $schema), function($stmt) {
                    $stmt = trim($stmt);
                    return !empty($stmt) && strlen($stmt) > 10;
                });
                
                foreach ($statements as $statement) {
                    $statement = trim($statement);
                    if (!empty($statement)) {
                        $pdo->exec($statement);
                    }
                }
                
                $pdo->commit();
                $info[] = "✓ Database schema imported successfully";
            } catch (PDOException $e) {
                $pdo->rollBack();
                throw new Exception("Failed to import schema: " . $e->getMessage());
            }
        } else {
            $info[] = "✓ Database tables already exist (" . count($tables) . " tables)";
        }
        
        // Check/update admin user
        $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$adminEmail]);
        $adminExists = $stmt->fetch();
        
        if ($adminExists) {
            // Update existing admin password
            $passwordHash = password_hash($adminPassword, PASSWORD_DEFAULT);
            $stmt = $pdo->prepare("UPDATE users SET password_hash = ?, role = 'admin', status = 'active' WHERE email = ?");
            $stmt->execute([$passwordHash, $adminEmail]);
            $info[] = "✓ Admin user password updated";
        } else {
            // Create admin user
            $passwordHash = password_hash($adminPassword, PASSWORD_DEFAULT);
            $stmt = $pdo->prepare("INSERT INTO users (email, password_hash, business_name, contact_name, role, status) VALUES (?, ?, ?, ?, 'admin', 'active')");
            $stmt->execute([$adminEmail, $passwordHash, 'Marketing Platform Admin', 'System Administrator']);
            $info[] = "✓ Admin user created";
        }
        
        // Verify subscription tiers exist
        $stmt = $pdo->query("SELECT COUNT(*) FROM subscription_tiers");
        if ($stmt->fetchColumn() == 0) {
            $pdo->exec("INSERT INTO subscription_tiers 
                        (tier_name, tier_level, monthly_price, google_ads_budget, facebook_ads_budget, 
                         email_contacts_limit, email_campaigns_limit, discount_codes_limit, 
                         direct_mail_quantity, direct_mail_frequency, features) VALUES
                        ('Starter', 1, 299.00, 200.00, 150.00, 1000, 4, 5, 500, 'quarterly', 'Basic discount codes, Monthly metrics report'),
                        ('Professional', 2, 599.00, 500.00, 400.00, 5000, NULL, NULL, 1500, 'monthly', 'Advanced discount management, Real-time dashboard, A/B testing, Campaign optimization'),
                        ('Enterprise', 3, 999.00, 1200.00, 800.00, NULL, NULL, NULL, 5000, 'monthly', 'Premium discount system, Full API access, Dedicated account manager, Advanced analytics, Multi-channel automation, Custom integrations')");
            $info[] = "✓ Subscription tiers created";
        }
        
        // Mark setup as complete
        file_put_contents(__DIR__ . '/.setup_complete', date('Y-m-d H:i:s'));
        $setupComplete = true;
        $info[] = "✓ Setup completed successfully!";
        
    } catch (Exception $e) {
        $errors[] = $e->getMessage();
    }
}

// Check PHP version
$phpVersion = PHP_VERSION;
$phpVersionOk = version_compare($phpVersion, '8.0.0', '>=');
if (!$phpVersionOk) {
    $errors[] = "PHP 8.0+ required. Current version: $phpVersion";
}

// Check required extensions
$requiredExtensions = ['pdo', 'pdo_mysql', 'mbstring', 'json', 'curl'];
$missingExtensions = [];
foreach ($requiredExtensions as $ext) {
    if (!extension_loaded($ext)) {
        $missingExtensions[] = $ext;
    }
}
if (!empty($missingExtensions)) {
    $warnings[] = "Missing PHP extensions: " . implode(', ', $missingExtensions) . ". Some features may not work.";
}

// Check file permissions
$writableDirs = ['config', 'assets'];
foreach ($writableDirs as $dir) {
    if (file_exists($dir) && !is_writable($dir)) {
        $warnings[] = "Directory '$dir' is not writable. Some features may not work.";
    }
}

// Check if config exists
$configExists = file_exists(__DIR__ . '/config/database.php');
if ($configExists) {
    require_once __DIR__ . '/config/database.php';
    $info[] = "✓ Configuration file exists";
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Marketing Platform - Setup</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { background: #f8f9fa; }
        .setup-container { max-width: 800px; margin: 50px auto; }
        .status-card { margin-bottom: 20px; }
        .status-item { padding: 10px; margin: 5px 0; border-radius: 5px; }
        .status-success { background: #d4edda; color: #155724; }
        .status-error { background: #f8d7da; color: #721c24; }
        .status-warning { background: #fff3cd; color: #856404; }
        .status-info { background: #d1ecf1; color: #0c5460; }
    </style>
</head>
<body>
    <div class="container setup-container">
        <div class="card shadow">
            <div class="card-header bg-primary text-white">
                <h2 class="mb-0">Marketing Platform Setup</h2>
            </div>
            <div class="card-body">
                <?php if ($setupComplete): ?>
                    <div class="alert alert-success">
                        <h4>✓ Setup Completed Successfully!</h4>
                        <p>Your Marketing Platform is now ready to use.</p>
                        <hr>
                        <p><strong>Admin Login:</strong></p>
                        <ul>
                            <li>Email: <?= htmlspecialchars($adminEmail ?? 'admin@marketingplatform.com') ?></li>
                            <li>Password: <?= htmlspecialchars($adminPassword ?? 'admin123') ?></li>
                        </ul>
                        <a href="login.php" class="btn btn-success btn-lg">Go to Login Page</a>
                    </div>
                <?php else: ?>
                    <!-- System Requirements Check -->
                    <h4>System Requirements</h4>
                    <div class="status-card">
                        <div class="status-item <?= $phpVersionOk ? 'status-success' : 'status-error' ?>">
                            <?= $phpVersionOk ? '✓' : '✗' ?> PHP Version: <?= $phpVersion ?> <?= $phpVersionOk ? '' : '(8.0+ required)' ?>
                        </div>
                        <?php foreach ($requiredExtensions as $ext): ?>
                            <div class="status-item <?= extension_loaded($ext) ? 'status-success' : 'status-warning' ?>">
                                <?= extension_loaded($ext) ? '✓' : '⚠' ?> PHP Extension: <?= $ext ?>
                            </div>
                        <?php endforeach; ?>
                    </div>

                    <!-- Status Messages -->
                    <?php foreach ($info as $msg): ?>
                        <div class="alert alert-info"><?= htmlspecialchars($msg) ?></div>
                    <?php endforeach; ?>
                    <?php foreach ($warnings as $msg): ?>
                        <div class="alert alert-warning"><?= htmlspecialchars($msg) ?></div>
                    <?php endforeach; ?>
                    <?php foreach ($errors as $msg): ?>
                        <div class="alert alert-danger"><?= htmlspecialchars($msg) ?></div>
                    <?php endforeach; ?>

                    <!-- Setup Form -->
                    <form method="POST" action="">
                        <h4 class="mt-4">Database Configuration</h4>
                        <div class="mb-3">
                            <label for="db_host" class="form-label">Database Host</label>
                            <input type="text" class="form-control" id="db_host" name="db_host" value="<?= htmlspecialchars(defined('DB_HOST') ? DB_HOST : 'localhost') ?>" required>
                        </div>
                        <div class="mb-3">
                            <label for="db_name" class="form-label">Database Name</label>
                            <input type="text" class="form-control" id="db_name" name="db_name" value="<?= htmlspecialchars(defined('DB_NAME') ? DB_NAME : 'mindsybq_market') ?>" required>
                        </div>
                        <div class="mb-3">
                            <label for="db_user" class="form-label">Database User</label>
                            <input type="text" class="form-control" id="db_user" name="db_user" value="<?= htmlspecialchars(defined('DB_USER') ? DB_USER : 'root') ?>" required>
                        </div>
                        <div class="mb-3">
                            <label for="db_pass" class="form-label">Database Password</label>
                            <input type="password" class="form-control" id="db_pass" name="db_pass" value="<?= htmlspecialchars(defined('DB_PASS') ? DB_PASS : '') ?>">
                        </div>

                        <h4 class="mt-4">Admin Account</h4>
                        <div class="mb-3">
                            <label for="admin_email" class="form-label">Admin Email</label>
                            <input type="email" class="form-control" id="admin_email" name="admin_email" value="admin@marketingplatform.com" required>
                        </div>
                        <div class="mb-3">
                            <label for="admin_password" class="form-label">Admin Password</label>
                            <input type="password" class="form-control" id="admin_password" name="admin_password" value="admin123" required>
                            <small class="form-text text-muted">Change this password after first login</small>
                        </div>

                        <button type="submit" name="run_setup" class="btn btn-primary btn-lg w-100">Run Setup</button>
                    </form>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

