<?php
/**
 * Unified Email Sender
 * Combines Ad Builder and Email Campaigns in one page
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireClient();
$pageTitle = 'Email Sender - Create Ads & Send Emails';

$userId = $_SESSION['user_id'];
$db = getDB();

// Get campaigns for adding ads
$stmt = $db->prepare("
    SELECT id, campaign_name, campaign_type 
    FROM campaigns 
    WHERE user_id = ? AND campaign_type IN ('google_ads', 'facebook_ads', 'multi_channel', 'email')
    ORDER BY campaign_name
");
$stmt->execute([$userId]);
$campaigns = $stmt->fetchAll();

// Get existing email campaigns
$stmt = $db->prepare("
    SELECT ec.*, c.campaign_name as parent_campaign_name
    FROM email_campaigns ec
    LEFT JOIN campaigns c ON ec.campaign_id = c.id
    WHERE ec.user_id = ?
    ORDER BY ec.created_at DESC
    LIMIT 10
");
$stmt->execute([$userId]);
$emailCampaigns = $stmt->fetchAll();

// Get subscribers count
$stmt = $db->prepare("SELECT COUNT(*) FROM email_subscribers WHERE user_id = ? AND status = 'subscribed'");
$stmt->execute([$userId]);
$subscriberCount = $stmt->fetchColumn();

// Get campaigns with ads for insertion
$stmt = $db->prepare("
    SELECT DISTINCT c.id, c.campaign_name
    FROM campaigns c
    INNER JOIN campaign_ads ca ON c.id = ca.campaign_id
    WHERE c.user_id = ? AND ca.status = 'active'
    ORDER BY c.campaign_name
");
$stmt->execute([$userId]);
$campaignsWithAds = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/../includes/csrf.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h1><i class="bi bi-envelope-paper-heart"></i> Email Sender</h1>
                <p class="text-muted mb-0">Create ads and send email campaigns all in one place</p>
            </div>
            <div class="text-end">
                <div class="badge bg-info">CRM: <?= number_format($subscriberCount) ?> customers</div>
            </div>
        </div>
    </div>
</div>

<!-- Tabs: Ad Builder | Email Campaign -->
<ul class="nav nav-tabs mb-4" id="unifiedTabs" role="tablist">
    <li class="nav-item" role="presentation">
        <button class="nav-link active" id="ad-builder-tab" data-bs-toggle="tab" data-bs-target="#ad-builder" type="button">
            <i class="bi bi-palette"></i> Create Ad
        </button>
    </li>
    <li class="nav-item" role="presentation">
        <button class="nav-link" id="email-compose-tab" data-bs-toggle="tab" data-bs-target="#email-compose" type="button">
            <i class="bi bi-envelope"></i> Compose Email
        </button>
    </li>
    <li class="nav-item" role="presentation">
        <button class="nav-link" id="email-campaigns-tab" data-bs-toggle="tab" data-bs-target="#email-campaigns" type="button">
            <i class="bi bi-list-ul"></i> My Email Campaigns
        </button>
    </li>
</ul>

<div class="tab-content" id="unifiedTabsContent">
    
    <!-- Ad Builder Tab -->
    <div class="tab-pane fade show active" id="ad-builder" role="tabpanel">
        <div class="row">
            <!-- Template Selection & Settings -->
            <div class="col-md-3">
                <div class="card mb-3">
                    <div class="card-header">
                        <h6 class="mb-0">Templates</h6>
                    </div>
                    <div class="card-body">
                        <div class="list-group list-group-flush">
                            <button class="list-group-item list-group-item-action template-btn active" data-template="custom">
                                <i class="bi bi-pencil"></i> Custom (Blank)
                            </button>
                            <button class="list-group-item list-group-item-action template-btn" data-template="retail">
                                <i class="bi bi-shop"></i> Retail
                            </button>
                            <button class="list-group-item list-group-item-action template-btn" data-template="food">
                                <i class="bi bi-cup-hot"></i> Food
                            </button>
                            <button class="list-group-item list-group-item-action template-btn" data-template="beauty">
                                <i class="bi bi-star"></i> Beauty
                            </button>
                            <button class="list-group-item list-group-item-action template-btn" data-template="fashion">
                                <i class="bi bi-bag"></i> Fashion
                            </button>
                            <button class="list-group-item list-group-item-action template-btn" data-template="service">
                                <i class="bi bi-tools"></i> Service
                            </button>
                        </div>
                    </div>
                </div>

                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0">Ad Settings</h6>
                    </div>
                    <div class="card-body">
                        <div class="mb-3">
                            <label class="form-label">Ad Name</label>
                            <input type="text" class="form-control" id="unified_adName" value="My Ad">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Platform</label>
                            <select class="form-select" id="unified_adPlatform">
                                <option value="google_ads">Google Ads</option>
                                <option value="facebook_ads">Facebook Ads</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Add to Campaign</label>
                            <select class="form-select" id="unified_campaignSelect">
                                <option value="">Select campaign...</option>
                                <?php foreach ($campaigns as $camp): ?>
                                    <option value="<?= $camp['id'] ?>"><?= escape($camp['campaign_name']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="d-grid gap-2">
                            <button class="btn btn-success" id="unified_saveAdBtn">
                                <i class="bi bi-save"></i> Save Ad
                            </button>
                            <button class="btn btn-primary" id="unified_addToCampaignBtn">
                                <i class="bi bi-plus-circle"></i> Add to Campaign
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Quick Send Email Section -->
                <div class="card mt-3">
                    <div class="card-header bg-warning text-dark">
                        <h6 class="mb-0"><i class="bi bi-lightning-charge"></i> Quick Send Email</h6>
                    </div>
                    <div class="card-body">
                        <p class="text-muted small mb-3">Send this ad directly in an email without saving it first!</p>
                        <div class="d-grid gap-2">
                            <button class="btn btn-warning btn-lg" id="unified_quickSendBtn">
                                <i class="bi bi-send-fill"></i> Quick Send Email
                            </button>
                        </div>
                        <small class="text-muted d-block mt-2">The current ad will be embedded in the email automatically</small>
                    </div>
                </div>
            </div>

            <!-- Ad Preview Canvas -->
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Ad Preview</h5>
                    </div>
                    <div class="card-body">
                        <div id="unified_adCanvas" style="
                            width: 100%;
                            max-width: 600px;
                            margin: 0 auto;
                            background: #fff;
                            border: 2px solid #ddd;
                            border-radius: 8px;
                            overflow: hidden;
                            position: relative;
                            aspect-ratio: 16/9;
                            min-height: 350px;
                        ">
                            <!-- Ad content rendered here -->
                        </div>
                    </div>
                </div>
            </div>

            <!-- Editor Panel -->
            <div class="col-md-3">
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0">Edit Ad Content</h6>
                    </div>
                    <div class="card-body">
                        <div class="mb-3">
                            <label class="form-label">Headline</label>
                            <input type="text" class="form-control" id="unified_headline" placeholder="Enter headline">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" id="unified_description" rows="3" placeholder="Enter description"></textarea>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Call-to-Action</label>
                            <input type="text" class="form-control" id="unified_cta" placeholder="Shop Now, Learn More, etc.">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Destination URL</label>
                            <input type="url" class="form-control" id="unified_destinationUrl" placeholder="https://your-website.com/product">
                            <small class="text-muted">Where users go when they click</small>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">CTA Button Color</label>
                            <input type="color" class="form-control form-control-color" id="unified_ctaColor" value="#007bff">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Background Color</label>
                            <input type="color" class="form-control form-control-color" id="unified_bgColor" value="#ffffff">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Text Color</label>
                            <input type="color" class="form-control form-control-color" id="unified_textColor" value="#000000">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Product Image URL</label>
                            <input type="url" class="form-control" id="unified_productImage" placeholder="https://...">
                            <small class="text-muted">Or upload: <input type="file" id="unified_productImageFile" accept="image/*" class="form-control form-control-sm mt-1"></small>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Background Image URL</label>
                            <input type="url" class="form-control" id="unified_bgImage" placeholder="https://...">
                            <small class="text-muted">Or upload: <input type="file" id="unified_bgImageFile" accept="image/*" class="form-control form-control-sm mt-1"></small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Email Compose Tab -->
    <div class="tab-pane fade" id="email-compose" role="tabpanel">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Create & Send Email Campaign</h5>
            </div>
            <div class="card-body">
                <form method="POST" action="<?= url('api/email_service.php') ?>" id="unifiedEmailForm">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="create_campaign">
                    
                    <div class="row">
                        <div class="col-md-8">
                            <div class="mb-3">
                                <label for="unified_campaign_name" class="form-label">Campaign Name</label>
                                <input type="text" class="form-control" id="unified_campaign_name" name="campaign_name" required>
                            </div>
                            <div class="mb-3">
                                <label for="unified_subject_line" class="form-label">Subject Line</label>
                                <input type="text" class="form-control" id="unified_subject_line" name="subject_line" required>
                            </div>
                            
                            <div class="mb-3">
                                <label for="unified_email_content" class="form-label">Email Content (HTML)</label>
                                <div class="btn-toolbar mb-2" role="toolbar">
                                    <div class="btn-group btn-group-sm" role="group">
                                        <button type="button" class="btn btn-outline-secondary" id="unified_insertAdBtn" title="Insert Ad from Campaign">
                                            <i class="bi bi-image"></i> Insert Ad
                                        </button>
                                        <button type="button" class="btn btn-outline-secondary" id="unified_previewEmailBtn" title="Preview Email">
                                            <i class="bi bi-eye"></i> Preview
                                        </button>
                                        <button type="button" class="btn btn-outline-secondary" id="unified_testEmailBtn" title="Send Test Email">
                                            <i class="bi bi-envelope-check"></i> Test Email
                                        </button>
                                    </div>
                                </div>
                                <textarea class="form-control font-monospace" id="unified_email_content" name="email_content" rows="15" required></textarea>
                                <div class="form-text">Use the toolbar to insert ads, preview, or send a test email</div>
                            </div>
                        </div>
                        
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label">Link to Campaign (Optional)</label>
                                <select class="form-select" id="unified_parent_campaign_id" name="campaign_id">
                                    <option value="">None - Standalone</option>
                                    <?php foreach ($campaigns as $camp): ?>
                                        <option value="<?= $camp['id'] ?>"><?= escape($camp['campaign_name']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Recipient Source</label>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="recipient_source" id="unified_recipient_manual" value="manual" checked>
                                    <label class="form-check-label" for="unified_recipient_manual">
                                        Manual Entry
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="recipient_source" id="unified_recipient_crm" value="crm">
                                    <label class="form-check-label" for="unified_recipient_crm">
                                        CRM Customers (<?= number_format($subscriberCount) ?>)
                                    </label>
                                </div>
                            </div>
                            
                            <div class="mb-3" id="unified_manual_recipients_div">
                                <label for="unified_recipients" class="form-label">Recipients (comma-separated)</label>
                                <textarea class="form-control" id="unified_recipients" name="recipients" rows="3" placeholder="email1@example.com, email2@example.com"></textarea>
                            </div>
                            
                            <div class="mb-3" id="unified_crm_recipients_div" style="display: none;">
                                <label for="unified_daily_limit" class="form-label">Daily Email Limit</label>
                                <input type="number" class="form-control" id="unified_daily_limit" name="daily_limit" min="1" placeholder="e.g., 100">
                                <small class="text-muted">Sends 1 email per minute, up to this limit per day</small>
                            </div>
                            
                            <div class="d-grid gap-2">
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-send"></i> Create & Queue Campaign
                                </button>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Email Campaigns List Tab -->
    <div class="tab-pane fade" id="email-campaigns" role="tabpanel">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">My Email Campaigns</h5>
            </div>
            <div class="card-body">
                <?php if (empty($emailCampaigns)): ?>
                    <p class="text-muted text-center py-5">No email campaigns yet. Create one in the "Compose Email" tab!</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Campaign Name</th>
                                    <th>Subject</th>
                                    <th>Recipients</th>
                                    <th>Sent</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($emailCampaigns as $campaign): ?>
                                    <tr>
                                        <td><strong><?= escape($campaign['campaign_name']) ?></strong></td>
                                        <td><?= escape($campaign['subject_line']) ?></td>
                                        <td><?= number_format($campaign['recipient_count']) ?></td>
                                        <td><?= number_format($campaign['sent_count']) ?></td>
                                        <td>
                                            <span class="badge bg-<?= 
                                                $campaign['status'] === 'sent' ? 'success' : 
                                                ($campaign['status'] === 'scheduled' ? 'info' : 
                                                ($campaign['status'] === 'sending' ? 'warning' : 'secondary')) 
                                            ?>">
                                                <?= escape($campaign['status']) ?>
                                            </span>
                                        </td>
                                        <td>
                                            <div class="btn-group" role="group">
                                                <button class="btn btn-sm btn-info view-email-campaign-btn" data-id="<?= $campaign['id'] ?>" title="View Campaign">
                                                    <i class="bi bi-eye"></i> View
                                                </button>
                                                <button class="btn btn-sm btn-success send-ad-individually-btn" data-campaign-id="<?= $campaign['id'] ?>" title="Send Ad Individually">
                                                    <i class="bi bi-envelope-paper"></i> Send Ad
                                                </button>
                                                <button class="btn btn-sm btn-danger delete-campaign-btn" data-id="<?= $campaign['id'] ?>" title="Delete Campaign">
                                                    <i class="bi bi-trash"></i> Delete
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

</div>

<!-- Ad Templates (same as ad_builder.php) -->
<script src="https://cdn.jsdelivr.net/npm/html2canvas@1.4.1/dist/html2canvas.min.js"></script>
<script>
const adTemplates = {
    retail: {
        headline: "Shop the Latest Deals!",
        description: "Discover amazing products at unbeatable prices. Limited time offer!",
        cta: "Shop Now",
        ctaColor: "#ff6b6b",
        bgColor: "#ffffff",
        textColor: "#2c3e50",
        bgImage: "",
        style: "modern"
    },
    food: {
        headline: "Fresh & Delicious",
        description: "Experience mouth-watering flavors that will tantalize your taste buds.",
        cta: "Order Now",
        ctaColor: "#ff9800",
        bgColor: "#fff5e6",
        textColor: "#8b4513",
        bgImage: "",
        style: "warm"
    },
    beauty: {
        headline: "Feel Beautiful Inside & Out",
        description: "Transform your beauty routine with our premium collection.",
        cta: "Explore Products",
        ctaColor: "#e91e63",
        bgColor: "#fce4ec",
        textColor: "#880e4f",
        bgImage: "",
        style: "elegant"
    },
    fashion: {
        headline: "Style That Speaks",
        description: "Trendsetting fashion pieces for the modern you.",
        cta: "Shop Collection",
        ctaColor: "#000000",
        bgColor: "#f5f5f5",
        textColor: "#212121",
        bgImage: "",
        style: "minimalist"
    },
    service: {
        headline: "Expert Service You Can Trust",
        description: "Professional solutions tailored to your needs.",
        cta: "Get Started",
        ctaColor: "#2196f3",
        bgColor: "#ffffff",
        textColor: "#1565c0",
        bgImage: "",
        style: "professional"
    },
    custom: {
        headline: "",
        description: "",
        cta: "",
        ctaColor: "#007bff",
        bgColor: "#ffffff",
        textColor: "#000000",
        bgImage: "",
        style: "custom"
    }
};

let currentAd = {
    template: 'custom',
    headline: '',
    description: '',
    cta: '',
    destinationUrl: '',
    ctaColor: '#007bff',
    bgColor: '#ffffff',
    textColor: '#000000',
    bgImage: '',
    productImage: '',
    name: 'My Ad',
    platform: 'google_ads',
    size: '1200x628'
};

document.addEventListener('DOMContentLoaded', function() {
    // Template selection
    document.querySelectorAll('.template-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            document.querySelectorAll('.template-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            const template = this.dataset.template;
            loadTemplate(template);
        });
    });
    
    // Editor inputs
    ['unified_headline', 'unified_description', 'unified_cta', 'unified_ctaColor', 'unified_bgColor', 
     'unified_textColor', 'unified_bgImage', 'unified_productImage', 'unified_destinationUrl'].forEach(id => {
        const element = document.getElementById(id);
        if (element) {
            element.addEventListener('input', updateAd);
            element.addEventListener('change', updateAd);
        }
    });
    
    // File uploads
    document.getElementById('unified_productImageFile')?.addEventListener('change', function() {
        uploadFile(this.files[0], 'unified_productImage');
    });
    document.getElementById('unified_bgImageFile')?.addEventListener('change', function() {
        uploadFile(this.files[0], 'unified_bgImage');
    });
    
    document.getElementById('unified_adName')?.addEventListener('input', function() {
        currentAd.name = this.value;
    });
    
    document.getElementById('unified_adPlatform')?.addEventListener('change', function() {
        currentAd.platform = this.value;
    });
    
    // Save ad
    document.getElementById('unified_saveAdBtn')?.addEventListener('click', saveAd);
    document.getElementById('unified_addToCampaignBtn')?.addEventListener('click', addToCampaign);
    
    // Quick send email
    document.getElementById('unified_quickSendBtn')?.addEventListener('click', quickSendEmail);
    
    // Email form handlers
    document.querySelectorAll('input[name="recipient_source"]').forEach(radio => {
        radio.addEventListener('change', toggleRecipientInputs);
    });
    
    document.getElementById('unified_insertAdBtn')?.addEventListener('click', function() {
        insertAdToEmail();
    });
    
    document.getElementById('unified_previewEmailBtn')?.addEventListener('click', previewEmailWithAds);
    document.getElementById('unified_testEmailBtn')?.addEventListener('click', testEmailWithAds);
    
    document.getElementById('unifiedEmailForm')?.addEventListener('submit', handleEmailSubmit);
    
    // Initialize
    updateAd();
    
    // Auto-fill email content when ad is saved/added to campaign
    window.addEventListener('adSaved', function(e) {
        const savedAdId = e.detail.adId;
        console.log('Ad saved, ID:', savedAdId);
        // Optionally switch to email compose tab
        const emailTab = document.getElementById('email-compose-tab');
        if (emailTab) {
            emailTab.click();
        }
    });
});

function toggleRecipientInputs() {
    const source = document.querySelector('input[name="recipient_source"]:checked')?.value || 'manual';
    if (source === 'crm') {
        document.getElementById('unified_manual_recipients_div').style.display = 'none';
        document.getElementById('unified_crm_recipients_div').style.display = 'block';
        document.getElementById('unified_recipients').removeAttribute('required');
        document.getElementById('unified_daily_limit').setAttribute('required', 'required');
    } else {
        document.getElementById('unified_manual_recipients_div').style.display = 'block';
        document.getElementById('unified_crm_recipients_div').style.display = 'none';
        document.getElementById('unified_recipients').setAttribute('required', 'required');
        document.getElementById('unified_daily_limit').removeAttribute('required');
    }
}

function loadTemplate(templateName) {
    currentAd.template = templateName;
    const template = adTemplates[templateName];
    
    if (template) {
        const headlineEl = document.getElementById('unified_headline');
        const descriptionEl = document.getElementById('unified_description');
        const ctaEl = document.getElementById('unified_cta');
        
        if (headlineEl && !headlineEl.value.trim()) headlineEl.value = template.headline;
        if (descriptionEl && !descriptionEl.value.trim()) descriptionEl.value = template.description;
        if (ctaEl && !ctaEl.value.trim()) ctaEl.value = template.cta;
        
        document.getElementById('unified_ctaColor').value = template.ctaColor;
        document.getElementById('unified_bgColor').value = template.bgColor;
        document.getElementById('unified_textColor').value = template.textColor;
        
        const bgImageEl = document.getElementById('unified_bgImage');
        if (bgImageEl && !bgImageEl.value.trim()) bgImageEl.value = template.bgImage || '';
        
        const destinationUrlEl = document.getElementById('unified_destinationUrl');
        if (destinationUrlEl) destinationUrlEl.value = '';
        
        updateAd();
    }
}

function updateAd() {
    const headlineEl = document.getElementById('unified_headline');
    const descriptionEl = document.getElementById('unified_description');
    const ctaEl = document.getElementById('unified_cta');
    const destinationUrlEl = document.getElementById('unified_destinationUrl');
    
    currentAd.headline = headlineEl ? headlineEl.value.trim() : '';
    currentAd.description = descriptionEl ? descriptionEl.value.trim() : '';
    currentAd.cta = ctaEl ? ctaEl.value.trim() : '';
    currentAd.destinationUrl = destinationUrlEl ? destinationUrlEl.value.trim() : '';
    currentAd.ctaColor = document.getElementById('unified_ctaColor')?.value || '#007bff';
    currentAd.bgColor = document.getElementById('unified_bgColor')?.value || '#ffffff';
    currentAd.textColor = document.getElementById('unified_textColor')?.value || '#000000';
    currentAd.bgImage = document.getElementById('unified_bgImage')?.value.trim() || '';
    currentAd.productImage = document.getElementById('unified_productImage')?.value.trim() || '';
    
    const canvas = document.getElementById('unified_adCanvas');
    if (!canvas) return;
    
    let html = `<div style="width: 100%; height: 100%; background: ${currentAd.bgColor}; background-image: ${currentAd.bgImage ? `url('${currentAd.bgImage}')` : 'none'}; background-size: cover; background-position: center; display: flex; flex-direction: column; justify-content: center; align-items: center; padding: 40px; position: relative; overflow: hidden; box-sizing: border-box;">`;
    
    if (currentAd.productImage) {
        html += `<img src="${escapeHtml(currentAd.productImage)}" alt="Product" style="max-width: 40%; max-height: 40%; object-fit: contain; margin-bottom: 20px; border-radius: 8px;" onerror="this.style.display='none'">`;
    }
    
    if (currentAd.headline) {
        html += `<h2 style="color: ${currentAd.textColor}; font-size: clamp(18px, 4vw, 32px); font-weight: bold; text-align: center; margin: 0 0 15px 0; text-shadow: 2px 2px 4px rgba(0,0,0,0.2); z-index: 1;">${escapeHtml(currentAd.headline)}</h2>`;
    }
    
    if (currentAd.description) {
        html += `<p style="color: ${currentAd.textColor}; font-size: clamp(12px, 2.5vw, 16px); text-align: center; margin: 0 0 25px 0; opacity: 0.9; z-index: 1; line-height: 1.4;">${escapeHtml(currentAd.description)}</p>`;
    }
    
    if (currentAd.cta) {
        html += `<button style="background: ${currentAd.ctaColor}; color: white; border: none; padding: 15px 40px; font-size: clamp(14px, 3vw, 18px); font-weight: bold; border-radius: 50px; cursor: pointer; box-shadow: 0 4px 6px rgba(0,0,0,0.3); transition: transform 0.2s; z-index: 1;">${escapeHtml(currentAd.cta)}</button>`;
    }
    
    html += '</div>';
    canvas.innerHTML = html;
}

function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function uploadFile(file, targetId) {
    if (!file) return;
    
    const formData = new FormData();
    formData.append('file', file);
    formData.append('upload_type', 'direct_mail_design');
    formData.append('csrf_token', '<?= $_SESSION['csrf_token'] ?? '' ?>');
    
    fetch('<?= url("api/file_upload.php") ?>', {
        method: 'POST',
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            const imageUrl = '<?= url("") ?>' + data.file_path;
            document.getElementById(targetId).value = imageUrl;
            updateAd();
        } else {
            alert('Upload failed: ' + (data.error || 'Unknown error'));
        }
    })
    .catch(err => {
        alert('Upload error: ' + err.message);
    });
}

function saveAd() {
    updateAd();
    
    currentAd.name = document.getElementById('unified_adName')?.value.trim() || 'My Ad';
    currentAd.platform = document.getElementById('unified_adPlatform')?.value || 'google_ads';
    
    const hasUserContent = currentAd.headline || currentAd.description || currentAd.productImage;
    if (!hasUserContent) {
        alert('Please enter ad content (headline, description, or image) before saving.');
        return;
    }
    
    const formData = new FormData();
    formData.append('action', 'save_ad');
    formData.append('ad_data', JSON.stringify(currentAd));
    formData.append('csrf_token', document.querySelector('input[name="csrf_token"]')?.value || '<?= generateCSRFToken() ?>');
    
    fetch('<?= url("api/ad_builder.php") ?>', {
        method: 'POST',
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            alert('Ad saved successfully! Ad ID: ' + (data.ad_id || 'N/A'));
            window.dispatchEvent(new CustomEvent('adSaved', { detail: { adId: data.ad_id } }));
        } else {
            alert('Error: ' + (data.error || 'Unknown error'));
        }
    })
    .catch(err => {
        console.error('Save error:', err);
        alert('Error saving ad: ' + err.message);
    });
}

function addToCampaign() {
    const campaignId = document.getElementById('unified_campaignSelect')?.value;
    
    if (!campaignId) {
        alert('Please select a campaign first');
        return;
    }
    
    updateAd();
    currentAd.name = document.getElementById('unified_adName')?.value.trim() || 'My Ad';
    currentAd.platform = document.getElementById('unified_adPlatform')?.value || 'google_ads';
    
    const hasUserContent = currentAd.headline || currentAd.description || currentAd.productImage;
    if (!hasUserContent) {
        alert('Please enter ad content before adding to campaign.');
        return;
    }
    
    const formData = new FormData();
    formData.append('action', 'add_to_campaign');
    formData.append('campaign_id', campaignId);
    formData.append('ad_data', JSON.stringify(currentAd));
    formData.append('csrf_token', document.querySelector('input[name="csrf_token"]')?.value || '<?= generateCSRFToken() ?>');
    
    const btn = document.getElementById('unified_addToCampaignBtn');
    const originalText = btn.innerHTML;
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Adding...';
    
    fetch('<?= url("api/ad_builder.php") ?>', {
        method: 'POST',
        body: formData
    })
    .then(async r => {
        const text = await r.text();
        if (!r.ok) {
            throw new Error(`HTTP ${r.status}: ${text}`);
        }
        return JSON.parse(text);
    })
    .then(data => {
        if (data.success) {
            alert('Ad added to campaign successfully!');
            window.dispatchEvent(new CustomEvent('adSaved', { detail: { adId: data.ad_id } }));
            // Switch to email compose tab
            document.getElementById('email-compose-tab').click();
        } else {
            alert('Error: ' + (data.error || 'Unknown error'));
        }
    })
    .catch(err => {
        console.error('Add to campaign error:', err);
        alert('Error: ' + err.message);
    })
    .finally(() => {
        btn.disabled = false;
        btn.innerHTML = originalText;
    });
}

// Ad insertion into email (from email_campaigns.php)
const campaignsWithAds = <?= json_encode($campaignsWithAds) ?>;

function insertAdToEmail() {
    if (campaignsWithAds.length === 0) {
        alert('No campaigns with ads found. Create an ad first and add it to a campaign.');
        return;
    }
    
    let modalHtml = '<div class="modal fade" id="selectCampaignModal" tabindex="-1"><div class="modal-dialog"><div class="modal-content">';
    modalHtml += '<div class="modal-header"><h5 class="modal-title">Select Campaign with Ads</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>';
    modalHtml += '<div class="modal-body">';
    modalHtml += '<div class="list-group">';
    
    campaignsWithAds.forEach(campaign => {
        modalHtml += `<button type="button" class="list-group-item list-group-item-action" onclick="loadCampaignAds(${campaign.id}, 'unified_email_content')">${escapeHtml(campaign.campaign_name)}</button>`;
    });
    
    modalHtml += '</div></div></div></div></div>';
    
    const existing = document.getElementById('selectCampaignModal');
    if (existing) existing.remove();
    
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    const modal = new bootstrap.Modal(document.getElementById('selectCampaignModal'));
    modal.show();
    
    document.getElementById('selectCampaignModal').addEventListener('hidden.bs.modal', function() {
        this.remove();
    });
}

window.loadCampaignAds = function(campaignId, targetTextarea) {
    const campaignModal = document.getElementById('selectCampaignModal');
    if (campaignModal) {
        bootstrap.Modal.getInstance(campaignModal).hide();
    }
    
    fetch(`<?= url("api/ad_builder.php") ?>?action=list_campaign_ads&campaign_id=${campaignId}`)
        .then(async r => {
            const text = await r.text();
            if (!r.ok) {
                throw new Error(`HTTP ${r.status}: ${text}`);
            }
            return JSON.parse(text);
        })
        .then(data => {
            if (data.success && data.data && data.data.length > 0) {
                const validAds = data.data.filter(ad => {
                    const adData = ad.ad_data || {};
                    return adData.headline || adData.description || adData.productImage || adData.cta;
                });
                
                if (validAds.length === 0) {
                    alert('No ads with content found. Please ensure ads have been saved with content.');
                    return;
                }
                
                showAdSelectionModal(validAds, targetTextarea || 'unified_email_content');
            } else {
                alert('No ads found for this campaign');
            }
        })
        .catch(err => {
            alert('Error loading ads: ' + err.message);
        });
};

function showAdSelectionModal(ads, targetTextareaId) {
    const modalId = 'selectAdModal_' + Date.now();
    window['selectedAds_' + modalId] = ads;
    
    let modalHtml = '<div class="modal fade" id="' + modalId + '" tabindex="-1"><div class="modal-dialog modal-lg"><div class="modal-content">';
    modalHtml += '<div class="modal-header"><h5 class="modal-title">Select Ad to Insert</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>';
    modalHtml += '<div class="modal-body"><div class="row">';
    
    ads.forEach((ad, index) => {
        const adData = ad.ad_data || {};
        const campaignAdId = ad.id || ad.campaign_ad_id || 'unknown';
        
        const mockIndicators = ['Shop the Latest Deals', 'Discover amazing products', 'Limited time offer', 
                                'Your Headline Here', 'Your description goes here'];
        const isMockData = mockIndicators.some(mock => 
            (adData.headline && adData.headline.includes(mock)) || 
            (adData.description && adData.description.includes(mock))
        );
        
        modalHtml += '<div class="col-md-6 mb-3">';
        modalHtml += '<div class="card' + (isMockData ? ' border-warning' : '') + '">';
        modalHtml += '<div class="card-body">';
        modalHtml += `<h6>${escapeHtml(ad.ad_name)}</h6>`;
        
        if (isMockData) {
            modalHtml += '<div class="alert alert-warning mt-2 mb-2 py-1 px-2"><small><strong>⚠️ Template Data</strong></small></div>';
        }
        
        if (adData.headline || adData.productImage) {
            modalHtml += '<div class="mt-2 p-2" style="border: 1px solid #ddd; border-radius: 4px; min-height: 100px; background: ' + (adData.bgColor || '#ffffff') + '; text-align: center;">';
            if (adData.productImage) {
                modalHtml += `<img src="${escapeHtml(adData.productImage)}" style="max-width: 100px; height: auto;" onerror="this.style.display='none'">`;
            }
            if (adData.headline) {
                modalHtml += `<p class="mb-0 mt-2"><strong>${escapeHtml(adData.headline)}</strong></p>`;
            }
            modalHtml += '</div>';
        }
        
        modalHtml += '<div class="mt-2">';
        modalHtml += `<button class="btn btn-sm btn-primary w-100 insert-ad-btn" data-ad-index="${index}" data-modal-id="${modalId}" data-textarea-id="${targetTextareaId}">Insert This Ad</button>`;
        modalHtml += '</div>';
        modalHtml += '</div></div></div>';
    });
    
    modalHtml += '</div></div></div></div></div>';
    
    document.querySelectorAll('[id^="selectAdModal_"]').forEach(el => el.remove());
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    const modal = new bootstrap.Modal(document.getElementById(modalId));
    modal.show();
    
    document.getElementById(modalId).addEventListener('click', function(e) {
        const btn = e.target.closest('.insert-ad-btn');
        if (btn) {
            const adIndex = parseInt(btn.dataset.adIndex);
            const modalId = btn.dataset.modalId;
            const textareaId = btn.dataset.textareaId;
            const ads = window['selectedAds_' + modalId];
            
            if (ads && ads[adIndex]) {
                insertAdHTMLDirect(ads[adIndex], textareaId);
                bootstrap.Modal.getInstance(document.getElementById(modalId)).hide();
            }
        }
    });
    
    document.getElementById(modalId).addEventListener('hidden.bs.modal', function() {
        delete window['selectedAds_' + modalId];
        this.remove();
    });
}

function insertAdHTMLDirect(ad, textareaId) {
    if (!ad || !ad.ad_data || typeof ad.ad_data !== 'object') {
        alert('Error: Ad data not found.');
        return;
    }
    
    const adData = ad.ad_data;
    const textarea = document.getElementById(textareaId);
    if (!textarea) {
        alert('Textarea not found');
        return;
    }
    
    const campaignAdId = ad.id || ad.campaign_ad_id || null;
    const destinationUrl = adData.destinationUrl || '#';
    
    let ctaUrl = '#';
    if (campaignAdId) {
        const acceleratedadzBase = '<?= url("acceleratedadz.php") ?>';
        ctaUrl = acceleratedadzBase + '?campaign_ad=' + campaignAdId + '&source=campaign&redirect=' + encodeURIComponent(destinationUrl);
    } else if (destinationUrl !== '#') {
        ctaUrl = destinationUrl;
    }
    
    let adHTML = '\n<!-- Ad: ' + escapeHtml(ad.ad_name) + ' -->\n';
    adHTML += '<table cellpadding="0" cellspacing="0" border="0" width="100%" style="max-width: 600px; margin: 20px auto;">\n';
    adHTML += '<tr><td style="padding: 20px; background: ' + (adData.bgColor || '#ffffff') + '; text-align: center;">\n';
    
    if (adData.productImage) {
        adHTML += '<img src="' + escapeHtml(adData.productImage) + '" alt="' + escapeHtml(adData.headline || ad.ad_name || 'Ad') + '" style="max-width: 300px; height: auto; display: block; margin: 0 auto 15px;">\n';
    }
    
    if (adData.headline && adData.headline.trim()) {
        adHTML += '<h2 style="color: ' + (adData.textColor || '#000000') + '; margin: 10px 0;">' + escapeHtml(adData.headline.trim()) + '</h2>\n';
    }
    
    if (adData.description && adData.description.trim()) {
        adHTML += '<p style="color: ' + (adData.textColor || '#000000') + '; margin: 10px 0;">' + escapeHtml(adData.description.trim()) + '</p>\n';
    }
    
    if (adData.cta && adData.cta.trim()) {
        adHTML += '<a href="' + escapeHtml(ctaUrl) + '" style="display: inline-block; background: ' + (adData.ctaColor || '#007bff') + '; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin-top: 15px;">' + escapeHtml(adData.cta.trim()) + '</a>\n';
    }
    
    adHTML += '</td></tr>\n</table>\n';
    adHTML += '<!-- End Ad -->\n';
    
    const start = textarea.selectionStart || 0;
    const end = textarea.selectionEnd || 0;
    const text = textarea.value;
    textarea.value = text.substring(0, start) + adHTML + text.substring(end);
    textarea.selectionStart = textarea.selectionEnd = start + adHTML.length;
    textarea.focus();
}

function previewEmailWithAds() {
    const content = document.getElementById('unified_email_content')?.value || '';
    if (!content) {
        alert('Please enter email content first');
        return;
    }
    
    const previewWindow = window.open('', '_blank', 'width=800,height=800,scrollbars=yes');
    previewWindow.document.write('<!DOCTYPE html><html><head><title>Email Preview</title><meta charset="UTF-8"><style>body { font-family: Arial, sans-serif; padding: 20px; margin: 0; background: #f5f5f5; } .preview-container { max-width: 600px; margin: 0 auto; background: white; padding: 20px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }</style></head><body><div class="preview-container">');
    previewWindow.document.write('<h3>📧 Email Preview</h3>');
    previewWindow.document.write('<div class="email-content">');
    previewWindow.document.write(content);
    previewWindow.document.write('</div></div></body></html>');
    previewWindow.document.close();
}

function testEmailWithAds() {
    const content = document.getElementById('unified_email_content')?.value || '';
    const subject = document.getElementById('unified_subject_line')?.value || 'Test Email';
    const testEmail = prompt('Enter your test email address:');
    
    if (!testEmail || !testEmail.includes('@')) {
        alert('Please enter a valid email address');
        return;
    }
    
    if (!content) {
        alert('Please enter email content first');
        return;
    }
    
    const formData = new FormData();
    formData.append('action', 'send_test_email');
    formData.append('to', testEmail);
    formData.append('subject', subject);
    formData.append('html_content', content);
    formData.append('csrf_token', document.querySelector('input[name="csrf_token"]')?.value || '<?= generateCSRFToken() ?>');
    
    fetch('<?= url("api/email_service.php") ?>', {
        method: 'POST',
        body: formData
    })
    .then(async r => {
        const text = await r.text();
        if (!r.ok) {
            throw new Error(`HTTP ${r.status}: ${text}`);
        }
        return JSON.parse(text);
    })
    .then(data => {
        if (data.success) {
            alert('Test email sent successfully to ' + testEmail + '!');
        } else {
            alert('Error: ' + (data.error || 'Failed to send test email'));
        }
    })
    .catch(err => {
        alert('Error: ' + err.message);
    });
}

function handleEmailSubmit(e) {
    e.preventDefault();
    const formData = new FormData(this);
    const submitBtn = this.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;
    submitBtn.disabled = true;
    submitBtn.textContent = 'Creating...';
    
    fetch(this.action, {
        method: 'POST',
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            alert(data.message || 'Campaign created successfully!');
            // Switch to campaigns tab
            document.getElementById('email-campaigns-tab').click();
            // Reload campaigns list (you might want to fetch fresh data)
            setTimeout(() => window.location.reload(), 1000);
        } else {
            alert('Error: ' + (data.error || 'Failed to create campaign'));
            submitBtn.disabled = false;
            submitBtn.textContent = originalText;
        }
    })
    .catch(err => {
        alert('Error: ' + err.message);
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
    });
}

// Quick Send Email - uses current ad without saving
function quickSendEmail() {
    // Update ad data from form first
    updateAd();
    
    // Validate ad has content
    const hasContent = currentAd.headline || currentAd.description || currentAd.productImage;
    if (!hasContent) {
        alert('Please enter ad content (headline, description, or image) before sending.');
        return;
    }
    
    // Generate ad HTML for email
    const adHTML = generateQuickAdHTML(currentAd);
    
    // Show quick send modal
    const modalHtml = `
        <div class="modal fade" id="quickSendModal" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header bg-warning text-dark">
                        <h5 class="modal-title"><i class="bi bi-lightning-charge"></i> Quick Send Email</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-info">
                            <small><i class="bi bi-info-circle"></i> This ad will be embedded in the email without saving it to your ads library.</small>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Campaign Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="quick_campaign_name" placeholder="e.g., Quick Launch Email" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Subject Line <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="quick_subject_line" placeholder="Your email subject" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Additional Email Content (Optional)</label>
                            <textarea class="form-control" id="quick_email_content" rows="4" placeholder="Add any additional text before or after the ad..."></textarea>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Recipient Source <span class="text-danger">*</span></label>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="quick_recipient_source" id="quick_recipient_manual" value="manual" checked>
                                <label class="form-check-label" for="quick_recipient_manual">
                                    Manual Entry (comma-separated emails)
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="quick_recipient_source" id="quick_recipient_crm" value="crm">
                                <label class="form-check-label" for="quick_recipient_crm">
                                    CRM Customers (<?= number_format($subscriberCount) ?> subscribed)
                                </label>
                            </div>
                        </div>
                        <div class="mb-3" id="quick_manual_recipients_div">
                            <label for="quick_recipients" class="form-label">Recipients <span class="text-danger">*</span></label>
                            <textarea class="form-control" id="quick_recipients" rows="3" placeholder="email1@example.com, email2@example.com" required></textarea>
                        </div>
                        <div class="mb-3" id="quick_crm_recipients_div" style="display: none;">
                            <div class="alert alert-success">
                                <h6 class="alert-heading"><i class="bi bi-robot"></i> Automated CRM Sending</h6>
                                <p class="mb-2"><strong>How it works:</strong></p>
                                <ul class="mb-0 small">
                                    <li>Emails are automatically sent to your CRM customers</li>
                                    <li><strong>Rate:</strong> 1 email per minute (automated)</li>
                                    <li><strong>Daily Limit:</strong> Set maximum emails per day</li>
                                    <li>Sending runs continuously via background processor</li>
                                </ul>
                            </div>
                            <label for="quick_daily_limit" class="form-label">
                                Maximum Daily Emails <span class="text-danger">*</span>
                                <span class="badge bg-info ms-2">Total: <?= number_format($subscriberCount) ?> customers</span>
                            </label>
                            <input type="number" class="form-control" id="quick_daily_limit" min="1" max="<?= $subscriberCount ?>" placeholder="e.g., 100" required>
                            <small class="text-muted">
                                <i class="bi bi-info-circle"></i> 
                                Enter the maximum number of emails to send per day. 
                                Sending starts immediately and continues automatically at 1 per minute until daily limit is reached.
                            </small>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Ad Preview</label>
                            <div class="border rounded p-3" style="max-height: 300px; overflow-y: auto;">
                                ${adHTML}
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-warning" id="quick_send_confirm_btn">
                            <i class="bi bi-send-fill"></i> Send Email Now
                        </button>
                        <button type="button" class="btn btn-success" id="quick_automate_crm_btn" style="display: none;">
                            <i class="bi bi-robot"></i> Start CRM Automation
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    document.querySelectorAll('#quickSendModal').forEach(el => el.remove());
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    const modal = new bootstrap.Modal(document.getElementById('quickSendModal'));
    modal.show();
    
    // Toggle recipient inputs
    document.querySelectorAll('input[name="quick_recipient_source"]').forEach(radio => {
        radio.addEventListener('change', function() {
            if (this.value === 'crm') {
                document.getElementById('quick_manual_recipients_div').style.display = 'none';
                document.getElementById('quick_crm_recipients_div').style.display = 'block';
                document.getElementById('quick_recipients').removeAttribute('required');
                document.getElementById('quick_daily_limit').setAttribute('required', 'required');
                // Show automation button, hide regular send
                document.getElementById('quick_automate_crm_btn').style.display = 'inline-block';
                document.getElementById('quick_send_confirm_btn').style.display = 'none';
            } else {
                document.getElementById('quick_manual_recipients_div').style.display = 'block';
                document.getElementById('quick_crm_recipients_div').style.display = 'none';
                document.getElementById('quick_recipients').setAttribute('required', 'required');
                document.getElementById('quick_daily_limit').removeAttribute('required');
                // Show regular send, hide automation button
                document.getElementById('quick_automate_crm_btn').style.display = 'none';
                document.getElementById('quick_send_confirm_btn').style.display = 'inline-block';
            }
        });
    });
    
    // Manual send handler
    document.getElementById('quick_send_confirm_btn').addEventListener('click', function() {
        handleQuickSend(false);
    });
    
    // CRM Automation handler
    document.getElementById('quick_automate_crm_btn').addEventListener('click', function() {
        handleQuickSend(true);
    });
    
    function handleQuickSend(isAutomated) {
        const campaignName = document.getElementById('quick_campaign_name').value.trim();
        const subjectLine = document.getElementById('quick_subject_line').value.trim();
        const additionalContent = document.getElementById('quick_email_content').value.trim();
        const recipientSource = document.querySelector('input[name="quick_recipient_source"]:checked')?.value || 'manual';
        const recipients = document.getElementById('quick_recipients').value.trim();
        const dailyLimit = document.getElementById('quick_daily_limit').value;
        
        if (!campaignName || !subjectLine) {
            alert('Please fill in campaign name and subject line.');
            return;
        }
        
        if (recipientSource === 'manual' && !recipients) {
            alert('Please enter recipients.');
            return;
        }
        
        if (recipientSource === 'crm' && (!dailyLimit || parseInt(dailyLimit) < 1)) {
            alert('Please enter a valid daily email limit (minimum 1).');
            return;
        }
        
        // Build email content with ad
        let emailContent = additionalContent ? additionalContent + '\n\n' : '';
        emailContent += adHTML;
        
        const formData = new FormData();
        formData.append('action', 'create_campaign');
        formData.append('campaign_name', campaignName);
        formData.append('subject_line', subjectLine);
        formData.append('email_content', emailContent);
        formData.append('recipient_source', recipientSource);
        formData.append('recipients', recipients);
        if (dailyLimit) formData.append('daily_limit', parseInt(dailyLimit));
        formData.append('csrf_token', document.querySelector('input[name="csrf_token"]')?.value || '<?= generateCSRFToken() ?>');
        
        const btn = isAutomated ? document.getElementById('quick_automate_crm_btn') : document.getElementById('quick_send_confirm_btn');
        const originalText = btn.innerHTML;
        btn.disabled = true;
        btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> ' + (isAutomated ? 'Starting Automation...' : 'Sending...');
        
        fetch('<?= url("api/email_service.php") ?>', {
            method: 'POST',
            body: formData
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                let message = data.message || 'Email campaign created successfully!';
                if (isAutomated && recipientSource === 'crm') {
                    message = `✅ CRM Automation Started!\n\n` +
                             `Campaign: ${campaignName}\n` +
                             `Daily Limit: ${dailyLimit} emails\n` +
                             `Rate: 1 email per minute\n` +
                             `Total Customers: ${data.recipient_count || 'N/A'}\n\n` +
                             `Emails are now being sent automatically. Check the "My Email Campaigns" tab to monitor progress.`;
                }
                alert(message);
                bootstrap.Modal.getInstance(document.getElementById('quickSendModal')).hide();
                // Switch to campaigns tab
                document.getElementById('email-campaigns-tab').click();
                setTimeout(() => window.location.reload(), 1500);
            } else {
                alert('Error: ' + (data.error || 'Failed to create campaign'));
                btn.disabled = false;
                btn.innerHTML = originalText;
            }
        })
        .catch(err => {
            alert('Error: ' + err.message);
            btn.disabled = false;
            btn.innerHTML = originalText;
        });
    }
    
    document.getElementById('quickSendModal').addEventListener('hidden.bs.modal', function() {
        this.remove();
    });
}

function generateQuickAdHTML(adData) {
    const destinationUrl = adData.destinationUrl || '#';
    
    let html = '\n<!-- Quick Send Ad (Not Saved) -->\n';
    html += '<table cellpadding="0" cellspacing="0" border="0" width="100%" style="max-width: 600px; margin: 20px auto;">\n';
    html += '<tr><td style="padding: 20px; background: ' + (adData.bgColor || '#ffffff') + '; text-align: center;">\n';
    
    if (adData.productImage) {
        html += '<img src="' + escapeHtml(adData.productImage) + '" alt="' + escapeHtml(adData.headline || 'Ad') + '" style="max-width: 300px; height: auto; display: block; margin: 0 auto 15px;">\n';
    }
    
    if (adData.headline && adData.headline.trim()) {
        html += '<h2 style="color: ' + (adData.textColor || '#000000') + '; margin: 10px 0;">' + escapeHtml(adData.headline.trim()) + '</h2>\n';
    }
    
    if (adData.description && adData.description.trim()) {
        html += '<p style="color: ' + (adData.textColor || '#000000') + '; margin: 10px 0;">' + escapeHtml(adData.description.trim()) + '</p>\n';
    }
    
    if (adData.cta && adData.cta.trim()) {
        html += '<a href="' + escapeHtml(destinationUrl) + '" style="display: inline-block; background: ' + (adData.ctaColor || '#007bff') + '; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin-top: 15px;">' + escapeHtml(adData.cta.trim()) + '</a>\n';
    }
    
    html += '</td></tr>\n</table>\n';
    html += '<!-- End Quick Send Ad -->\n';
    
    return html;
}

// Delete Campaign Handler
document.querySelectorAll('.delete-campaign-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        const campaignId = this.dataset.id;
        const campaignName = this.closest('tr').querySelector('td:first-child strong').textContent;
        
        if (!confirm('Are you sure you want to delete the campaign "' + campaignName + '"?\n\nThis action cannot be undone.')) {
            return;
        }
        
        const originalText = this.innerHTML;
        this.disabled = true;
        this.innerHTML = '<span class="spinner-border spinner-border-sm"></span>';
        
        const formData = new FormData();
        formData.append('action', 'delete_campaign');
        formData.append('campaign_id', campaignId);
        formData.append('csrf_token', document.querySelector('input[name="csrf_token"]')?.value || '<?= generateCSRFToken() ?>');
        
        fetch('<?= url("api/email_service.php") ?>', {
            method: 'POST',
            body: formData
        })
        .then(async r => {
            const text = await r.text();
            if (!r.ok) {
                throw new Error(`HTTP ${r.status}: ${text}`);
            }
            return JSON.parse(text);
        })
        .then(data => {
            if (data.success) {
                alert('Campaign deleted successfully!');
                location.reload();
            } else {
                alert('Error: ' + (data.error || 'Failed to delete campaign'));
                this.disabled = false;
                this.innerHTML = originalText;
            }
        })
        .catch(err => {
            alert('Error: ' + err.message);
            this.disabled = false;
            this.innerHTML = originalText;
        });
    });
});

// Send Ad Individually Handler
document.querySelectorAll('.send-ad-individually-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        const campaignId = this.dataset.campaignId;
        
        // Show modal to enter recipient email
        const modalHtml = `
            <div class="modal fade" id="sendAdIndividuallyModal" tabindex="-1">
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title"><i class="bi bi-envelope-paper"></i> Send Ad Individually</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <div class="mb-3">
                                <label class="form-label">Recipient Email <span class="text-danger">*</span></label>
                                <input type="email" class="form-control" id="individualRecipientEmail" 
                                       placeholder="recipient@example.com" required>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Subject Line</label>
                                <input type="text" class="form-control" id="individualEmailSubject" 
                                       value="" placeholder="Optional - uses campaign subject if empty">
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Additional Message (Optional)</label>
                                <textarea class="form-control" id="individualEmailMessage" rows="3" 
                                          placeholder="Add a personal message before the ad..."></textarea>
                            </div>
                            <div id="individualSendResult"></div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="button" class="btn btn-success" id="confirmSendAdIndividuallyBtn">
                                <i class="bi bi-send"></i> Send Ad
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Remove existing modal if any
        document.querySelectorAll('#sendAdIndividuallyModal').forEach(el => el.remove());
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        const modal = new bootstrap.Modal(document.getElementById('sendAdIndividuallyModal'));
        modal.show();
        
        // Fetch campaign details
        fetch(`<?= url("api/email_service.php") ?>?action=get_campaign&campaign_id=${campaignId}`)
            .then(async r => {
                const text = await r.text();
                if (!r.ok) {
                    throw new Error(`HTTP ${r.status}: ${text}`);
                }
                return JSON.parse(text);
            })
            .then(data => {
                if (data.success && data.campaign) {
                    document.getElementById('individualEmailSubject').value = data.campaign.subject_line || '';
                }
            })
            .catch(err => {
                console.error('Error fetching campaign:', err);
            });
        
        // Handle send button click
        document.getElementById('confirmSendAdIndividuallyBtn').addEventListener('click', function() {
            const recipientEmail = document.getElementById('individualRecipientEmail').value;
            const subject = document.getElementById('individualEmailSubject').value;
            const additionalMessage = document.getElementById('individualEmailMessage').value;
            const resultDiv = document.getElementById('individualSendResult');
            
            if (!recipientEmail || !recipientEmail.includes('@')) {
                resultDiv.innerHTML = '<div class="alert alert-warning">Please enter a valid email address.</div>';
                return;
            }
            
            const sendBtn = this;
            const originalText = sendBtn.innerHTML;
            sendBtn.disabled = true;
            sendBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Sending...';
            resultDiv.innerHTML = '';
            
            const formData = new FormData();
            formData.append('action', 'send_ad_individually');
            formData.append('campaign_id', campaignId);
            formData.append('recipient_email', recipientEmail);
            formData.append('subject', subject);
            formData.append('additional_message', additionalMessage);
            formData.append('csrf_token', document.querySelector('input[name="csrf_token"]')?.value || '<?= generateCSRFToken() ?>');
            
            fetch('<?= url("api/email_service.php") ?>', {
                method: 'POST',
                body: formData
            })
            .then(async r => {
                const text = await r.text();
                if (!r.ok) {
                    throw new Error(`HTTP ${r.status}: ${text}`);
                }
                return JSON.parse(text);
            })
            .then(data => {
                if (data.success) {
                    resultDiv.innerHTML = '<div class="alert alert-success"><i class="bi bi-check-circle"></i> Ad sent successfully to ' + recipientEmail + '!</div>';
                    setTimeout(() => {
                        modal.hide();
                        document.getElementById('sendAdIndividuallyModal').remove();
                    }, 2000);
                } else {
                    resultDiv.innerHTML = '<div class="alert alert-danger"><i class="bi bi-x-circle"></i> Error: ' + (data.error || 'Failed to send ad') + '</div>';
                    sendBtn.disabled = false;
                    sendBtn.innerHTML = originalText;
                }
            })
            .catch(err => {
                resultDiv.innerHTML = '<div class="alert alert-danger"><i class="bi bi-x-circle"></i> Error: ' + err.message + '</div>';
                sendBtn.disabled = false;
                sendBtn.innerHTML = originalText;
            });
        });
        
        // Clean up on modal close
        document.getElementById('sendAdIndividuallyModal').addEventListener('hidden.bs.modal', function() {
            this.remove();
        });
    });
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

