<?php
/**
 * Testing Dashboard
 * Comprehensive testing interface for all marketing platform flows
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireClient();
$pageTitle = 'Testing Dashboard';

$userId = $_SESSION['user_id'];
$db = getDB();

// Get current data for status display
$stmt = $db->prepare("SELECT COUNT(*) FROM campaigns WHERE user_id = ?");
$stmt->execute([$userId]);
$campaignCount = $stmt->fetchColumn();

$stmt = $db->prepare("SELECT COUNT(*) FROM saved_ads WHERE user_id = ?");
$stmt->execute([$userId]);
$adsCount = $stmt->fetchColumn();

$stmt = $db->prepare("SELECT COUNT(*) FROM email_campaigns WHERE user_id = ?");
$stmt->execute([$userId]);
$emailCampaignsCount = $stmt->fetchColumn();

$stmt = $db->prepare("SELECT COUNT(*) FROM email_subscribers WHERE user_id = ? AND status = 'subscribed'");
$stmt->execute([$userId]);
$crmCustomersCount = $stmt->fetchColumn();

// Get latest campaign with ads
$stmt = $db->prepare("
    SELECT c.*, COUNT(ca.id) as ad_count
    FROM campaigns c
    LEFT JOIN campaign_ads ca ON c.id = ca.campaign_id AND ca.status = 'active'
    WHERE c.user_id = ?
    GROUP BY c.id
    ORDER BY c.created_at DESC
    LIMIT 1
");
$stmt->execute([$userId]);
$latestCampaign = $stmt->fetch();

// Get latest email campaign
$stmt = $db->prepare("SELECT * FROM email_campaigns WHERE user_id = ? ORDER BY created_at DESC LIMIT 1");
$stmt->execute([$userId]);
$latestEmailCampaign = $stmt->fetch();

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <h1><i class="bi bi-speedometer2"></i> Testing Dashboard</h1>
        <p class="text-muted">Complete flow testing interface for email-first campaign workflow</p>
    </div>
</div>

<!-- Status Overview -->
<div class="row mb-4">
    <div class="col-md-3 mb-3">
        <div class="card">
            <div class="card-body">
                <h6 class="card-title">Campaigns</h6>
                <h3 class="mb-0"><?= $campaignCount ?></h3>
                <?php if ($latestCampaign): ?>
                    <small class="text-muted">Latest: <?= escape($latestCampaign['campaign_name']) ?></small>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card">
            <div class="card-body">
                <h6 class="card-title">Saved Ads</h6>
                <h3 class="mb-0"><?= $adsCount ?></h3>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card">
            <div class="card-body">
                <h6 class="card-title">Email Campaigns</h6>
                <h3 class="mb-0"><?= $emailCampaignsCount ?></h3>
                <?php if ($latestEmailCampaign): ?>
                    <small class="text-muted">Latest: <?= escape($latestEmailCampaign['campaign_name']) ?></small>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card">
            <div class="card-body">
                <h6 class="card-title">CRM Customers</h6>
                <h3 class="mb-0"><?= $crmCustomersCount ?></h3>
            </div>
        </div>
    </div>
</div>

<!-- Flow Testing Steps -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-list-check"></i> Complete Flow Testing</h5>
            </div>
            <div class="card-body">
                <div class="accordion" id="flowAccordion">
                    <!-- Step 1: Create Campaign -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button" type="button" data-bs-toggle="collapse" data-bs-target="#step1">
                                <strong>Step 1:</strong> Create Email Campaign (Default)
                            </button>
                        </h2>
                        <div id="step1" class="accordion-collapse collapse show" data-bs-parent="#flowAccordion">
                            <div class="accordion-body">
                                <p>Create a campaign with Email Marketing as the default type.</p>
                                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createCampaignModal">
                                    <i class="bi bi-plus-circle"></i> Create Campaign
                                </button>
                                <?php if ($latestCampaign): ?>
                                    <div class="alert alert-success mt-2">
                                        <strong>✓ Latest Campaign:</strong> <?= escape($latestCampaign['campaign_name']) ?> 
                                        (<?= ucfirst(str_replace('_', ' ', $latestCampaign['campaign_type'])) ?>, 
                                        <?= $latestCampaign['ad_count'] ?> ads)
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- Step 2: Create Ad -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#step2">
                                <strong>Step 2:</strong> Create Ad in Ad Builder
                            </button>
                        </h2>
                        <div id="step2" class="accordion-collapse collapse" data-bs-parent="#flowAccordion">
                            <div class="accordion-body">
                                <p>Use the Ad Builder to create a visual ad.</p>
                                <a href="<?= url('client/ad_builder.php') ?>" class="btn btn-primary" target="_blank">
                                    <i class="bi bi-palette"></i> Open Ad Builder
                                </a>
                            </div>
                        </div>
                    </div>

                    <!-- Step 3: Add Ad to Campaign -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#step3">
                                <strong>Step 3:</strong> Add Ad to Campaign
                            </button>
                        </h2>
                        <div id="step3" class="accordion-collapse collapse" data-bs-parent="#flowAccordion">
                            <div class="accordion-body">
                                <p>In the Ad Builder, click "Add to Campaign" and select your email campaign.</p>
                                <?php if ($latestCampaign && $latestCampaign['ad_count'] > 0): ?>
                                    <div class="alert alert-success">
                                        <strong>✓ Campaign has ads:</strong> <?= $latestCampaign['ad_count'] ?> ad(s) linked
                                    </div>
                                <?php else: ?>
                                    <div class="alert alert-warning">
                                        No ads linked to latest campaign yet.
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- Step 4: Create Email Content -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#step4">
                                <strong>Step 4:</strong> Create Email Campaign Content
                            </button>
                        </h2>
                        <div id="step4" class="accordion-collapse collapse" data-bs-parent="#flowAccordion">
                            <div class="accordion-body">
                                <p>Create email campaign and write content. Use "Insert Ad" to add campaign ads.</p>
                                <a href="<?= url('client/email_campaigns.php') ?>" class="btn btn-primary" target="_blank">
                                    <i class="bi bi-envelope"></i> Open Email Campaigns
                                </a>
                                <?php if ($latestEmailCampaign): ?>
                                    <div class="alert alert-success mt-2">
                                        <strong>✓ Latest Email Campaign:</strong> <?= escape($latestEmailCampaign['campaign_name']) ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- Step 5: Insert Ads -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#step5">
                                <strong>Step 5:</strong> Insert Ads from Campaign
                            </button>
                        </h2>
                        <div id="step5" class="accordion-collapse collapse" data-bs-parent="#flowAccordion">
                            <div class="accordion-body">
                                <p>In Email Campaigns, click "Insert Ad" button and select ads from your campaigns.</p>
                                <div class="alert alert-info">
                                    <strong>Note:</strong> Ads are inserted with tracking links to <code>acceleratedadz.php</code>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Step 6: Preview -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#step6">
                                <strong>Step 6:</strong> Preview Email with Ads
                            </button>
                        </h2>
                        <div id="step6" class="accordion-collapse collapse" data-bs-parent="#flowAccordion">
                            <div class="accordion-body">
                                <p>Use the "Preview" button in Email Campaigns to see the complete email with inserted ads.</p>
                            </div>
                        </div>
                    </div>

                    <!-- Step 7: Send to CRM -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#step7">
                                <strong>Step 7:</strong> Send to CRM Customers
                            </button>
                        </h2>
                        <div id="step7" class="accordion-collapse collapse" data-bs-parent="#flowAccordion">
                            <div class="accordion-body">
                                <p>When creating email campaign, select "Send to CRM Customers" and set daily limit.</p>
                                <p><strong>Rate Limiting:</strong> 1 email per minute, respects daily limit.</p>
                                <div class="alert alert-info">
                                    <strong>CRM Customers:</strong> <?= $crmCustomersCount ?> subscribed customers available
                                </div>
                                <div class="alert alert-warning">
                                    <strong>Queue Processor:</strong> Run <code>php market/api/email_queue_processor.php</code> via cron every minute
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Step 8: View AcceleratedAdz -->
                    <div class="accordion-item">
                        <h2 class="accordion-header">
                            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#step8">
                                <strong>Step 8:</strong> View Ad on AcceleratedAdz
                            </button>
                        </h2>
                        <div id="step8" class="accordion-collapse collapse" data-bs-parent="#flowAccordion">
                            <div class="accordion-body">
                                <p>Ads inserted in emails link to <code>public_html/acceleratedadz/acceleratedadz.php</code> for display and tracking.</p>
                                <?php if ($latestCampaign && $latestCampaign['ad_count'] > 0): ?>
                                    <?php
                                    // Get first ad from campaign
                                    $stmt = $db->prepare("
                                        SELECT ca.id 
                                        FROM campaign_ads ca
                                        WHERE ca.campaign_id = ? AND ca.status = 'active'
                                        LIMIT 1
                                    ");
                                    $stmt->execute([$latestCampaign['id']]);
                                    $firstAd = $stmt->fetch();
                                    if ($firstAd):
                                    ?>
                                        <a href="/acceleratedadz/acceleratedadz.php?campaign_ad=<?= $firstAd['id'] ?>&source=campaign" 
                                           class="btn btn-primary" target="_blank">
                                            <i class="bi bi-box-arrow-up-right"></i> View Ad (<?= escape($latestCampaign['campaign_name']) ?>)
                                        </a>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <div class="alert alert-warning">
                                        Create an ad first to view it on AcceleratedAdz
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Quick Links -->
<div class="row mt-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-link-45deg"></i> Quick Links</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-4 mb-2">
                        <a href="<?= url('client/campaigns.php') ?>" class="btn btn-outline-primary w-100">
                            <i class="bi bi-bullseye"></i> Campaigns
                        </a>
                    </div>
                    <div class="col-md-4 mb-2">
                        <a href="<?= url('client/ad_builder.php') ?>" class="btn btn-outline-primary w-100">
                            <i class="bi bi-palette"></i> Ad Builder
                        </a>
                    </div>
                    <div class="col-md-4 mb-2">
                        <a href="<?= url('client/email_campaigns.php') ?>" class="btn btn-outline-primary w-100">
                            <i class="bi bi-envelope"></i> Email Campaigns
                        </a>
                    </div>
                    <div class="col-md-4 mb-2">
                        <a href="<?= url('client/crm.php') ?>" class="btn btn-outline-primary w-100">
                            <i class="bi bi-people"></i> CRM Customers
                        </a>
                    </div>
                    <div class="col-md-4 mb-2">
                        <a href="/acceleratedadz/acceleratedadz.php" class="btn btn-outline-primary w-100" target="_blank">
                            <i class="bi bi-box-arrow-up-right"></i> AcceleratedAdz
                        </a>
                    </div>
                    <div class="col-md-4 mb-2">
                        <a href="<?= url('client/settings.php') ?>" class="btn btn-outline-primary w-100">
                            <i class="bi bi-gear"></i> Settings (Demo Mode)
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Testing Data Display -->
<div class="row mt-4">
    <div class="col-md-6">
        <div class="card">
            <div class="card-header">
                <h6 class="mb-0">Campaigns with Ads</h6>
            </div>
            <div class="card-body">
                <?php
                $stmt = $db->prepare("
                    SELECT c.id, c.campaign_name, c.campaign_type, COUNT(ca.id) as ad_count
                    FROM campaigns c
                    LEFT JOIN campaign_ads ca ON c.id = ca.campaign_id AND ca.status = 'active'
                    WHERE c.user_id = ?
                    GROUP BY c.id
                    ORDER BY c.created_at DESC
                    LIMIT 5
                ");
                $stmt->execute([$userId]);
                $campaigns = $stmt->fetchAll();
                ?>
                <?php if (empty($campaigns)): ?>
                    <p class="text-muted">No campaigns yet</p>
                <?php else: ?>
                    <ul class="list-group">
                        <?php foreach ($campaigns as $camp): ?>
                            <li class="list-group-item d-flex justify-content-between">
                                <span><?= escape($camp['campaign_name']) ?> <small class="text-muted">(<?= $camp['ad_count'] ?> ads)</small></span>
                                <a href="<?= url('client/campaigns.php') ?>?view=<?= $camp['id'] ?>" class="btn btn-sm btn-outline-primary">View</a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <div class="col-md-6">
        <div class="card">
            <div class="card-header">
                <h6 class="mb-0">Email Campaigns</h6>
            </div>
            <div class="card-body">
                <?php
                $stmt = $db->prepare("
                    SELECT id, campaign_name, subject_line, status, recipient_count, sent_count
                    FROM email_campaigns
                    WHERE user_id = ?
                    ORDER BY created_at DESC
                    LIMIT 5
                ");
                $stmt->execute([$userId]);
                $emailCampaigns = $stmt->fetchAll();
                ?>
                <?php if (empty($emailCampaigns)): ?>
                    <p class="text-muted">No email campaigns yet</p>
                <?php else: ?>
                    <ul class="list-group">
                        <?php foreach ($emailCampaigns as $ec): ?>
                            <li class="list-group-item d-flex justify-content-between">
                                <span>
                                    <?= escape($ec['campaign_name']) ?>
                                    <small class="text-muted">(<?= $ec['sent_count'] ?>/<?= $ec['recipient_count'] ?> sent)</small>
                                </span>
                                <a href="<?= url('client/email_campaigns.php') ?>?view=<?= $ec['id'] ?>" class="btn btn-sm btn-outline-primary">View</a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Import Campaign Modal (from campaigns.php) -->
<div class="modal fade" id="createCampaignModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create New Campaign</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="<?= url('client/campaigns.php') ?>">
                <div class="modal-body">
                    <input type="hidden" name="action" value="create">
                    <div class="mb-3">
                        <label for="campaign_name" class="form-label">Campaign Name</label>
                        <input type="text" class="form-control" id="campaign_name" name="campaign_name" required>
                    </div>
                    <div class="mb-3">
                        <label for="campaign_type" class="form-label">Campaign Type</label>
                        <select class="form-select" id="campaign_type" name="campaign_type" required>
                            <option value="email" selected>Email Marketing (Recommended)</option>
                            <option value="google_ads">Google Ads</option>
                            <option value="facebook_ads">Facebook Ads</option>
                            <option value="direct_mail">Direct Mail</option>
                            <option value="multi_channel">Multi-Channel</option>
                        </select>
                        <small class="text-muted">Email Marketing is the default campaign type for easier workflow.</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Campaign</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

