<?php
/**
 * Client Settings - API Credentials Management
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/csrf.php';

requireClient();
$pageTitle = 'Settings';

$userId = $_SESSION['user_id'];
$db = getDB();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $serviceName = $_POST['service_name'] ?? '';
    
    if ($action === 'save_credentials') {
        // Simple encryption (in production, use proper encryption library)
        $encryptionKey = getenv('ENCRYPTION_KEY') ?: 'default_key_change_in_production';
        
        $credentialKey = $_POST['credential_key'] ?? 'api_key';
        $credentialValue = $_POST['credential_value'] ?? '';
        
        if (!empty($credentialValue)) {
            // Encrypt the credential value
            $encryptedValue = base64_encode(openssl_encrypt(
                $credentialValue,
                'AES-256-CBC',
                hash('sha256', $encryptionKey),
                0,
                substr(hash('sha256', $encryptionKey), 0, 16)
            ));
            
            // Store credentials as JSON
            $credentialsData = json_encode([
                $credentialKey => $credentialValue
            ]);
            
            $encryptedCredentials = base64_encode(openssl_encrypt(
                $credentialsData,
                'AES-256-CBC',
                hash('sha256', $encryptionKey),
                0,
                substr(hash('sha256', $encryptionKey), 0, 16)
            ));
            
            // Check if credentials already exist
            $stmt = $db->prepare("SELECT id FROM api_credentials WHERE user_id = ? AND service_name = ?");
            $stmt->execute([$userId, $serviceName]);
            $existing = $stmt->fetch();
            
            if ($existing) {
                // Update existing
                $stmt = $db->prepare("
                    UPDATE api_credentials 
                    SET credential_value = ?, credential_key = ?, updated_at = NOW()
                    WHERE user_id = ? AND service_name = ?
                ");
                $stmt->execute([$encryptedCredentials, $credentialKey, $userId, $serviceName]);
            } else {
                // Insert new
                $stmt = $db->prepare("
                    INSERT INTO api_credentials (user_id, service_name, credential_key, credential_value, is_encrypted, status)
                    VALUES (?, ?, ?, ?, 1, 'active')
                ");
                $stmt->execute([$userId, $serviceName, $credentialKey, $encryptedCredentials]);
            }
            
            setFlashMessage('success', ucfirst(str_replace('_', ' ', $serviceName)) . ' credentials saved successfully!');
        } else {
            // Delete credentials if empty
            $stmt = $db->prepare("DELETE FROM api_credentials WHERE user_id = ? AND service_name = ?");
            $stmt->execute([$userId, $serviceName]);
            setFlashMessage('success', ucfirst(str_replace('_', ' ', $serviceName)) . ' credentials removed.');
        }
        
        header('Location: ' . url('client/settings.php'));
        exit;
    } elseif ($action === 'test_connection') {
        $serviceName = $_POST['service_name'] ?? '';
        setFlashMessage('info', 'Connection testing is being implemented. This will validate your API credentials.');
        header('Location: ' . url('client/settings.php'));
        exit;
    } elseif ($action === 'toggle_demo_mode') {
        $demoMode = isset($_POST['demo_mode_enabled']) ? 1 : 0;
        
        // Check if demo_mode_enabled column exists, if not we'll add it via migration
        try {
            $stmt = $db->prepare("ALTER TABLE users ADD COLUMN demo_mode_enabled TINYINT(1) DEFAULT 0");
            $stmt->execute();
        } catch (PDOException $e) {
            // Column might already exist, ignore
        }
        
        $stmt = $db->prepare("UPDATE users SET demo_mode_enabled = ? WHERE id = ?");
        $stmt->execute([$demoMode, $userId]);
        
        setFlashMessage('success', $demoMode ? 'Demo mode enabled. Subscription limits are now bypassed.' : 'Demo mode disabled. Subscription limits are now enforced.');
        header('Location: ' . url('client/settings.php'));
        exit;
    } elseif ($action === 'toggle_show_all_features') {
        $showAll = isset($_POST['show_all_features']) ? 1 : 0;
        
        // Add column if it doesn't exist
        try {
            $stmt = $db->prepare("ALTER TABLE users ADD COLUMN show_all_features TINYINT(1) DEFAULT 1");
            $stmt->execute();
        } catch (PDOException $e) {
            // Column might already exist, ignore
        }
        
        $stmt = $db->prepare("UPDATE users SET show_all_features = ? WHERE id = ?");
        $stmt->execute([$showAll, $userId]);
        
        setFlashMessage('success', $showAll ? 'All features are now visible in navigation.' : 'Navigation menu simplified - only essential features shown.');
        header('Location: ' . url('client/settings.php'));
        exit;
    } elseif ($action === 'save_imap_settings') {
        require_once __DIR__ . '/../includes/csrf.php';
        validateCSRFToken();
        
        // Save IMAP settings to database
        $imapSettings = [
            'IMAP_HOST' => $_POST['imap_host'] ?? '',
            'IMAP_PORT' => $_POST['imap_port'] ?? '993',
            'IMAP_USER' => $_POST['imap_user'] ?? '',
            'IMAP_PASS' => $_POST['imap_pass'] ?? '',
            'IMAP_ENCRYPTION' => $_POST['imap_encryption'] ?? 'ssl',
            'IMAP_SENT_FOLDER' => $_POST['imap_sent_folder'] ?? 'Sent'
        ];
        
        // Store in api_credentials table for this user
        $encryptionKey = getenv('ENCRYPTION_KEY') ?: 'default_key_change_in_production';
        $credentialsJson = json_encode($imapSettings);
        $encryptedCredentials = base64_encode(openssl_encrypt(
            $credentialsJson,
            'AES-256-CBC',
            hash('sha256', $encryptionKey),
            0,
            substr(hash('sha256', $encryptionKey), 0, 16)
        ));
        
        // Check if credentials already exist
        $stmt = $db->prepare("SELECT id FROM api_credentials WHERE user_id = ? AND service_name = ?");
        $stmt->execute([$userId, 'imap']);
        $existing = $stmt->fetch();
        
        if ($existing) {
            // Update existing
            $stmt = $db->prepare("
                UPDATE api_credentials 
                SET credential_value = ?, credential_key = 'settings', updated_at = NOW()
                WHERE user_id = ? AND service_name = ?
            ");
            $stmt->execute([$encryptedCredentials, $userId, 'imap']);
        } else {
            // Insert new
            $stmt = $db->prepare("
                INSERT INTO api_credentials (user_id, service_name, credential_key, credential_value, is_encrypted, status)
                VALUES (?, ?, 'settings', ?, 1, 'active')
            ");
            $stmt->execute([$userId, 'imap', $encryptedCredentials]);
        }
        
        setFlashMessage('success', 'IMAP settings saved successfully!');
        header('Location: ' . url('client/settings.php'));
        exit;
    }
}

// Get user's demo mode status
$userDemoMode = false;
try {
    $stmt = $db->prepare("SELECT demo_mode_enabled FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    $userDemoMode = $user && $user['demo_mode_enabled'] == 1;
} catch (PDOException $e) {
    // Column might not exist yet, default to false
}

// Get user's show_all_features status (for navigation visibility)
$showAllFeatures = true; // Default: show all (full version released)
try {
    $stmt = $db->prepare("SELECT show_all_features FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    $showAllFeatures = $user && isset($user['show_all_features']) ? $user['show_all_features'] == 1 : true;
} catch (PDOException $e) {
    // Column might not exist yet, default to true (show all)
}

// Get all saved credentials
$stmt = $db->prepare("
    SELECT service_name, credential_key, credential_value, status, updated_at
    FROM api_credentials
    WHERE user_id = ? AND status = 'active'
    ORDER BY service_name
");
$stmt->execute([$userId]);
$savedCredentials = $stmt->fetchAll();

// Create array for easy lookup
$credentialsMap = [];
foreach ($savedCredentials as $cred) {
    $credentialsMap[$cred['service_name']] = $cred;
}

// Get IMAP settings if saved
$imapSettings = [];
if (isset($credentialsMap['imap']) && !empty($credentialsMap['imap']['credential_value'])) {
    try {
        $encryptionKey = getenv('ENCRYPTION_KEY') ?: 'default_key_change_in_production';
        $decrypted = openssl_decrypt(
            base64_decode($credentialsMap['imap']['credential_value']),
            'AES-256-CBC',
            hash('sha256', $encryptionKey),
            0,
            substr(hash('sha256', $encryptionKey), 0, 16)
        );
        if ($decrypted) {
            $imapSettings = json_decode($decrypted, true) ?: [];
        }
    } catch (Exception $e) {
        // Ignore decryption errors
        error_log('IMAP settings decryption error: ' . $e->getMessage());
    }
}

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row">
    <div class="col-12 mb-4">
        <h1>Settings</h1>
        <p class="text-muted">Manage your API credentials and account settings</p>
    </div>
</div>

<div class="row">
    <div class="col-12">
        <!-- Navigation Visibility Control -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="bi bi-list"></i> Navigation Menu Settings
                </h5>
            </div>
            <div class="card-body">
                <form method="POST">
                    <input type="hidden" name="action" value="toggle_show_all_features">
                    <div class="form-check form-switch">
                        <input class="form-check-input" type="checkbox" id="show_all_features" name="show_all_features" <?= $showAllFeatures ? 'checked' : '' ?>>
                        <label class="form-check-label" for="show_all_features">
                            <strong>Show All Features in Navigation</strong>
                        </label>
                    </div>
                    <div class="alert alert-info mt-3">
                        <small>
                            <strong>What this does:</strong><br>
                            When <strong>enabled</strong> (checked): Shows all navigation menu items (Ad Builder, Email Marketing, A/B Testing, Webhooks, etc.)<br>
                            When <strong>disabled</strong> (unchecked): Hides advanced features - only shows essential items (Dashboard, Campaigns, Email Sender, Settings, Testing)<br>
                            Use this to simplify the interface until the full version is released.
                        </small>
                    </div>
                    <button type="submit" class="btn btn-primary mt-2">Save Navigation Settings</button>
                </form>
            </div>
        </div>
        
        <!-- Demo Mode Override -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="bi bi-toggle-on"></i> Demo Mode / Testing Settings
                </h5>
            </div>
            <div class="card-body">
                <form method="POST">
                    <input type="hidden" name="action" value="toggle_demo_mode">
                    <div class="form-check form-switch">
                        <input class="form-check-input" type="checkbox" id="demo_mode_enabled" name="demo_mode_enabled" <?= $userDemoMode ? 'checked' : '' ?>>
                        <label class="form-check-label" for="demo_mode_enabled">
                            <strong>Enable Demo Mode (Override Subscription Limits)</strong>
                        </label>
                    </div>
                    <div class="alert alert-info mt-3">
                        <small>
                            <strong>What this does:</strong><br>
                            When enabled, all subscription tier limits (email campaigns, discount codes, contacts, etc.) are bypassed.<br>
                            This is useful for testing and demo purposes. In production, keep this disabled to respect subscription limits.
                        </small>
                    </div>
                    <button type="submit" class="btn btn-primary mt-2">Save Settings</button>
                </form>
            </div>
        </div>
        
        <!-- Google Ads API -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="bi bi-google"></i> Google Ads API Credentials
                    <?php if (isset($credentialsMap['google_ads'])): ?>
                        <span class="badge bg-success ms-2">Configured</span>
                    <?php else: ?>
                        <span class="badge bg-secondary ms-2">Not Configured</span>
                    <?php endif; ?>
                </h5>
            </div>
            <div class="card-body">
                <form method="POST" action="<?= url('client/settings_save.php') ?>">
                    <input type="hidden" name="action" value="save_credentials">
                    <input type="hidden" name="service_name" value="google_ads">
                    
                    <div class="mb-3">
                        <label class="form-label">Client ID</label>
                        <input type="text" class="form-control" name="client_id" 
                               placeholder="Enter your Google Ads Client ID" 
                               value="<?= isset($credentialsMap['google_ads']) ? '••••••••' : '' ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Client Secret</label>
                        <input type="password" class="form-control" name="client_secret" 
                               placeholder="Enter your Google Ads Client Secret">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Developer Token</label>
                        <input type="text" class="form-control" name="developer_token" 
                               placeholder="Enter your Developer Token">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Refresh Token</label>
                        <input type="text" class="form-control" name="refresh_token" 
                               placeholder="Enter your Refresh Token">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Customer ID</label>
                        <input type="text" class="form-control" name="customer_id" 
                               placeholder="Enter your Google Ads Customer ID">
                    </div>
                    
                    <div class="alert alert-info">
                        <small>
                            <strong>How to get Google Ads API credentials:</strong><br>
                            1. Create a Google Ads API account at <a href="https://ads.google.com/" target="_blank">ads.google.com</a><br>
                            2. Apply for a Developer Token in Google Ads<br>
                            3. Create OAuth 2.0 credentials in Google Cloud Console<br>
                            4. Generate a Refresh Token using OAuth 2.0 flow<br>
                            5. Enter your Customer ID (the account you want to manage)
                        </small>
                    </div>
                    
                    <button type="submit" class="btn btn-primary">Save Credentials</button>
                    <?php if (isset($credentialsMap['google_ads'])): ?>
                        <small class="text-muted ms-3">
                            Last updated: <?= formatDate($credentialsMap['google_ads']['updated_at']) ?>
                        </small>
                    <?php endif; ?>
                </form>
            </div>
        </div>
        
        <!-- Facebook Ads API -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="bi bi-facebook"></i> Facebook Marketing API Credentials
                    <?php if (isset($credentialsMap['facebook_ads'])): ?>
                        <span class="badge bg-success ms-2">Configured</span>
                    <?php else: ?>
                        <span class="badge bg-secondary ms-2">Not Configured</span>
                    <?php endif; ?>
                </h5>
            </div>
            <div class="card-body">
                <form method="POST" action="<?= url('client/settings_save.php') ?>">
                    <input type="hidden" name="action" value="save_credentials">
                    <input type="hidden" name="service_name" value="facebook_ads">
                    
                    <div class="mb-3">
                        <label class="form-label">App ID</label>
                        <input type="text" class="form-control" name="app_id" 
                               placeholder="Enter your Facebook App ID">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">App Secret</label>
                        <input type="password" class="form-control" name="app_secret" 
                               placeholder="Enter your Facebook App Secret">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Access Token</label>
                        <input type="text" class="form-control" name="access_token" 
                               placeholder="Enter your Long-Lived Access Token">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Ad Account ID</label>
                        <input type="text" class="form-control" name="ad_account_id" 
                               placeholder="Enter your Facebook Ad Account ID (e.g., act_123456789)">
                    </div>
                    
                    <div class="alert alert-info">
                        <small>
                            <strong>How to get Facebook Marketing API credentials:</strong><br>
                            1. Create a Facebook App at <a href="https://developers.facebook.com/" target="_blank">developers.facebook.com</a><br>
                            2. Get your App ID and App Secret from App Dashboard<br>
                            3. Generate a User Access Token with ads_management permission<br>
                            4. Exchange for Long-Lived Access Token<br>
                            5. Find your Ad Account ID in Facebook Ads Manager
                        </small>
                    </div>
                    
                    <button type="submit" class="btn btn-primary">Save Credentials</button>
                    <?php if (isset($credentialsMap['facebook_ads'])): ?>
                        <small class="text-muted ms-3">
                            Last updated: <?= formatDate($credentialsMap['facebook_ads']['updated_at']) ?>
                        </small>
                    <?php endif; ?>
                </form>
            </div>
        </div>
        
        <!-- Email Service (SendGrid) -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="bi bi-envelope"></i> SendGrid Email Service
                    <?php if (isset($credentialsMap['sendgrid'])): ?>
                        <span class="badge bg-success ms-2">Configured</span>
                    <?php else: ?>
                        <span class="badge bg-secondary ms-2">Not Configured</span>
                    <?php endif; ?>
                </h5>
            </div>
            <div class="card-body">
                <form method="POST" action="<?= url('client/settings_save.php') ?>">
                    <input type="hidden" name="action" value="save_credentials">
                    <input type="hidden" name="service_name" value="sendgrid">
                    
                    <div class="mb-3">
                        <label class="form-label">API Key</label>
                        <input type="password" class="form-control" name="api_key" 
                               placeholder="Enter your SendGrid API Key"
                               value="<?= isset($credentialsMap['sendgrid']) ? '••••••••' : '' ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">From Email</label>
                        <input type="email" class="form-control" name="from_email" 
                               placeholder="your-email@example.com">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">From Name</label>
                        <input type="text" class="form-control" name="from_name" 
                               placeholder="Your Business Name">
                    </div>
                    
                    <div class="alert alert-info">
                        <small>
                            <strong>How to get SendGrid API key:</strong><br>
                            1. Sign up at <a href="https://sendgrid.com/" target="_blank">sendgrid.com</a><br>
                            2. Navigate to Settings > API Keys<br>
                            3. Create a new API Key with "Full Access" or "Mail Send" permissions<br>
                            4. Copy the API key (you won't see it again!)
                        </small>
                    </div>
                    
                    <button type="submit" class="btn btn-primary">Save Credentials</button>
                    <?php if (isset($credentialsMap['sendgrid'])): ?>
                        <small class="text-muted ms-3">
                            Last updated: <?= formatDate($credentialsMap['sendgrid']['updated_at']) ?>
                        </small>
                    <?php endif; ?>
                </form>
            </div>
        </div>
        
        <!-- Mailchimp Alternative -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="bi bi-envelope-heart"></i> Mailchimp Email Service
                    <?php if (isset($credentialsMap['mailchimp'])): ?>
                        <span class="badge bg-success ms-2">Configured</span>
                    <?php else: ?>
                        <span class="badge bg-secondary ms-2">Not Configured</span>
                    <?php endif; ?>
                </h5>
            </div>
            <div class="card-body">
                <form method="POST" action="<?= url('client/settings_save.php') ?>">
                    <input type="hidden" name="action" value="save_credentials">
                    <input type="hidden" name="service_name" value="mailchimp">
                    
                    <div class="mb-3">
                        <label class="form-label">API Key</label>
                        <input type="password" class="form-control" name="api_key" 
                               placeholder="Enter your Mailchimp API Key"
                               value="<?= isset($credentialsMap['mailchimp']) ? '••••••••' : '' ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Server Prefix</label>
                        <input type="text" class="form-control" name="server_prefix" 
                               placeholder="e.g., us1, us2, us3">
                        <small class="text-muted">Found in your Mailchimp account URL or API endpoint</small>
                    </div>
                    
                    <div class="alert alert-info">
                        <small>
                            <strong>How to get Mailchimp API key:</strong><br>
                            1. Login to your Mailchimp account<br>
                            2. Go to Account > Extras > API keys<br>
                            3. Create a new API key<br>
                            4. Copy the API key and server prefix
                        </small>
                    </div>
                    
                    <button type="submit" class="btn btn-primary">Save Credentials</button>
                    <?php if (isset($credentialsMap['mailchimp'])): ?>
                        <small class="text-muted ms-3">
                            Last updated: <?= formatDate($credentialsMap['mailchimp']['updated_at']) ?>
                        </small>
                    <?php endif; ?>
                </form>
            </div>
        </div>
        
        <!-- IMAP Email Settings -->
        <div class="card mb-4">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0">
                    <i class="bi bi-envelope-at"></i> IMAP Email Settings
                    <?php 
                    $imapConfigured = (!empty($imapSettings['IMAP_HOST']) && !empty($imapSettings['IMAP_USER'])) ||
                                      (defined('IMAP_HOST') && !empty(IMAP_HOST) && defined('IMAP_USER') && !empty(IMAP_USER));
                    ?>
                    <?php if ($imapConfigured): ?>
                        <span class="badge bg-success ms-2">Configured</span>
                    <?php else: ?>
                        <span class="badge bg-secondary ms-2">Not Configured</span>
                    <?php endif; ?>
                </h5>
            </div>
            <div class="card-body">
                <form method="POST" id="imapSettingsForm">
                    <input type="hidden" name="action" value="save_imap_settings">
                    <?= csrfField() ?>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">IMAP Host <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="imap_host" 
                                   placeholder="imap.gmail.com or mail.yourdomain.com"
                                   value="<?= !empty($imapSettings['IMAP_HOST']) ? escape($imapSettings['IMAP_HOST']) : (defined('IMAP_HOST') ? escape(IMAP_HOST) : '') ?>" required>
                            <small class="text-muted">IMAP server address</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">IMAP Port</label>
                            <input type="number" class="form-control" name="imap_port" 
                                   placeholder="993"
                                   value="<?= !empty($imapSettings['IMAP_PORT']) ? escape($imapSettings['IMAP_PORT']) : (defined('IMAP_PORT') ? escape(IMAP_PORT) : '993') ?>">
                            <small class="text-muted">Default: 993 (SSL) or 143 (TLS)</small>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">IMAP Username (Email) <span class="text-danger">*</span></label>
                            <input type="email" class="form-control" name="imap_user" 
                                   placeholder="your-email@example.com"
                                   value="<?= !empty($imapSettings['IMAP_USER']) ? escape($imapSettings['IMAP_USER']) : (defined('IMAP_USER') ? escape(IMAP_USER) : '') ?>" required>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">IMAP Password <span class="text-danger">*</span></label>
                            <input type="password" class="form-control" name="imap_pass" 
                                   placeholder="Enter your email password"
                                   value="<?= (!empty($imapSettings['IMAP_PASS']) || (defined('IMAP_PASS') && !empty(IMAP_PASS))) ? '••••••••' : '' ?>" required>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Encryption</label>
                            <select class="form-select" name="imap_encryption">
                                <?php 
                                $currentEncryption = !empty($imapSettings['IMAP_ENCRYPTION']) ? $imapSettings['IMAP_ENCRYPTION'] : (defined('IMAP_ENCRYPTION') ? IMAP_ENCRYPTION : 'ssl');
                                ?>
                                <option value="ssl" <?= $currentEncryption === 'ssl' ? 'selected' : '' ?>>SSL (Port 993)</option>
                                <option value="tls" <?= $currentEncryption === 'tls' ? 'selected' : '' ?>>TLS (Port 143)</option>
                                <option value="" <?= empty($currentEncryption) ? 'selected' : '' ?>>None</option>
                            </select>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Sent Folder Name</label>
                            <input type="text" class="form-control" name="imap_sent_folder" 
                                   placeholder="Sent"
                                   value="<?= !empty($imapSettings['IMAP_SENT_FOLDER']) ? escape($imapSettings['IMAP_SENT_FOLDER']) : (defined('IMAP_SENT_FOLDER') ? escape(IMAP_SENT_FOLDER) : 'Sent') ?>">
                            <small class="text-muted">Where to save sent emails (e.g., "Sent", "Sent Messages")</small>
                        </div>
                    </div>
                    
                    <div class="alert alert-info">
                        <small>
                            <strong>How this works:</strong><br>
                            • Emails are sent via SMTP using your IMAP credentials<br>
                            • The system automatically converts IMAP host to SMTP (e.g., imap.gmail.com → smtp.gmail.com)<br>
                            • Sent emails are saved to your IMAP Sent folder automatically<br>
                            • Use your full email address as the username
                        </small>
                    </div>
                    
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save"></i> Save IMAP Settings
                        </button>
                        <button type="button" class="btn btn-success" id="testImapEmailBtn">
                            <i class="bi bi-envelope-check"></i> Test Email Sender
                        </button>
                    </div>
                </form>
                
                <!-- Test Email Modal -->
                <div class="modal fade" id="testImapEmailModal" tabindex="-1">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title">Test IMAP Email Sender</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <div class="mb-3">
                                    <label class="form-label">Test Email Address</label>
                                    <input type="email" class="form-control" id="testImapEmailAddress" 
                                           placeholder="test@example.com" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Subject</label>
                                    <input type="text" class="form-control" id="testImapEmailSubject" 
                                           value="Test Email from Marketing Platform" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Message</label>
                                    <textarea class="form-control" id="testImapEmailMessage" rows="5" required>This is a test email to verify your IMAP email settings are working correctly.

If you receive this email, your IMAP configuration is successful!</textarea>
                                </div>
                                <div id="testImapEmailResult"></div>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                <button type="button" class="btn btn-success" id="sendTestImapEmailBtn">
                                    <i class="bi bi-send"></i> Send Test Email
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Security Notice -->
        <div class="card">
            <div class="card-header bg-warning">
                <h5 class="mb-0">
                    <i class="bi bi-shield-lock"></i> Security Information
                </h5>
            </div>
            <div class="card-body">
                <div class="alert alert-warning mb-0">
                    <h6>Important Security Notes:</h6>
                    <ul class="mb-0">
                        <li>All API credentials are <strong>encrypted</strong> before storage</li>
                        <li>Never share your API keys with unauthorized parties</li>
                        <li>Rotate your API keys regularly for security</li>
                        <li>If you suspect your keys are compromised, regenerate them immediately</li>
                        <li>Credentials are stored securely and only used for your account's marketing operations</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Test IMAP Email Sender
document.getElementById('testImapEmailBtn')?.addEventListener('click', function() {
    const modal = new bootstrap.Modal(document.getElementById('testImapEmailModal'));
    modal.show();
});

document.getElementById('sendTestImapEmailBtn')?.addEventListener('click', function() {
    const btn = this;
    const originalText = btn.innerHTML;
    const resultDiv = document.getElementById('testImapEmailResult');
    
    const email = document.getElementById('testImapEmailAddress').value;
    const subject = document.getElementById('testImapEmailSubject').value;
    const message = document.getElementById('testImapEmailMessage').value;
    
    if (!email || !subject || !message) {
        resultDiv.innerHTML = '<div class="alert alert-warning">Please fill in all fields.</div>';
        return;
    }
    
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Sending...';
    resultDiv.innerHTML = '';
    
    // Get IMAP settings from form
    const formData = new FormData(document.getElementById('imapSettingsForm'));
    formData.append('action', 'test_imap_email');
    formData.append('test_email', email);
    formData.append('test_subject', subject);
    formData.append('test_message', message);
    
    fetch('<?= url("api/email_service.php") ?>', {
        method: 'POST',
        body: formData
    })
    .then(async r => {
        const text = await r.text();
        if (!r.ok) {
            throw new Error(`HTTP ${r.status}: ${text}`);
        }
        return JSON.parse(text);
    })
    .then(data => {
        if (data.success) {
            resultDiv.innerHTML = '<div class="alert alert-success"><i class="bi bi-check-circle"></i> Test email sent successfully! Check your inbox.</div>';
        } else {
            resultDiv.innerHTML = '<div class="alert alert-danger"><i class="bi bi-x-circle"></i> Error: ' + (data.error || 'Failed to send test email') + '</div>';
        }
    })
    .catch(err => {
        resultDiv.innerHTML = '<div class="alert alert-danger"><i class="bi bi-x-circle"></i> Error: ' + err.message + '</div>';
    })
    .finally(() => {
        btn.disabled = false;
        btn.innerHTML = originalText;
    });
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

