<?php
/**
 * Client Reports Page
 * Exportable reports with custom date ranges
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireClient();
$pageTitle = 'Reports';

$userId = $_SESSION['user_id'];
$db = getDB();

// Get parametric date range
$startDate = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$endDate = $_GET['end_date'] ?? date('Y-m-d');
$format = $_GET['format'] ?? 'html'; // html, csv, pdf

// Get metrics for date range
$stmt = $db->prepare("
    SELECT 
        COUNT(*) as total_visits,
        COUNT(DISTINCT session_id) as unique_visitors,
        COUNT(CASE WHEN conversion_type IS NOT NULL THEN 1 END) as conversions,
        SUM(conversion_value) as total_revenue
    FROM user_metrics
    WHERE user_id = ? AND DATE(visited_at) BETWEEN ? AND ?
");
$stmt->execute([$userId, $startDate, $endDate]);
$metrics = $stmt->fetch();

// Export to CSV
if ($format === 'csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="report_' . $startDate . '_' . $endDate . '.csv"');
    
    $output = fopen('php://output', 'w');
    fputcsv($output, ['Metric', 'Value']);
    fputcsv($output, ['Total Visits', $metrics['total_visits']]);
    fputcsv($output, ['Unique Visitors', $metrics['unique_visitors']]);
    fputcsv($output, ['Conversions', $metrics['conversions']]);
    fputcsv($output, ['Total Revenue', $metrics['total_revenue'] ?? 0]);
    fclose($output);
    exit;
}

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row">
    <div class="col-12 mb-4">
        <div class="d-flex justify-content-between align-items-center">
            <h1>Reports</h1>
            <div>
                <a href="?start_date=<?= $startDate ?>&end_date=<?= $endDate ?>&format=csv" class="btn btn-success">
                    <i class="bi bi-download"></i> Export CSV
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Date Range Filter -->
<div class="row mb-4">
    <div class="col-md-6">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Date Range</h5>
            </div>
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-5">
                        <label class="form-label">Start Date</label>
                        <input type="date" class="form-control" name="start_date" value="<?= $startDate ?>" required>
                    </div>
                    <div class="col-md-5">
                        <label class="form-label">End Date</label>
                        <input type="date" class="form-control" name="end_date" value="<?= $endDate ?>" required>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">&nbsp;</label>
                        <button type="submit" class="btn btn-primary w-100">Filter</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Report Summary -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Report Summary (<?= formatDate($startDate) ?> - <?= formatDate($endDate) ?>)</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <h4><?= number_format($metrics['total_visits']) ?></h4>
                        <p class="text-muted">Total Visits</p>
                    </div>
                    <div class="col-md-3">
                        <h4><?= number_format($metrics['unique_visitors']) ?></h4>
                        <p class="text-muted">Unique Visitors</p>
                    </div>
                    <div class="col-md-3">
                        <h4><?= number_format($metrics['conversions']) ?></h4>
                        <p class="text-muted">Conversions</p>
                    </div>
                    <div class="col-md-3">
                        <h4><?= formatCurrency($metrics['total_revenue'] ?? 0) ?></h4>
                        <p class="text-muted">Total Revenue</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

