<?php
/**
 * Email Campaigns Dashboard
 * Complete email marketing management: campaigns, templates, scheduling, segmentation
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireClient();
$pageTitle = 'Email Campaigns';

$userId = $_SESSION['user_id'];
$db = getDB();

// Get email campaigns
$stmt = $db->prepare("
    SELECT ec.*, c.campaign_name as parent_campaign_name
    FROM email_campaigns ec
    LEFT JOIN campaigns c ON ec.campaign_id = c.id
    WHERE ec.user_id = ?
    ORDER BY ec.created_at DESC
");
$stmt->execute([$userId]);
$emailCampaigns = $stmt->fetchAll();

// Get email templates
$stmt = $db->prepare("
    SELECT * FROM email_templates 
    WHERE (user_id = ? OR is_public = 1)
    ORDER BY is_public DESC, created_at DESC
    LIMIT 10
");
$stmt->execute([$userId]);
$templates = $stmt->fetchAll();

// Get segments
$stmt = $db->prepare("SELECT * FROM email_segments WHERE user_id = ? ORDER BY created_at DESC");
$stmt->execute([$userId]);
$segments = $stmt->fetchAll();

// Get subscribers count
$stmt = $db->prepare("SELECT COUNT(*) FROM email_subscribers WHERE user_id = ? AND status = 'subscribed'");
$stmt->execute([$userId]);
$subscriberCount = $stmt->fetchColumn();

// Get scheduled campaigns
$stmt = $db->prepare("
    SELECT COUNT(*) FROM email_campaigns 
    WHERE user_id = ? AND status IN ('scheduled', 'sending')
");
$stmt->execute([$userId]);
$scheduledCount = $stmt->fetchColumn();

// Get available campaigns for ad insertion
$stmt = $db->prepare("
    SELECT DISTINCT c.id, c.campaign_name
    FROM campaigns c
    INNER JOIN campaign_ads ca ON c.id = ca.campaign_id
    WHERE c.user_id = ? AND ca.status = 'active'
    ORDER BY c.campaign_name
");
$stmt->execute([$userId]);
$campaignsWithAds = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/../includes/csrf.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <h1><i class="bi bi-envelope"></i> Email Campaigns</h1>
            <div>
                <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#createEmailCampaignModal">
                    <i class="bi bi-plus-circle"></i> Create Email Campaign
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Quick Stats -->
<div class="row mb-4">
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-primary">
            <div class="card-body">
                <h6 class="card-title">Total Campaigns</h6>
                <h3 class="mb-0"><?= count($emailCampaigns) ?></h3>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-info">
            <div class="card-body">
                <h6 class="card-title">Scheduled</h6>
                <h3 class="mb-0"><?= $scheduledCount ?></h3>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-success">
            <div class="card-body">
                <h6 class="card-title">Subscribers</h6>
                <h3 class="mb-0"><?= $subscriberCount ?></h3>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-warning">
            <div class="card-body">
                <h6 class="card-title">Segments</h6>
                <h3 class="mb-0"><?= count($segments) ?></h3>
            </div>
        </div>
    </div>
</div>

<!-- Tabs -->
<ul class="nav nav-tabs mb-4" id="emailTabs" role="tablist">
    <li class="nav-item" role="presentation">
        <button class="nav-link active" id="campaigns-tab" data-bs-toggle="tab" data-bs-target="#campaigns" type="button">
            <i class="bi bi-envelope-paper"></i> Campaigns
        </button>
    </li>
    <li class="nav-item" role="presentation">
        <button class="nav-link" id="templates-tab" data-bs-toggle="tab" data-bs-target="#templates" type="button">
            <i class="bi bi-file-earmark"></i> Templates
        </button>
    </li>
    <li class="nav-item" role="presentation">
        <button class="nav-link" id="segments-tab" data-bs-toggle="tab" data-bs-target="#segments" type="button">
            <i class="bi bi-people"></i> Segments
        </button>
    </li>
    <li class="nav-item" role="presentation">
        <button class="nav-link" id="subscribers-tab" data-bs-toggle="tab" data-bs-target="#subscribers" type="button">
            <i class="bi bi-person-plus"></i> Subscribers
        </button>
    </li>
</ul>

<div class="tab-content" id="emailTabsContent">
    <!-- Campaigns Tab -->
    <div class="tab-pane fade show active" id="campaigns" role="tabpanel">
        <div class="card">
            <div class="card-body">
                <?php if (empty($emailCampaigns)): ?>
                    <p class="text-muted text-center py-5">No email campaigns yet. Create your first one!</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Campaign Name</th>
                                    <th>Subject</th>
                                    <th>Recipients</th>
                                    <th>Sent</th>
                                    <th>Opened</th>
                                    <th>Clicked</th>
                                    <th>Status</th>
                                    <th>Scheduled</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($emailCampaigns as $campaign): ?>
                                    <tr>
                                        <td><strong><?= escape($campaign['campaign_name']) ?></strong></td>
                                        <td><?= escape($campaign['subject_line']) ?></td>
                                        <td><?= number_format($campaign['recipient_count']) ?></td>
                                        <td><?= number_format($campaign['sent_count']) ?></td>
                                        <td>
                                            <?= number_format($campaign['opened_count']) ?>
                                            <?php if ($campaign['sent_count'] > 0): ?>
                                                <small class="text-muted">(<?= number_format(($campaign['opened_count'] / $campaign['sent_count']) * 100, 1) ?>%)</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?= number_format($campaign['clicked_count']) ?>
                                            <?php if ($campaign['sent_count'] > 0): ?>
                                                <small class="text-muted">(<?= number_format(($campaign['clicked_count'] / $campaign['sent_count']) * 100, 1) ?>%)</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?= 
                                                $campaign['status'] === 'sent' ? 'success' : 
                                                ($campaign['status'] === 'scheduled' ? 'info' : 
                                                ($campaign['status'] === 'sending' ? 'warning' : 'secondary')) 
                                            ?>">
                                                <?= escape($campaign['status']) ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($campaign['scheduled_at']): ?>
                                                <?= formatDateTime($campaign['scheduled_at']) ?>
                                            <?php else: ?>
                                                <span class="text-muted">Not scheduled</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <button class="btn btn-secondary import-btn" data-id="<?= $campaign['id'] ?>" data-name="<?= escape($campaign['campaign_name']) ?>">
                                                    <i class="bi bi-file-earmark-arrow-up"></i> Import CSV
                                                </button>
                                                <?php if ($campaign['status'] === 'draft'): ?>
                                                    <button class="btn btn-primary btn-sm schedule-btn" data-id="<?= $campaign['id'] ?>">
                                                        <i class="bi bi-clock"></i> Schedule
                                                    </button>
                                                <?php endif; ?>
                                                <a href="?view=<?= $campaign['id'] ?>" class="btn btn-info btn-sm">
                                                    <i class="bi bi-eye"></i> View
                                                </a>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Templates Tab -->
    <div class="tab-pane fade" id="templates" role="tabpanel">
        <div class="card">
            <div class="card-header d-flex justify-content-between">
                <h5 class="mb-0">Email Templates</h5>
                <button class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#createTemplateModal">
                    <i class="bi bi-plus"></i> Create Template
                </button>
            </div>
            <div class="card-body">
                <?php if (empty($templates)): ?>
                    <p class="text-muted">No templates yet. Create your first template!</p>
                <?php else: ?>
                    <div class="row">
                        <?php foreach ($templates as $template): ?>
                            <div class="col-md-4 mb-3">
                                <div class="card">
                                    <div class="card-body">
                                        <h6><?= escape($template['template_name']) ?></h6>
                                        <?php if ($template['subject_line']): ?>
                                            <p class="small text-muted"><?= escape($template['subject_line']) ?></p>
                                        <?php endif; ?>
                                        <?php if ($template['is_public']): ?>
                                            <span class="badge bg-success">Public</span>
                                        <?php endif; ?>
                                        <div class="mt-2">
                                            <button class="btn btn-sm btn-primary use-template-btn" data-id="<?= $template['id'] ?>">
                                                Use Template
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Segments Tab -->
    <div class="tab-pane fade" id="segments" role="tabpanel">
        <div class="card">
            <div class="card-header d-flex justify-content-between">
                <h5 class="mb-0">Subscriber Segments</h5>
                <button class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#createSegmentModal">
                    <i class="bi bi-plus"></i> Create Segment
                </button>
            </div>
            <div class="card-body">
                <?php if (empty($segments)): ?>
                    <p class="text-muted">No segments yet. Create your first segment!</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Segment Name</th>
                                    <th>Subscribers</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($segments as $segment): ?>
                                    <tr>
                                        <td><strong><?= escape($segment['segment_name']) ?></strong></td>
                                        <td><?= number_format($segment['subscriber_count']) ?></td>
                                        <td><?= formatDateTime($segment['created_at']) ?></td>
                                        <td>
                                            <button class="btn btn-sm btn-primary use-segment-btn" data-id="<?= $segment['id'] ?>">
                                                Use Segment
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Subscribers Tab -->
    <div class="tab-pane fade" id="subscribers" role="tabpanel">
        <div class="card">
            <div class="card-header d-flex justify-content-between">
                <h5 class="mb-0">Email Subscribers</h5>
                <button class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#addSubscriberModal">
                    <i class="bi bi-person-plus"></i> Add Subscriber
                </button>
            </div>
            <div class="card-body">
                <p>Manage your email subscribers. Total: <strong><?= $subscriberCount ?></strong></p>
                <a href="#" class="btn btn-primary">Import Subscribers</a>
                <a href="#" class="btn btn-secondary">Export List</a>
            </div>
        </div>
    </div>
</div>

<!-- Include modals and JS will be in a separate section -->
<?php require_once __DIR__ . '/../includes/footer.php'; ?>

<script>
// Email campaign scheduling and template usage functionality
document.addEventListener('DOMContentLoaded', function() {
    // Schedule campaign
    document.querySelectorAll('.schedule-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const campaignId = this.dataset.id;
            // Open scheduling modal
            // Implementation will be in modal section
        });
    });
    
    // Use template
    document.querySelectorAll('.use-template-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const templateId = this.dataset.id;
            // Load template into campaign creation form
            fetch(`<?= url('api/email_templates.php') ?>?action=get&id=${templateId}`)
                .then(r => r.json())
                .then(data => {
                    if (data.success) {
                        // Populate form with template data
                        // Implementation
                    }
                });
        });
    });

    // CSV Import modal + upload
    const importModalEl = document.getElementById('importRecipientsModal');
    if (importModalEl) {
        const importCampaignNameEl = document.getElementById('importCampaignName');
        const importCampaignIdEl = document.getElementById('importCampaignId');
        const importFormEl = document.getElementById('importRecipientsForm');
        const importResultEl = document.getElementById('importResult');

        document.querySelectorAll('.import-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                const id = this.dataset.id;
                const name = this.dataset.name || 'Campaign';
                importCampaignIdEl.value = id;
                importCampaignNameEl.textContent = name;
                importFormEl.reset();
                importResultEl.textContent = '';
                const modal = new bootstrap.Modal(importModalEl);
                modal.show();
            });
        });

        importFormEl.addEventListener('submit', async function(e) {
            e.preventDefault();
            importResultEl.textContent = 'Uploading...';
            const formData = new FormData(importFormEl);
            try {
                const res = await fetch('<?= url('api/email_campaign_import.php') ?>', {
                    method: 'POST',
                    body: formData
                });
                const data = await res.json();
                if (data.success) {
                    importResultEl.textContent = `Imported ${data.inserted} recipients, ${data.invalid} invalid.`;
                    setTimeout(() => { window.location.reload(); }, 1200);
                } else {
                    importResultEl.textContent = data.message || 'Import failed';
                }
            } catch (err) {
                importResultEl.textContent = 'Network error';
            }
        });
    }
    
    // Ad insertion and email testing functions
    window.insertAdToEmail = function() {
        showCampaignAdSelector('email_content');
    };
    
    window.insertAdToTemplate = function() {
        showCampaignAdSelector('template_content');
    };
    
    function toggleRecipientInputs() {
        const source = document.querySelector('input[name="recipient_source"]:checked').value;
        if (source === 'crm') {
            document.getElementById('manual_recipients_div').style.display = 'none';
            document.getElementById('crm_recipients_div').style.display = 'block';
            document.getElementById('recipients').removeAttribute('required');
            document.getElementById('daily_limit').setAttribute('required', 'required');
        } else {
            document.getElementById('manual_recipients_div').style.display = 'block';
            document.getElementById('crm_recipients_div').style.display = 'none';
            document.getElementById('recipients').setAttribute('required', 'required');
            document.getElementById('daily_limit').removeAttribute('required');
        }
    }
    
    function showCampaignAdSelector(targetTextarea) {
        const campaigns = <?= json_encode($campaignsWithAds) ?>;
        if (campaigns.length === 0) {
            alert('No campaigns with ads found. Create ads first using the Ad Builder.');
            return;
        }
        
        // Show campaign selection modal
        let modalHtml = '<div class="modal fade" id="selectCampaignModal" tabindex="-1"><div class="modal-dialog"><div class="modal-content">';
        modalHtml += '<div class="modal-header"><h5 class="modal-title">Select Campaign with Ads</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>';
        modalHtml += '<div class="modal-body">';
        modalHtml += '<div class="list-group">';
        
        campaigns.forEach(campaign => {
            modalHtml += `<button type="button" class="list-group-item list-group-item-action" onclick="loadCampaignAds(${campaign.id}, '${targetTextarea}')">${escapeHtml(campaign.campaign_name)}</button>`;
        });
        
        modalHtml += '</div></div></div></div></div>';
        
        // Remove existing modal if any
        const existing = document.getElementById('selectCampaignModal');
        if (existing) existing.remove();
        
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        const modal = new bootstrap.Modal(document.getElementById('selectCampaignModal'));
        modal.show();
        
        // Clean up on hide
        document.getElementById('selectCampaignModal').addEventListener('hidden.bs.modal', function() {
            this.remove();
        });
    }
    
    window.loadCampaignAds = function(campaignId, targetTextarea) {
        // Close campaign selection modal
        const campaignModal = document.getElementById('selectCampaignModal');
        if (campaignModal) {
            bootstrap.Modal.getInstance(campaignModal).hide();
        }
        
        fetch(`<?= url("api/ad_builder.php") ?>?action=list_campaign_ads&campaign_id=${campaignId}`)
            .then(async r => {
                const text = await r.text();
                if (!r.ok) {
                    throw new Error(`HTTP ${r.status}: ${text}`);
                }
                return JSON.parse(text);
            })
            .then(data => {
                if (data.success && data.data && data.data.length > 0) {
                    // Validate that ads have real content before showing modal
                    const validAds = data.data.filter(ad => {
                        const adData = ad.ad_data || {};
                        const hasContent = adData.headline || adData.description || adData.productImage || adData.cta;
                        if (!hasContent) {
                            console.warn('Skipping ad with no content:', ad);
                        }
                        return hasContent;
                    });
                    
                    if (validAds.length === 0) {
                        alert('No ads with content found for this campaign. Please ensure ads have been saved with headline, description, or image.');
                        return;
                    }
                    
                    // Log ads for debugging
                    console.log('Loaded ads for campaign:', validAds.map(ad => ({
                        id: ad.id,
                        name: ad.ad_name,
                        headline: ad.ad_data?.headline || 'empty',
                        hasContent: !!(ad.ad_data?.headline || ad.ad_data?.description || ad.ad_data?.productImage)
                    })));
                    
                    showAdSelectionModal(validAds, targetTextarea || 'email_content');
                } else {
                    alert('No ads found for this campaign');
                }
            })
            .catch(err => {
                alert('Error loading ads: ' + err.message);
            });
    };
    
    function showAdSelectionModal(ads, targetTextareaId) {
        // Store ads with a unique identifier to avoid closure issues
        const modalId = 'selectAdModal_' + Date.now();
        window['selectedAds_' + modalId] = ads;
        
        let modalHtml = '<div class="modal fade" id="' + modalId + '" tabindex="-1"><div class="modal-dialog modal-lg"><div class="modal-content">';
        modalHtml += '<div class="modal-header"><h5 class="modal-title">Select Ad to Insert</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>';
        modalHtml += '<div class="modal-body"><div class="row">';
        
        ads.forEach((ad, index) => {
            const adData = ad.ad_data || {};
            const campaignAdId = ad.id || ad.campaign_ad_id || 'unknown';
            
            // Check if this is mock/template data
            const mockIndicators = ['Shop the Latest Deals', 'Discover amazing products', 'Limited time offer', 
                                    'Your Headline Here', 'Your description goes here', 'Fresh & Delicious',
                                    'Feel Beautiful Inside & Out', 'Style That Speaks', 'Expert Service You Can Trust'];
            const isMockData = mockIndicators.some(mock => 
                (adData.headline && adData.headline.includes(mock)) || 
                (adData.description && adData.description.includes(mock))
            );
            
            modalHtml += '<div class="col-md-6 mb-3">';
            modalHtml += '<div class="card' + (isMockData ? ' border-warning' : '') + '">';
            modalHtml += '<div class="card-body">';
            modalHtml += `<h6>${escapeHtml(ad.ad_name)}</h6>`;
            modalHtml += `<small class="text-muted">${ad.ad_platform || 'N/A'} (ID: ${campaignAdId})</small>`;
            
            if (isMockData) {
                modalHtml += '<div class="alert alert-warning mt-2 mb-2 py-1 px-2"><small><strong>⚠️ Template Data:</strong> This ad contains template/mock content. Edit in Ad Builder before using.</small></div>';
            }
            
            // Show ad preview with actual data
            if (adData.headline || adData.productImage || adData.description) {
                modalHtml += '<div class="mt-2 p-2" style="border: 1px solid #ddd; border-radius: 4px; min-height: 100px; background: ' + (adData.bgColor || '#ffffff') + '; text-align: center;">';
                if (adData.productImage) {
                    modalHtml += `<img src="${escapeHtml(adData.productImage)}" style="max-width: 100px; height: auto;" onerror="this.style.display='none'">`;
                }
                if (adData.headline) {
                    modalHtml += `<p class="mb-0 mt-2"><strong>${escapeHtml(adData.headline)}</strong></p>`;
                }
                if (adData.description) {
                    const descShort = adData.description.length > 50 ? adData.description.substring(0, 50) + '...' : adData.description;
                    modalHtml += `<p class="mb-0 mt-1 small">${escapeHtml(descShort)}</p>`;
                }
                modalHtml += '</div>';
            } else {
                modalHtml += '<div class="alert alert-danger mt-2 mb-2 py-1 px-2"><small><strong>⚠️ No Content:</strong> This ad has no content. Edit in Ad Builder first.</small></div>';
            }
            
            modalHtml += '<div class="mt-2">';
            // Store ad index in data attribute and use modal ID to get correct ads array
            const buttonClass = isMockData ? 'btn btn-sm btn-warning w-100 insert-ad-btn' : 'btn btn-sm btn-primary w-100 insert-ad-btn';
            const buttonText = isMockData ? '⚠️ Insert (Has Template Data)' : 'Insert This Ad';
            modalHtml += `<button class="${buttonClass}" data-ad-index="${index}" data-modal-id="${modalId}" data-textarea-id="${targetTextareaId}">${buttonText}</button>`;
            modalHtml += '</div>';
            modalHtml += '</div></div></div>';
        });
        
        modalHtml += '</div></div></div></div></div>';
        
        // Remove existing modal if any
        document.querySelectorAll('[id^="selectAdModal_"]').forEach(el => el.remove());
        
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        const modal = new bootstrap.Modal(document.getElementById(modalId));
        modal.show();
        
        // Add event listeners using event delegation to avoid closure issues
        document.getElementById(modalId).addEventListener('click', function(e) {
            const btn = e.target.closest('.insert-ad-btn');
            if (btn) {
                const adIndex = parseInt(btn.dataset.adIndex);
                const modalId = btn.dataset.modalId;
                const textareaId = btn.dataset.textareaId;
                const ads = window['selectedAds_' + modalId];
                
                if (ads && ads[adIndex]) {
                    insertAdHTMLDirect(ads[adIndex], textareaId);
                    bootstrap.Modal.getInstance(document.getElementById(modalId)).hide();
                } else {
                    console.error('Ad not found:', { adIndex, modalId, adsLength: ads ? ads.length : 0 });
                    alert('Error: Could not find the selected ad. Please try again.');
                }
            }
        });
        
        // Clean up on hide
        document.getElementById(modalId).addEventListener('hidden.bs.modal', function() {
            delete window['selectedAds_' + modalId];
            this.remove();
        });
    }
    
    // Direct insertion function that receives the ad object directly
    function insertAdHTMLDirect(ad, textareaId) {
        // Validate that we have ad data from saved_ads
        if (!ad || !ad.ad_data || typeof ad.ad_data !== 'object') {
            alert('Error: Ad data not found. Please ensure the ad has been saved properly.');
            console.error('Ad data missing:', ad);
            return;
        }
        
        const adData = ad.ad_data; // Use actual saved ad data, no fallback to empty object
        
        // Check for mock/placeholder content and warn user
        const mockIndicators = ['Shop the Latest Deals', 'Discover amazing products', 'Limited time offer', 
                                'Your Headline Here', 'Your description goes here', 'Fresh & Delicious',
                                'Feel Beautiful Inside & Out', 'Style That Speaks', 'Expert Service You Can Trust'];
        const headlineIsMock = mockIndicators.some(mock => adData.headline && adData.headline.includes(mock));
        const descIsMock = mockIndicators.some(mock => adData.description && adData.description.includes(mock));
        
        if (headlineIsMock || descIsMock) {
            const proceed = confirm('Warning: This ad appears to contain template/mock data. If you want to use your own content, please edit the ad in Ad Builder first. Continue inserting anyway?');
            if (!proceed) {
                return;
            }
        }
        
        // Validate that ad has actual content (not empty)
        const hasContent = adData.headline || adData.description || adData.productImage || adData.cta;
        if (!hasContent) {
            const confirmInsert = confirm('Warning: This ad appears to have no content (no headline, description, image, or CTA). Do you want to insert it anyway?');
            if (!confirmInsert) {
                return;
            }
        }
        
        // Log ad data for debugging - show actual values
        console.log('Inserting ad with data:', {
            ad_name: ad.ad_name,
            campaign_ad_id: ad.id || ad.campaign_ad_id,
            saved_ad_id: ad.saved_ad_id,
            headline: adData.headline || 'EMPTY',
            description: adData.description || 'EMPTY',
            cta: adData.cta || 'EMPTY',
            destination_url: adData.destinationUrl || 'EMPTY',
            productImage: adData.productImage || 'EMPTY',
            has_headline: !!adData.headline,
            has_description: !!adData.description,
            has_image: !!adData.productImage,
            has_cta: !!adData.cta
        });
        
        // Warn if we're inserting empty or mock data
        if (!adData.headline && !adData.description && !adData.productImage) {
            console.error('CRITICAL: Attempting to insert ad with NO CONTENT:', ad);
            alert('Error: This ad has no content. Please edit the ad in Ad Builder and save it with your content before inserting.');
            return;
        }
        const textarea = document.getElementById(textareaId);
        
        if (!textarea) {
            alert('Textarea not found');
            return;
        }
        
        // Get campaign_ad_id (ad.id from campaign_ads table)
        const campaignAdId = ad.id || ad.campaign_ad_id || null;
        
        // Get destination URL from saved ad data
        const destinationUrl = adData.destinationUrl || '#';
        
        // Generate acceleratedadz.php URL with campaign_ad_id for tracking
        let ctaUrl = '#';
        if (campaignAdId) {
            const acceleratedadzBase = '<?= url("acceleratedadz.php") ?>';
            ctaUrl = acceleratedadzBase + '?campaign_ad=' + campaignAdId + '&source=campaign&redirect=' + encodeURIComponent(destinationUrl);
        } else if (destinationUrl !== '#') {
            ctaUrl = destinationUrl;
        }
        
        // Generate email-compatible HTML using ONLY the saved ad data (no mock/placeholder data)
        let adHTML = '\n<!-- Ad from saved_ads: ' + escapeHtml(ad.ad_name) + ' (Campaign Ad ID: ' + (campaignAdId || 'N/A') + ', Saved Ad ID: ' + (ad.saved_ad_id || 'N/A') + ') -->\n';
        adHTML += '<table cellpadding="0" cellspacing="0" border="0" width="100%" style="max-width: 600px; margin: 20px auto;">\n';
        adHTML += '<tr><td style="padding: 20px; background: ' + (adData.bgColor || '#ffffff') + ';';
        if (adData.bgImage) {
            adHTML += ' background-image: url(\'' + escapeHtml(adData.bgImage) + '\'); background-size: cover;';
        }
        adHTML += ' text-align: center;">\n';
        
        if (adData.productImage) {
            adHTML += '<img src="' + escapeHtml(adData.productImage) + '" alt="' + escapeHtml(adData.headline || ad.ad_name || 'Ad') + '" style="max-width: 300px; height: auto; display: block; margin: 0 auto 15px;">\n';
        }
        
        // ONLY use actual ad data - never insert mock/placeholder data
        // Check if headline exists and is not empty
        if (adData.headline && adData.headline.trim()) {
            adHTML += '<h2 style="color: ' + (adData.textColor || '#000000') + '; margin: 10px 0;">' + escapeHtml(adData.headline.trim()) + '</h2>\n';
        } else {
            console.warn('Ad headline is missing or empty - skipping headline in HTML');
        }
        
        // Check if description exists and is not empty
        if (adData.description && adData.description.trim()) {
            adHTML += '<p style="color: ' + (adData.textColor || '#000000') + '; margin: 10px 0;">' + escapeHtml(adData.description.trim()) + '</p>\n';
        } else {
            console.warn('Ad description is missing or empty - skipping description in HTML');
        }
        
        // Check if CTA exists and is not empty
        if (adData.cta && adData.cta.trim()) {
            adHTML += '<a href="' + escapeHtml(ctaUrl) + '" style="display: inline-block; background: ' + (adData.ctaColor || '#007bff') + '; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin-top: 15px;">' + escapeHtml(adData.cta.trim()) + '</a>\n';
        } else {
            console.warn('Ad CTA is missing or empty - skipping CTA button in HTML');
        }
        
        adHTML += '</td></tr>\n';
        adHTML += '</table>\n';
        adHTML += '<!-- End Ad -->\n';
        
        // Insert at cursor position
        const start = textarea.selectionStart || 0;
        const end = textarea.selectionEnd || 0;
        const text = textarea.value;
        const before = text.substring(0, start);
        const after = text.substring(end, text.length);
        textarea.value = before + adHTML + after;
        
        // Set cursor position after inserted ad
        textarea.selectionStart = textarea.selectionEnd = start + adHTML.length;
        textarea.focus();
    }
    
    // Legacy function kept for backwards compatibility, but now uses the new direct function
    window.insertAdHTML = function(adIndex, textareaId) {
        console.warn('Legacy insertAdHTML called - this should not happen with new modal system');
        if (window.selectedAds && window.selectedAds[adIndex]) {
            insertAdHTMLDirect(window.selectedAds[adIndex], textareaId);
        } else {
            alert('Error: Ad selection data not available. Please try selecting the ad again.');
        }
    };
    
    window.previewEmailWithAds = function() {
        const content = document.getElementById('email_content')?.value || '';
        if (!content) {
            alert('Please enter email content first');
            return;
        }
        
        // Extract campaign_ad IDs from content to verify they're real ads
        const adComments = content.match(/<!-- Ad from saved_ads:.*?Campaign Ad ID: (\d+).*?-->/g) || [];
        const campaignAdIds = adComments.map(comment => {
            const match = comment.match(/Campaign Ad ID: (\d+)/);
            return match ? parseInt(match[1]) : null;
        }).filter(id => id !== null);
        
        // Validate that ads exist and log details
        if (campaignAdIds.length > 0) {
            console.log('Preview contains ads with IDs:', campaignAdIds);
            
            // Check for any mock/placeholder text in the content
            const mockIndicators = ['Shop the Latest Deals', 'Discover amazing products', 'Limited time offer', 'Placeholder', 'Sample', 'Mock'];
            const hasMockData = mockIndicators.some(indicator => content.includes(indicator));
            if (hasMockData) {
                console.warn('⚠️ Preview content contains possible mock/placeholder text. Verify ad data is correct.');
            }
        }
        
        // Log a snippet of the content for debugging
        const contentSnippet = content.substring(0, 200);
        console.log('Preview content snippet:', contentSnippet);
        
        // Open preview in new window with proper email styling
        const previewWindow = window.open('', '_blank', 'width=800,height=800,scrollbars=yes');
        previewWindow.document.write('<!DOCTYPE html>');
        previewWindow.document.write('<html><head>');
        previewWindow.document.write('<title>Email Preview</title>');
        previewWindow.document.write('<meta charset="UTF-8">');
        previewWindow.document.write('<meta name="viewport" content="width=device-width, initial-scale=1.0">');
        previewWindow.document.write('<style>');
        previewWindow.document.write('body { font-family: Arial, sans-serif; padding: 20px; margin: 0; background: #f5f5f5; }');
        previewWindow.document.write('.preview-container { max-width: 600px; margin: 0 auto; background: white; padding: 20px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }');
        previewWindow.document.write('.preview-header { border-bottom: 2px solid #ddd; padding-bottom: 10px; margin-bottom: 20px; }');
        previewWindow.document.write('.preview-header h3 { margin: 0; color: #333; }');
        previewWindow.document.write('</style>');
        previewWindow.document.write('</head><body>');
        previewWindow.document.write('<div class="preview-container">');
        previewWindow.document.write('<div class="preview-header"><h3>📧 Email Preview</h3></div>');
        previewWindow.document.write('<div class="email-content">');
        previewWindow.document.write(content);
        previewWindow.document.write('</div>');
        previewWindow.document.write('</div></body></html>');
        previewWindow.document.close();
    };
    
    window.testEmailWithAds = function() {
        const content = document.getElementById('email_content')?.value || '';
        const subject = document.getElementById('subject_line')?.value || 'Test Email';
        const testEmail = prompt('Enter your test email address:');
        
        if (!testEmail || !testEmail.includes('@')) {
            alert('Please enter a valid email address');
            return;
        }
        
        if (!content) {
            alert('Please enter email content first');
            return;
        }
        
        const formData = new FormData();
        formData.append('action', 'send_test_email');
        formData.append('to', testEmail);
        formData.append('subject', subject);
        formData.append('html_content', content);
        formData.append('csrf_token', document.querySelector('input[name="csrf_token"]')?.value || '<?= generateCSRFToken() ?>');
        
        // Show loading
        const loadingMsg = alert('Sending test email to ' + testEmail + '...');
        
        fetch('<?= url("api/email_service.php") ?>', {
            method: 'POST',
            body: formData
        })
        .then(async r => {
            const text = await r.text();
            if (!r.ok) {
                throw new Error(`HTTP ${r.status}: ${text}`);
            }
            return JSON.parse(text);
        })
        .then(data => {
            if (data.success) {
                alert('Test email sent successfully to ' + testEmail + '!');
            } else {
                alert('Error: ' + (data.error || 'Failed to send test email'));
            }
        })
        .catch(err => {
            alert('Error: ' + err.message);
        });
    };
    
    function escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

});
</script>

<!-- Create Email Campaign Modal -->
<div class="modal fade" id="createEmailCampaignModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create Email Campaign</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST" action="<?= url('api/email_service.php') ?>" id="createEmailCampaignForm">
                <div class="modal-body">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="create_campaign">
                    <div class="mb-3">
                        <label for="campaign_name" class="form-label">Campaign Name</label>
                        <input type="text" class="form-control" id="campaign_name" name="campaign_name" required>
                    </div>
                    <div class="mb-3">
                        <label for="subject_line" class="form-label">Subject Line</label>
                        <input type="text" class="form-control" id="subject_line" name="subject_line" required>
                    </div>
                    <div class="mb-3">
                        <label for="email_content" class="form-label">Email Content (HTML)</label>
                        <div class="btn-toolbar mb-2" role="toolbar">
                            <div class="btn-group btn-group-sm" role="group">
                                <button type="button" class="btn btn-outline-secondary" onclick="insertAdToEmail()" title="Insert Ad">
                                    <i class="bi bi-image"></i> Insert Ad
                                </button>
                                <button type="button" class="btn btn-outline-secondary" onclick="previewEmailWithAds()" title="Preview Email">
                                    <i class="bi bi-eye"></i> Preview
                                </button>
                                <button type="button" class="btn btn-outline-secondary" onclick="testEmailWithAds()" title="Test Email">
                                    <i class="bi bi-envelope-check"></i> Test Email
                                </button>
                            </div>
                        </div>
                        <textarea class="form-control font-monospace" id="email_content" name="email_content" rows="10" required></textarea>
                        <div class="form-text">Use the toolbar buttons to insert ads or test your email</div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Recipient Source</label>
                        <div class="form-check">
                            <input class="form-check-input" type="radio" name="recipient_source" id="recipient_source_manual" value="manual" checked onchange="toggleRecipientInputs()">
                            <label class="form-check-label" for="recipient_source_manual">
                                Manual Entry (comma-separated emails)
                            </label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="radio" name="recipient_source" id="recipient_source_crm" value="crm" onchange="toggleRecipientInputs()">
                            <label class="form-check-label" for="recipient_source_crm">
                                Send to CRM Customers (from CRM.php)
                            </label>
                        </div>
                    </div>
                    <div class="mb-3" id="manual_recipients_div">
                        <label for="recipients" class="form-label">Recipients (comma-separated emails)</label>
                        <textarea class="form-control" id="recipients" name="recipients" rows="3" placeholder="email1@example.com, email2@example.com"></textarea>
                    </div>
                    <div class="mb-3" id="crm_recipients_div" style="display: none;">
                        <div class="alert alert-info">
                            <small><strong>CRM Customers:</strong> Emails will be sent to all subscribed customers from your CRM. Rate limiting: 1 email per minute.</small>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <label for="daily_limit" class="form-label">Daily Email Limit</label>
                                <input type="number" class="form-control" id="daily_limit" name="daily_limit" min="1" placeholder="e.g., 100">
                                <small class="text-muted">Number of customers to email per day</small>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Total CRM Customers</label>
                                <input type="text" class="form-control" id="crm_total_count" readonly value="<?= $subscriberCount ?> customers">
                                <small class="text-muted">All subscribed customers will be queued</small>
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="parent_campaign_id" class="form-label">Link to Campaign (Optional)</label>
                        <select class="form-select" id="parent_campaign_id" name="campaign_id">
                            <option value="">None - Standalone Email Campaign</option>
                            <?php
                            // Get user's campaigns for linking
                            $stmt = $db->prepare("SELECT id, campaign_name FROM campaigns WHERE user_id = ? ORDER BY campaign_name");
                            $stmt->execute([$userId]);
                            $userCampaigns = $stmt->fetchAll();
                            foreach ($userCampaigns as $camp): ?>
                                <option value="<?= $camp['id'] ?>"><?= escape($camp['campaign_name']) ?></option>
                            <?php endforeach; ?>
                        </select>
                        <small class="text-muted">Link this email campaign to a parent campaign for tracking</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Campaign</button>
                </div>
            </form>
            
            <script>
            (function() {
                const form = document.getElementById('createEmailCampaignForm');
                if (form) {
                    form.addEventListener('submit', function(e) {
                        e.preventDefault();
                        const formData = new FormData(this);
                        const submitBtn = this.querySelector('button[type="submit"]');
                        const originalText = submitBtn.textContent;
                        submitBtn.disabled = true;
                        submitBtn.textContent = 'Creating...';
                        
                        fetch(this.action, {
                            method: 'POST',
                            body: formData
                        })
                        .then(r => r.json())
                        .then(data => {
                            if (data.success) {
                                alert(data.message || 'Campaign created successfully!');
                                window.location.reload();
                            } else {
                                alert('Error: ' + (data.error || 'Failed to create campaign'));
                                submitBtn.disabled = false;
                                submitBtn.textContent = originalText;
                            }
                        })
                        .catch(err => {
                            alert('Error: ' + err.message);
                            submitBtn.disabled = false;
                            submitBtn.textContent = originalText;
                        });
                    });
                }
            })();
            </script>
        </div>
    </div>
</div>

<!-- Create Template Modal -->
<div class="modal fade" id="createTemplateModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create Email Template</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST" action="<?= url('api/email_templates.php') ?>">
                <div class="modal-body">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="create">
                    <div class="mb-3">
                        <label for="template_name" class="form-label">Template Name</label>
                        <input type="text" class="form-control" id="template_name" name="template_name" required>
                    </div>
                    <div class="mb-3">
                        <label for="template_subject" class="form-label">Subject Line</label>
                        <input type="text" class="form-control" id="template_subject" name="subject_line">
                    </div>
                    <div class="mb-3">
                        <label for="template_content" class="form-label">Template Content (HTML)</label>
                        <div class="btn-toolbar mb-2" role="toolbar">
                            <div class="btn-group btn-group-sm" role="group">
                                <button type="button" class="btn btn-outline-secondary" onclick="insertAdToTemplate()" title="Insert Ad">
                                    <i class="bi bi-image"></i> Insert Ad
                                </button>
                            </div>
                        </div>
                        <textarea class="form-control font-monospace" id="template_content" name="email_content" rows="10" required></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Template</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Create Segment Modal -->
<div class="modal fade" id="createSegmentModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create Subscriber Segment</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST" action="<?= url('api/email_segmentation.php') ?>">
                <div class="modal-body">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="create">
                    <div class="mb-3">
                        <label for="segment_name" class="form-label">Segment Name</label>
                        <input type="text" class="form-control" id="segment_name" name="segment_name" required>
                    </div>
                    <div class="mb-3">
                        <label for="segment_criteria" class="form-label">Criteria (JSON)</label>
                        <textarea class="form-control" id="segment_criteria" name="criteria" rows="5" placeholder='{"status": "subscribed"}'></textarea>
                        <div class="form-text">Enter segment criteria as JSON</div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Segment</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Add Subscriber Modal -->
<div class="modal fade" id="addSubscriberModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add Email Subscriber</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST" action="<?= url('api/email_service.php') ?>">
                <div class="modal-body">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="add_subscriber">
                    <div class="mb-3">
                        <label for="subscriber_email" class="form-label">Email Address</label>
                        <input type="email" class="form-control" id="subscriber_email" name="email" required>
                    </div>
                    <div class="mb-3">
                        <label for="subscriber_name" class="form-label">Name (Optional)</label>
                        <input type="text" class="form-control" id="subscriber_name" name="name">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Subscriber</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Import Recipients Modal -->
<div class="modal fade" id="importRecipientsModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Import Recipients for <span id="importCampaignName"></span></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="importRecipientsForm">
                    <input type="hidden" name="campaign_id" id="importCampaignId" />
                    <div class="mb-3">
                        <label class="form-label">CSV File</label>
                        <input type="file" name="file" accept=".csv" class="form-control" required />
                        <div class="form-text">Headers: email[,name]</div>
                    </div>
                    <div class="text-muted small" id="importResult"></div>
                    <div class="mt-3 d-flex justify-content-end">
                        <button type="button" class="btn btn-secondary me-2" data-bs-dismiss="modal">Close</button>
                        <button type="submit" class="btn btn-primary">Upload</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
</div>

