<?php
/**
 * Discount Code Manager
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireClient();
$pageTitle = 'Discount Codes';

$userId = $_SESSION['user_id'];
$db = getDB();
$subscription = getCurrentSubscription();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        // Check if user can create discount code
        if (!canCreateDiscountCode($userId)) {
            setFlashMessage('danger', 'You have reached your monthly discount code limit for your current plan.');
            header('Location: ' . url('client/discounts.php'));
            exit;
        }
        
        $code = strtoupper(trim($_POST['code'] ?? ''));
        if (empty($code)) {
            $code = generateDiscountCode(8, 'DISC');
        }
        
        // Check if code is unique
        if (!isDiscountCodeUnique($code)) {
            setFlashMessage('danger', 'Discount code already exists. Please choose a different code.');
            header('Location: ' . url('client/discounts.php'));
            exit;
        }
        
        $discountType = $_POST['discount_type'] ?? 'percentage';
        $discountValue = floatval($_POST['discount_value'] ?? 0);
        $minPurchase = !empty($_POST['min_purchase_amount']) ? floatval($_POST['min_purchase_amount']) : null;
        $maxDiscount = !empty($_POST['max_discount_amount']) ? floatval($_POST['max_discount_amount']) : null;
        $usageLimit = !empty($_POST['usage_limit']) ? intval($_POST['usage_limit']) : null;
        $startDate = !empty($_POST['start_date']) ? $_POST['start_date'] : date('Y-m-d');
        $endDate = !empty($_POST['end_date']) ? $_POST['end_date'] : null;
        
        $stmt = $db->prepare("
            INSERT INTO discount_codes (user_id, code, discount_type, discount_value, min_purchase_amount, max_discount_amount, usage_limit, start_date, end_date, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'active')
        ");
        $stmt->execute([$userId, $code, $discountType, $discountValue, $minPurchase, $maxDiscount, $usageLimit, $startDate, $endDate]);
        
        setFlashMessage('success', 'Discount code created successfully!');
        header('Location: ' . url('client/discounts.php'));
        exit;
    } elseif ($action === 'update_status') {
        $codeId = intval($_POST['code_id'] ?? 0);
        $status = $_POST['status'] ?? 'inactive';
        
        $stmt = $db->prepare("UPDATE discount_codes SET status = ? WHERE id = ? AND user_id = ?");
        $stmt->execute([$status, $codeId, $userId]);
        
        setFlashMessage('success', 'Discount code updated successfully!');
        header('Location: ' . url('client/discounts.php'));
        exit;
    }
}

// Get all discount codes
$stmt = $db->prepare("
    SELECT * FROM discount_codes 
    WHERE user_id = ? 
    ORDER BY created_at DESC
");
$stmt->execute([$userId]);
$discounts = $stmt->fetchAll();

// Check remaining codes for the month
$remainingCodes = null;
$currentMonthCount = 0;
if ($subscription) {
    $stmt = $db->prepare("
        SELECT COUNT(*) as count
        FROM discount_codes 
        WHERE user_id = ? AND MONTH(created_at) = MONTH(NOW()) AND YEAR(created_at) = YEAR(NOW())
    ");
    $stmt->execute([$userId]);
    $currentMonthCount = $stmt->fetchColumn();
    
    if ($subscription['discount_codes_limit'] !== null) {
        $remainingCodes = $subscription['discount_codes_limit'] - $currentMonthCount;
    }
}

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row">
    <div class="col-12 mb-4">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h1>Discount Codes</h1>
                <?php if ($subscription && $subscription['discount_codes_limit'] !== null): ?>
                    <p class="text-muted mb-0">
                        Used: <?= $currentMonthCount ?> / <?= $subscription['discount_codes_limit'] ?> 
                        (<?= $remainingCodes ?> remaining this month)
                    </p>
                <?php else: ?>
                    <p class="text-muted mb-0">Unlimited discount codes</p>
                <?php endif; ?>
            </div>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createDiscountModal">
                <i class="bi bi-plus-circle"></i> Create Discount Code
            </button>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <?php if (empty($discounts)): ?>
                    <p class="text-muted text-center py-5">No discount codes yet. Create your first one!</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Code</th>
                                    <th>Type</th>
                                    <th>Value</th>
                                    <th>Usage</th>
                                    <th>Start Date</th>
                                    <th>End Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($discounts as $discount): ?>
                                    <tr>
                                        <td><strong><?= escape($discount['code']) ?></strong></td>
                                        <td><?= ucfirst(str_replace('_', ' ', $discount['discount_type'])) ?></td>
                                        <td>
                                            <?php if ($discount['discount_type'] === 'percentage'): ?>
                                                <?= number_format($discount['discount_value']) ?>%
                                            <?php else: ?>
                                                <?= formatCurrency($discount['discount_value']) ?>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?= $discount['usage_count'] ?> / <?= $discount['usage_limit'] ?? '∞' ?>
                                        </td>
                                        <td><?= formatDate($discount['start_date']) ?></td>
                                        <td><?= $discount['end_date'] ? formatDate($discount['end_date']) : 'No expiration' ?></td>
                                        <td>
                                            <span class="badge bg-<?= $discount['status'] === 'active' ? 'success' : 'secondary' ?>">
                                                <?= escape($discount['status']) ?>
                                            </span>
                                        </td>
                                        <td>
                                            <form method="POST" class="d-inline">
                                                <input type="hidden" name="action" value="update_status">
                                                <input type="hidden" name="code_id" value="<?= $discount['id'] ?>">
                                                <input type="hidden" name="status" value="<?= $discount['status'] === 'active' ? 'inactive' : 'active' ?>">
                                                <button type="submit" class="btn btn-sm btn-<?= $discount['status'] === 'active' ? 'warning' : 'success' ?>">
                                                    <?= $discount['status'] === 'active' ? 'Deactivate' : 'Activate' ?>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Create Discount Modal -->
<div class="modal fade" id="createDiscountModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create Discount Code</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="create">
                    
                    <div class="mb-3">
                        <label for="code" class="form-label">Discount Code</label>
                        <input type="text" class="form-control" id="code" name="code" placeholder="Leave empty to auto-generate">
                        <small class="text-muted">Will be auto-generated if left empty</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="discount_type" class="form-label">Discount Type</label>
                        <select class="form-select" id="discount_type" name="discount_type">
                            <option value="percentage">Percentage Off</option>
                            <option value="fixed_amount">Fixed Amount Off</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="discount_value" class="form-label">Discount Value</label>
                        <input type="number" step="0.01" class="form-control" id="discount_value" name="discount_value" required>
                        <small class="text-muted">Enter percentage (e.g., 20) or dollar amount (e.g., 10.00)</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="min_purchase_amount" class="form-label">Minimum Purchase Amount (Optional)</label>
                        <input type="number" step="0.01" class="form-control" id="min_purchase_amount" name="min_purchase_amount">
                    </div>
                    
                    <div class="mb-3">
                        <label for="max_discount_amount" class="form-label">Maximum Discount Amount (Optional)</label>
                        <input type="number" step="0.01" class="form-control" id="max_discount_amount" name="max_discount_amount">
                    </div>
                    
                    <div class="mb-3">
                        <label for="usage_limit" class="form-label">Usage Limit (Optional)</label>
                        <input type="number" class="form-control" id="usage_limit" name="usage_limit">
                        <small class="text-muted">Leave empty for unlimited usage</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="start_date" class="form-label">Start Date</label>
                        <input type="date" class="form-control" id="start_date" name="start_date" value="<?= date('Y-m-d') ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="end_date" class="form-label">End Date (Optional)</label>
                        <input type="date" class="form-control" id="end_date" name="end_date">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Discount Code</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Analytics Tab -->
<div class="row mt-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-graph-up"></i> Discount Code Analytics</h5>
            </div>
            <div class="card-body">
                <div class="row mb-3">
                    <div class="col-md-4">
                        <button class="btn btn-primary w-100" onclick="loadDiscountAnalytics('overview')">
                            <i class="bi bi-speedometer2"></i> Overview
                        </button>
                    </div>
                    <div class="col-md-4">
                        <button class="btn btn-info w-100" onclick="loadDiscountAnalytics('usage')">
                            <i class="bi bi-bar-chart"></i> Usage Tracking
                        </button>
                    </div>
                    <div class="col-md-4">
                        <button class="btn btn-warning w-100" onclick="loadDiscountAnalytics('expiring')">
                            <i class="bi bi-clock"></i> Expiring Soon
                        </button>
                    </div>
                </div>
                <div id="discountAnalyticsContent">
                    <p class="text-muted text-center">Click a button above to view analytics</p>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function loadDiscountAnalytics(type) {
    const content = document.getElementById('discountAnalyticsContent');
    content.innerHTML = '<p class="text-center">Loading...</p>';
    
    let url = '<?= url("api/discount_analytics.php") ?>?action=';
    if (type === 'overview') {
        url += 'overview';
    } else if (type === 'usage') {
        url += 'usage_tracking';
    } else if (type === 'expiring') {
        url += 'expiring_soon';
    }
    
    fetch(url)
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                let html = '<div class="table-responsive"><table class="table"><thead><tr>';
                
                if (type === 'overview') {
                    html += '<th>Metric</th><th>Value</th></tr></thead><tbody>';
                    html += `<tr><td>Total Codes</td><td><strong>${data.data.total_codes}</strong></td></tr>`;
                    html += `<tr><td>Active Codes</td><td><strong>${data.data.active_codes}</strong></td></tr>`;
                    html += `<tr><td>Total Usages</td><td><strong>${data.data.total_usages}</strong></td></tr>`;
                    html += `<tr><td>Total Discount Amount</td><td><strong>$${parseFloat(data.data.total_discount_amount || 0).toFixed(2)}</strong></td></tr>`;
                } else {
                    // Usage tracking or expiring
                    html += '</thead><tbody>';
                    data.data.forEach(item => {
                        html += '<tr>';
                        html += `<td>${item.code || item.discount_code_id}</td>`;
                        html += `<td>${item.usage_count || item.used_at || ''}</td>`;
                        html += '</tr>';
                    });
                }
                
                html += '</tbody></table></div>';
                content.innerHTML = html;
            }
        });
}
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

