<?php
/**
 * Direct Mail Campaign Management
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

// Helper function for file upload
function uploadDirectMailFile($file, $uploadType, $userId, $db) {
    $uploadDir = __DIR__ . '/../uploads/user_' . $userId . '/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $fileName = uniqid() . '_' . time() . '.' . $extension;
    $filePath = $uploadDir . $fileName;
    
    if (move_uploaded_file($file['tmp_name'], $filePath)) {
        $relativePath = 'uploads/user_' . $userId . '/' . $fileName;
        
        $stmt = $db->prepare("
            INSERT INTO file_uploads 
            (user_id, file_name, original_name, file_path, file_type, file_size, mime_type, upload_type)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $userId,
            $fileName,
            $file['name'],
            $relativePath,
            $extension,
            $file['size'],
            $file['type'],
            $uploadType
        ]);
        
        return ['success' => true, 'file_path' => $relativePath];
    }
    
    return ['success' => false];
}

requireClient();
$pageTitle = 'Direct Mail Campaigns';

$userId = $_SESSION['user_id'];
$subscription = getCurrentSubscription();
$db = getDB();

// Check if user has direct mail access
$canAccessDirectMail = false;
$directMailQuantity = 0;
$directMailFrequency = 'monthly';

if ($subscription) {
    $canAccessDirectMail = true;
    $directMailQuantity = $subscription['direct_mail_quantity'];
    $directMailFrequency = $subscription['direct_mail_frequency'];
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $canAccessDirectMail) {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        require_once __DIR__ . '/../includes/csrf.php';
        
        $campaignName = $_POST['campaign_name'] ?? '';
        $mailType = $_POST['mail_type'] ?? 'postcard';
        $quantity = intval($_POST['quantity'] ?? 0);
        
        // Validate required fields
        if (empty($campaignName) || $quantity <= 0) {
            setFlashMessage('danger', 'Please fill in all required fields.');
            header('Location: ' . url('client/direct_mail.php'));
            exit;
        }
        
        // Check quantity limit
        if ($quantity > $directMailQuantity) {
            setFlashMessage('danger', "Quantity exceeds your plan limit of {$directMailQuantity} per {$directMailFrequency}.");
            header('Location: ' . url('client/direct_mail.php'));
            exit;
        }
        
        // Handle file uploads
        $designFilePath = null;
        $mailingListFilePath = null;
        
        // Upload design file (required)
        if (isset($_FILES['design_file']) && $_FILES['design_file']['error'] === UPLOAD_ERR_OK) {
            $uploadResult = uploadDirectMailFile($_FILES['design_file'], 'direct_mail_design', $userId, $db);
            if ($uploadResult['success']) {
                $designFilePath = $uploadResult['file_path'];
            }
        }
        
        // Upload mailing list if provided
        $mailingSource = $_POST['mailing_source'] ?? 'upload';
        if ($mailingSource === 'upload' && isset($_FILES['mailing_list_file']) && $_FILES['mailing_list_file']['error'] === UPLOAD_ERR_OK) {
            $uploadResult = uploadDirectMailFile($_FILES['mailing_list_file'], 'mailing_list', $userId, $db);
            if ($uploadResult['success']) {
                $mailingListFilePath = $uploadResult['file_path'];
            }
        } elseif ($mailingSource === 'manual') {
            // Store manual addresses in a text file
            $manualAddresses = $_POST['manual_addresses'] ?? '';
            if (!empty($manualAddresses)) {
                $uploadDir = __DIR__ . '/../uploads/user_' . $userId . '/';
                if (!is_dir($uploadDir)) {
                    mkdir($uploadDir, 0755, true);
                }
                $fileName = 'manual_list_' . uniqid() . '_' . time() . '.txt';
                $filePath = $uploadDir . $fileName;
                file_put_contents($filePath, $manualAddresses);
                $mailingListFilePath = 'uploads/user_' . $userId . '/' . $fileName;
            }
        }
        
        // Collect all form data for storage
        $mailerData = [
            'sender' => [
                'name' => $_POST['sender_name'] ?? '',
                'address_line1' => $_POST['sender_address_line1'] ?? '',
                'address_line2' => $_POST['sender_address_line2'] ?? '',
                'city' => $_POST['sender_city'] ?? '',
                'state' => $_POST['sender_state'] ?? '',
                'zip' => $_POST['sender_zip'] ?? ''
            ],
            'mail_specs' => [
                'size' => $_POST['mail_size'] ?? 'standard',
                'weight' => floatval($_POST['mail_weight'] ?? 1),
                'paper_weight' => $_POST['paper_weight'] ?? '',
                'printing_side' => $_POST['printing_side'] ?? 'single',
                'dimensions' => [
                    'width' => !empty($_POST['dimension_width']) ? floatval($_POST['dimension_width']) : null,
                    'height' => !empty($_POST['dimension_height']) ? floatval($_POST['dimension_height']) : null,
                    'depth' => !empty($_POST['dimension_depth']) ? floatval($_POST['dimension_depth']) : null
                ],
                'special_requirements' => $_POST['special_requirements'] ?? []
            ],
            'post_office' => [
                'postage_type' => $_POST['postage_type'] ?? 'first_class',
                'additional_services' => $_POST['additional_services'] ?? [],
                'delivery_confirmation' => $_POST['delivery_confirmation'] ?? 'none'
            ],
            'design_notes' => $_POST['design_notes'] ?? '',
            'mailing_notes' => $_POST['mailing_notes'] ?? '',
            'mailing_source' => $mailingSource
        ];
        
        // Calculate costs
        $baseCostPerUnit = 0.30; // Printing cost
        $basePostagePerUnit = 0.50; // Base postage
        
        // Add service costs
        $serviceCosts = [
            'tracking' => 0.50,
            'certified' => 3.65,
            'return_receipt' => 2.80,
            'signature' => 2.85,
            'address_service' => 0.05
        ];
        
        $additionalServiceCost = 0;
        foreach ($mailerData['post_office']['additional_services'] as $service) {
            if (isset($serviceCosts[$service])) {
                $additionalServiceCost += $serviceCosts[$service];
            }
        }
        
        $costPerUnit = $baseCostPerUnit;
        $postagePerUnit = $basePostagePerUnit + $additionalServiceCost;
        $totalCost = ($costPerUnit * $quantity) + ($postagePerUnit * $quantity);
        
        $campaignId = !empty($_POST['campaign_id']) ? intval($_POST['campaign_id']) : null;
        $scheduledDate = !empty($_POST['scheduled_date']) ? $_POST['scheduled_date'] : null;
        
        // Store mailer data as JSON (we'll add a column or use existing design_file field)
        // For now, we'll add it to a JSON column if it exists, or store in a separate JSON file
        $mailerDataJson = json_encode($mailerData);
        
        $stmt = $db->prepare("
            INSERT INTO direct_mail_campaigns 
            (user_id, campaign_id, campaign_name, mail_type, quantity, design_file, mailing_list_file, cost_per_unit, postage_cost, total_cost, scheduled_date, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'draft')
        ");
        $stmt->execute([
            $userId, 
            $campaignId, 
            $campaignName, 
            $mailType, 
            $quantity,
            $designFilePath,
            $mailingListFilePath,
            $costPerUnit, 
            ($postagePerUnit * $quantity),
            $totalCost, 
            $scheduledDate
        ]);
        
        $mailCampaignId = $db->lastInsertId();
        
        // Store additional data in a JSON file for now (could be migrated to DB column later)
        if ($mailCampaignId) {
            $dataDir = __DIR__ . '/../uploads/user_' . $userId . '/mailer_data/';
            if (!is_dir($dataDir)) {
                mkdir($dataDir, 0755, true);
            }
            file_put_contents($dataDir . $mailCampaignId . '.json', $mailerDataJson);
        }
        
        setFlashMessage('success', 'Post office mailer campaign created successfully! Your order has been submitted for processing.');
        header('Location: ' . url('client/direct_mail.php'));
        exit;
    } elseif ($action === 'update_status') {
        $mailCampaignId = intval($_POST['mail_campaign_id'] ?? 0);
        $status = $_POST['status'] ?? 'draft';
        
        $stmt = $db->prepare("UPDATE direct_mail_campaigns SET status = ? WHERE id = ? AND user_id = ?");
        $stmt->execute([$status, $mailCampaignId, $userId]);
        
        setFlashMessage('success', 'Campaign status updated successfully!');
        header('Location: ' . url('client/direct_mail.php'));
        exit;
    }
}

// Get all direct mail campaigns
$stmt = $db->prepare("
    SELECT dm.*, c.campaign_name as parent_campaign_name
    FROM direct_mail_campaigns dm
    LEFT JOIN campaigns c ON dm.campaign_id = c.id
    WHERE dm.user_id = ? 
    ORDER BY dm.created_at DESC
");
$stmt->execute([$userId]);
$mailCampaigns = $stmt->fetchAll();

// Get available campaigns for dropdown
$stmt = $db->prepare("SELECT id, campaign_name FROM campaigns WHERE user_id = ? ORDER BY campaign_name");
$stmt->execute([$userId]);
$availableCampaigns = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
?>

<?php if (!$canAccessDirectMail): ?>
<div class="row">
    <div class="col-12">
        <div class="alert alert-warning">
            <h5>Direct Mail Not Available</h5>
            <p>Direct mail campaigns are not included in your current subscription plan. Please upgrade to access this feature.</p>
        </div>
    </div>
</div>
<?php else: ?>

<div class="row">
    <div class="col-12 mb-4">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h1>Direct Mail Campaigns</h1>
                <p class="text-muted mb-0">
                    Plan Limit: <?= number_format($directMailQuantity) ?> pieces per <?= $directMailFrequency ?>
                </p>
            </div>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createMailModal">
                <i class="bi bi-plus-circle"></i> Create Campaign
            </button>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <?php if (empty($mailCampaigns)): ?>
                    <p class="text-muted text-center py-5">No direct mail campaigns yet. Create your first one!</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Campaign Name</th>
                                    <th>Type</th>
                                    <th>Quantity</th>
                                    <th>Total Cost</th>
                                    <th>Scheduled Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($mailCampaigns as $campaign): ?>
                                    <tr>
                                        <td>
                                            <strong><?= escape($campaign['campaign_name']) ?></strong>
                                            <?php if ($campaign['parent_campaign_name']): ?>
                                                <br><small class="text-muted">Part of: <?= escape($campaign['parent_campaign_name']) ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= ucfirst($campaign['mail_type']) ?></td>
                                        <td><?= number_format($campaign['quantity']) ?></td>
                                        <td><?= formatCurrency($campaign['total_cost'] ?? 0) ?></td>
                                        <td><?= $campaign['scheduled_date'] ? formatDate($campaign['scheduled_date']) : 'Not scheduled' ?></td>
                                        <td>
                                            <span class="badge bg-<?= 
                                                $campaign['status'] === 'mailed' ? 'success' : 
                                                ($campaign['status'] === 'in_production' ? 'warning' : 'secondary') 
                                            ?>">
                                                <?= ucfirst(str_replace('_', ' ', $campaign['status'])) ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($campaign['status'] === 'draft' || $campaign['status'] === 'scheduled'): ?>
                                                <form method="POST" class="d-inline">
                                                    <input type="hidden" name="action" value="update_status">
                                                    <input type="hidden" name="mail_campaign_id" value="<?= $campaign['id'] ?>">
                                                    <input type="hidden" name="status" value="in_production">
                                                    <button type="submit" class="btn btn-sm btn-primary">Start Production</button>
                                                </form>
                                            <?php endif; ?>
                                            <?php if ($campaign['tracking_number']): ?>
                                                <br><small class="text-muted">Tracking: <?= escape($campaign['tracking_number']) ?></small>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Create Direct Mail Modal -->
<div class="modal fade" id="createMailModal" tabindex="-1">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bi bi-envelope-paper"></i> Create Post Office Mailer Campaign</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" enctype="multipart/form-data" id="directMailForm">
                <div class="modal-body">
                    <input type="hidden" name="action" value="create">
                    
                    <!-- Tab Navigation -->
                    <ul class="nav nav-tabs mb-4" id="mailerTabs" role="tablist">
                        <li class="nav-item" role="presentation">
                            <button class="nav-link active" id="campaign-tab" data-bs-toggle="tab" data-bs-target="#campaign" type="button" role="tab">Campaign Info</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="sender-tab" data-bs-toggle="tab" data-bs-target="#sender" type="button" role="tab">Sender Address</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="mailing-tab" data-bs-toggle="tab" data-bs-target="#mailing" type="button" role="tab">Mailing List</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="specs-tab" data-bs-toggle="tab" data-bs-target="#specs" type="button" role="tab">Mail Specifications</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="design-tab" data-bs-toggle="tab" data-bs-target="#design" type="button" role="tab">Design & Files</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="services-tab" data-bs-toggle="tab" data-bs-target="#services" type="button" role="tab">Post Office Services</button>
                        </li>
                    </ul>
                    
                    <!-- Tab Content -->
                    <div class="tab-content" id="mailerTabContent">
                        <!-- Campaign Info Tab -->
                        <div class="tab-pane fade show active" id="campaign" role="tabpanel">
                            <div class="mb-3">
                                <label for="campaign_name" class="form-label">Campaign Name <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="campaign_name" name="campaign_name" required placeholder="Summer Promotion Mailer">
                            </div>
                            
                            <div class="mb-3">
                                <label for="campaign_id" class="form-label">Link to Campaign (Optional)</label>
                                <select class="form-select" id="campaign_id" name="campaign_id">
                                    <option value="">None - Standalone Campaign</option>
                                    <?php foreach ($availableCampaigns as $camp): ?>
                                        <option value="<?= $camp['id'] ?>"><?= escape($camp['campaign_name']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label for="mail_type" class="form-label">Mail Type <span class="text-danger">*</span></label>
                                <select class="form-select" id="mail_type" name="mail_type" required>
                                    <option value="postcard">Postcard</option>
                                    <option value="letter">Letter (Standard)</option>
                                    <option value="flyer">Flyer/Brochure</option>
                                </select>
                                <small class="text-muted">Select the type of mail piece</small>
                            </div>
                            
                            <div class="mb-3">
                                <label for="quantity" class="form-label">Quantity <span class="text-danger">*</span></label>
                                <input type="number" class="form-control" id="quantity" name="quantity" min="1" max="<?= $directMailQuantity ?>" required>
                                <small class="text-muted">Maximum: <?= number_format($directMailQuantity) ?> (your plan limit)</small>
                            </div>
                            
                            <div class="mb-3">
                                <label for="scheduled_date" class="form-label">Requested Mail Date</label>
                                <input type="date" class="form-control" id="scheduled_date" name="scheduled_date" min="<?= date('Y-m-d') ?>">
                                <small class="text-muted">When do you want this campaign to be mailed?</small>
                            </div>
                        </div>
                        
                        <!-- Sender Address Tab -->
                        <div class="tab-pane fade" id="sender" role="tabpanel">
                            <div class="alert alert-info">
                                <i class="bi bi-info-circle"></i> This address will be used as the return address on all mail pieces.
                            </div>
                            
                            <div class="mb-3">
                                <label for="sender_name" class="form-label">Sender/Company Name <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="sender_name" name="sender_name" required placeholder="Your Company Name">
                            </div>
                            
                            <div class="mb-3">
                                <label for="sender_address_line1" class="form-label">Street Address <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="sender_address_line1" name="sender_address_line1" required placeholder="123 Main Street">
                            </div>
                            
                            <div class="mb-3">
                                <label for="sender_address_line2" class="form-label">Address Line 2 (Optional)</label>
                                <input type="text" class="form-control" id="sender_address_line2" name="sender_address_line2" placeholder="Suite 100">
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="sender_city" class="form-label">City <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="sender_city" name="sender_city" required placeholder="New York">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label for="sender_state" class="form-label">State <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="sender_state" name="sender_state" required placeholder="NY" maxlength="2">
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label for="sender_zip" class="form-label">ZIP Code <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="sender_zip" name="sender_zip" required placeholder="10001" pattern="[0-9]{5}(-[0-9]{4})?">
                                </div>
                            </div>
                        </div>
                        
                        <!-- Mailing List Tab -->
                        <div class="tab-pane fade" id="mailing" role="tabpanel">
                            <div class="mb-3">
                                <label class="form-label">Mailing List Source <span class="text-danger">*</span></label>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="mailing_source" id="mailing_upload" value="upload" checked>
                                    <label class="form-check-label" for="mailing_upload">
                                        Upload Mailing List (CSV/Excel)
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="mailing_source" id="mailing_manual" value="manual">
                                    <label class="form-check-label" for="mailing_manual">
                                        Enter Manually (for small lists)
                                    </label>
                                </div>
                            </div>
                            
                            <div id="uploadMailingList">
                                <div class="mb-3">
                                    <label for="mailing_list_file" class="form-label">Upload Mailing List File</label>
                                    <input type="file" class="form-control" id="mailing_list_file" name="mailing_list_file" accept=".csv,.xlsx,.xls">
                                    <small class="text-muted">CSV or Excel file with columns: Name, Address, City, State, ZIP (or similar)</small>
                                </div>
                                <div class="alert alert-light">
                                    <strong>File Format Requirements:</strong>
                                    <ul class="mb-0">
                                        <li>CSV or Excel format</li>
                                        <li>Required columns: Name, Address, City, State, ZIP</li>
                                        <li>First row should contain column headers</li>
                                    </ul>
                                </div>
                            </div>
                            
                            <div id="manualMailingList" style="display: none;">
                                <div class="mb-3">
                                    <label for="manual_addresses" class="form-label">Enter Addresses (one per line)</label>
                                    <textarea class="form-control" id="manual_addresses" name="manual_addresses" rows="8" placeholder="John Doe&#10;123 Main St&#10;New York, NY 10001&#10;&#10;Jane Smith&#10;456 Oak Ave&#10;Los Angeles, CA 90001"></textarea>
                                    <small class="text-muted">Format: Name on first line, Address on second line, City State ZIP on third line. Separate entries with blank line.</small>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="mailing_notes" class="form-label">Mailing List Notes (Optional)</label>
                                <textarea class="form-control" id="mailing_notes" name="mailing_notes" rows="2" placeholder="Any special instructions about the mailing list..."></textarea>
                            </div>
                        </div>
                        
                        <!-- Mail Specifications Tab -->
                        <div class="tab-pane fade" id="specs" role="tabpanel">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="mail_size" class="form-label">Mail Size <span class="text-danger">*</span></label>
                                    <select class="form-select" id="mail_size" name="mail_size" required>
                                        <option value="standard">Standard (Regular)</option>
                                        <option value="large">Large Envelope</option>
                                        <option value="flat">Flat</option>
                                        <option value="parcel">Parcel</option>
                                    </select>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="mail_weight" class="form-label">Weight (oz) <span class="text-danger">*</span></label>
                                    <input type="number" class="form-control" id="mail_weight" name="mail_weight" step="0.1" min="0.1" max="70" value="1" required>
                                    <small class="text-muted">Weight per piece in ounces</small>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="paper_weight" class="form-label">Paper Weight</label>
                                    <select class="form-select" id="paper_weight" name="paper_weight">
                                        <option value="">Not specified</option>
                                        <option value="20lb">20 lb</option>
                                        <option value="24lb">24 lb</option>
                                        <option value="28lb">28 lb</option>
                                        <option value="32lb">32 lb</option>
                                        <option value="60lb">60 lb Card Stock</option>
                                        <option value="80lb">80 lb Card Stock</option>
                                    </select>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="printing_side" class="form-label">Printing</label>
                                    <select class="form-select" id="printing_side" name="printing_side">
                                        <option value="single">Single-sided</option>
                                        <option value="double">Double-sided</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="dimensions" class="form-label">Dimensions (Optional)</label>
                                <div class="row">
                                    <div class="col-md-4">
                                        <input type="number" class="form-control" id="dimension_width" name="dimension_width" step="0.1" placeholder="Width (inches)">
                                    </div>
                                    <div class="col-md-4">
                                        <input type="number" class="form-control" id="dimension_height" name="dimension_height" step="0.1" placeholder="Height (inches)">
                                    </div>
                                    <div class="col-md-4">
                                        <input type="number" class="form-control" id="dimension_depth" name="dimension_depth" step="0.1" placeholder="Depth (inches)">
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Special Requirements</label>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="perforated" name="special_requirements[]" value="perforated">
                                    <label class="form-check-label" for="perforated">Perforated edges</label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="folded" name="special_requirements[]" value="folded">
                                    <label class="form-check-label" for="folded">Folded</label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="window_envelope" name="special_requirements[]" value="window_envelope">
                                    <label class="form-check-label" for="window_envelope">Window envelope</label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="color_printing" name="special_requirements[]" value="color_printing">
                                    <label class="form-check-label" for="color_printing">Full color printing</label>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Design & Files Tab -->
                        <div class="tab-pane fade" id="design" role="tabpanel">
                            <div class="mb-3">
                                <label for="design_file" class="form-label">Design File <span class="text-danger">*</span></label>
                                <input type="file" class="form-control" id="design_file" name="design_file" accept=".pdf,.jpg,.jpeg,.png,.ai,.psd,.eps" required>
                                <small class="text-muted">Accepted formats: PDF, JPG, PNG, AI, PSD, EPS. High resolution recommended (300 DPI minimum).</small>
                            </div>
                            
                            <div class="alert alert-warning">
                                <strong>Design Requirements:</strong>
                                <ul class="mb-0">
                                    <li>Minimum 300 DPI resolution for print quality</li>
                                    <li>CMYK color mode recommended for accurate printing</li>
                                    <li>Include bleed areas (0.125" on all sides if needed)</li>
                                    <li>Safe zones: Keep important text/content 0.25" from edges</li>
                                </ul>
                            </div>
                            
                            <div class="mb-3">
                                <label for="design_notes" class="form-label">Design Notes/Instructions</label>
                                <textarea class="form-control" id="design_notes" name="design_notes" rows="3" placeholder="Any special instructions about the design or printing..."></textarea>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Additional Files (Optional)</label>
                                <input type="file" class="form-control" id="additional_files" name="additional_files[]" multiple>
                                <small class="text-muted">Upload any supporting files, logos, or assets</small>
                            </div>
                        </div>
                        
                        <!-- Post Office Services Tab -->
                        <div class="tab-pane fade" id="services" role="tabpanel">
                            <div class="mb-3">
                                <label for="postage_type" class="form-label">Postage Type <span class="text-danger">*</span></label>
                                <select class="form-select" id="postage_type" name="postage_type" required>
                                    <option value="first_class">First-Class Mail</option>
                                    <option value="standard">Standard Mail (Bulk Rate)</option>
                                    <option value="priority">Priority Mail</option>
                                    <option value="presorted">Presorted Standard</option>
                                </select>
                                <small class="text-muted">Select the postage class for your mail pieces</small>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Additional Services</label>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="service_tracking" name="additional_services[]" value="tracking">
                                    <label class="form-check-label" for="service_tracking">
                                        Tracking Number <small class="text-muted">(+$0.50 per piece)</small>
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="service_certified" name="additional_services[]" value="certified">
                                    <label class="form-check-label" for="service_certified">
                                        Certified Mail <small class="text-muted">(+$3.65 per piece)</small>
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="service_return_receipt" name="additional_services[]" value="return_receipt">
                                    <label class="form-check-label" for="service_return_receipt">
                                        Return Receipt Requested <small class="text-muted">(+$2.80 per piece)</small>
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="service_signature" name="additional_services[]" value="signature">
                                    <label class="form-check-label" for="service_signature">
                                        Signature Confirmation <small class="text-muted">(+$2.85 per piece)</small>
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="service_address_service" name="additional_services[]" value="address_service">
                                    <label class="form-check-label" for="service_address_service">
                                        Address Change Service <small class="text-muted">(+$0.05 per piece)</small>
                                    </label>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="delivery_confirmation" class="form-label">Delivery Confirmation</label>
                                <select class="form-select" id="delivery_confirmation" name="delivery_confirmation">
                                    <option value="none">None</option>
                                    <option value="electronic">Electronic (Included with some services)</option>
                                    <option value="physical">Physical Receipt</option>
                                </select>
                            </div>
                            
                            <div class="alert alert-info">
                                <strong>Estimated Costs:</strong><br>
                                <span id="costEstimate">Base: $0.80 per piece</span> (includes printing and standard postage)<br>
                                <span id="totalEstimate">Total: $0.00</span><br>
                                <small class="text-muted">Additional services will increase the cost. Final pricing will be confirmed before processing.</small>
                            </div>
                        </div>
                    </div>
                    
                    <div class="alert alert-success mt-4">
                        <strong><i class="bi bi-check-circle"></i> Review Your Information</strong>
                        <p class="mb-0">Please review all tabs to ensure all information is correct before submitting. Required fields are marked with <span class="text-danger">*</span>.</p>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-send"></i> Submit to Post Office
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Mailing source toggle
    document.querySelectorAll('input[name="mailing_source"]').forEach(radio => {
        radio.addEventListener('change', function() {
            const uploadDiv = document.getElementById('uploadMailingList');
            const manualDiv = document.getElementById('manualMailingList');
            
            if (this.value === 'upload') {
                uploadDiv.style.display = 'block';
                manualDiv.style.display = 'none';
                document.getElementById('mailing_list_file').removeAttribute('disabled');
                document.getElementById('manual_addresses').setAttribute('disabled', 'disabled');
            } else {
                uploadDiv.style.display = 'none';
                manualDiv.style.display = 'block';
                document.getElementById('mailing_list_file').setAttribute('disabled', 'disabled');
                document.getElementById('manual_addresses').removeAttribute('disabled');
            }
        });
    });
    
    // Cost calculator
    function calculateCost() {
        const quantity = parseInt(document.getElementById('quantity').value) || 0;
        const baseCostPerUnit = 0.80;
        let additionalCostPerUnit = 0;
        
        // Calculate additional service costs
        document.querySelectorAll('input[name="additional_services[]"]:checked').forEach(checkbox => {
            const service = checkbox.value;
            const costs = {
                'tracking': 0.50,
                'certified': 3.65,
                'return_receipt': 2.80,
                'signature': 2.85,
                'address_service': 0.05
            };
            if (costs[service]) {
                additionalCostPerUnit += costs[service];
            }
        });
        
        const totalCostPerUnit = baseCostPerUnit + additionalCostPerUnit;
        const total = quantity * totalCostPerUnit;
        
        document.getElementById('costEstimate').textContent = 
            `Base: $${baseCostPerUnit.toFixed(2)} per piece` + 
            (additionalCostPerUnit > 0 ? ` + Services: $${additionalCostPerUnit.toFixed(2)}` : '');
        document.getElementById('totalEstimate').textContent = `Total: $${total.toFixed(2)}`;
    }
    
    document.getElementById('quantity').addEventListener('input', calculateCost);
    document.querySelectorAll('input[name="additional_services[]"]').forEach(checkbox => {
        checkbox.addEventListener('change', calculateCost);
    });
    
    // Form validation
    document.getElementById('directMailForm').addEventListener('submit', function(e) {
        const mailingSource = document.querySelector('input[name="mailing_source"]:checked').value;
        
        if (mailingSource === 'upload') {
            const fileInput = document.getElementById('mailing_list_file');
            if (!fileInput.files || fileInput.files.length === 0) {
                e.preventDefault();
                alert('Please upload a mailing list file or select manual entry.');
                document.getElementById('mailing-tab').click();
                return false;
            }
        } else {
            const manualInput = document.getElementById('manual_addresses');
            if (!manualInput.value.trim()) {
                e.preventDefault();
                alert('Please enter mailing addresses manually or upload a file.');
                document.getElementById('mailing-tab').click();
                return false;
            }
        }
        
        // Check design file
        const designFile = document.getElementById('design_file');
        if (!designFile.files || designFile.files.length === 0) {
            e.preventDefault();
            alert('Please upload a design file.');
            document.getElementById('design-tab').click();
            return false;
        }
    });
    
    // Initial cost calculation
    calculateCost();
});
</script>

<?php endif; ?>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

