<?php
/**
 * Client Dashboard
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireClient();
$pageTitle = 'Dashboard';

$userId = getEffectiveUserId(); // Support admin viewing as client
$subscription = getCurrentSubscription();
$db = getDB();

// Get metrics summary (last 30 days)
$metrics = getUserMetricsSummary($userId, 30);

// Get campaign performance
$adPerformance = getCampaignPerformanceSummary($userId, 30);

// Get recent campaigns
$stmt = $db->prepare("
    SELECT * FROM campaigns 
    WHERE user_id = ? 
    ORDER BY created_at DESC 
    LIMIT 5
");
$stmt->execute([$userId]);
$recentCampaigns = $stmt->fetchAll();

// Get active discount codes
$stmt = $db->prepare("
    SELECT * FROM discount_codes 
    WHERE user_id = ? AND status = 'active' 
    ORDER BY created_at DESC 
    LIMIT 5
");
$stmt->execute([$userId]);
$activeDiscounts = $stmt->fetchAll();

// Calculate budget usage
$googleAdsUsed = 0;
$facebookAdsUsed = 0;
if ($subscription) {
    $stmt = $db->prepare("
        SELECT ad_platform, SUM(cost) as total_cost
        FROM ad_performance
        WHERE user_id = ? AND date >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY ad_platform
    ");
    $stmt->execute([$userId]);
    $budgetUsage = $stmt->fetchAll();
    foreach ($budgetUsage as $usage) {
        if ($usage['ad_platform'] === 'google_ads') {
            $googleAdsUsed = $usage['total_cost'];
        } elseif ($usage['ad_platform'] === 'facebook_ads') {
            $facebookAdsUsed = $usage['total_cost'];
        }
    }
}

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row">
    <div class="col-12 mb-4">
        <h1>Dashboard</h1>
        <?php if ($subscription): ?>
            <p class="text-muted">Welcome, <?= escape($_SESSION['business_name']) ?> - <?= escape($subscription['tier_name']) ?> Plan</p>
        <?php endif; ?>
    </div>
</div>

<!-- Quick Actions: Marketing Playground -->
<div class="row mb-4">
    <div class="col-md-6 mb-3">
        <div class="card h-100">
            <div class="card-body">
                <h5 class="card-title">Email Sender</h5>
                <p class="card-text">Create ads and send email campaigns all in one place.</p>
                <a class="btn btn-primary" href="<?= url('client/unified_email_sender.php') ?>">Open Email Sender</a>
            </div>
        </div>
    </div>
    <div class="col-md-6 mb-3">
        <div class="card h-100">
            <div class="card-body">
                <h5 class="card-title">Website Generator</h5>
                <p class="card-text">Generate a quick landing site for your campaign.</p>
                <a class="btn btn-outline-primary" href="<?= url('ad-generator/website-generator.html') ?>">Open Website Generator</a>
            </div>
        </div>
    </div>
    </div>

<!-- Key Metrics Cards -->
<div class="row mb-4">
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-primary">
            <div class="card-body">
                <h5 class="card-title">Total Visits</h5>
                <h2 class="mb-0"><?= number_format($metrics['total_visits'] ?? 0) ?></h2>
                <small>Last 30 days</small>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-success">
            <div class="card-body">
                <h5 class="card-title">Conversions</h5>
                <h2 class="mb-0"><?= number_format($metrics['conversions'] ?? 0) ?></h2>
                <small>Last 30 days</small>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-info">
            <div class="card-body">
                <h5 class="card-title">Revenue</h5>
                <h2 class="mb-0"><?= formatCurrency($metrics['total_revenue'] ?? 0) ?></h2>
                <small>Last 30 days</small>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-warning">
            <div class="card-body">
                <h5 class="card-title">Unique Visitors</h5>
                <h2 class="mb-0"><?= number_format($metrics['unique_visitors'] ?? 0) ?></h2>
                <small>Last 30 days</small>
            </div>
        </div>
    </div>
</div>

<?php if ($subscription): ?>
<!-- Budget Usage -->
<div class="row mb-4">
    <div class="col-md-6 mb-3">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Google Ads Budget</h5>
            </div>
            <div class="card-body">
                <?php 
                $googleBudgetLimit = $subscription['google_ads_budget'];
                $googlePercent = $googleBudgetLimit > 0 ? ($googleAdsUsed / $googleBudgetLimit * 100) : 0;
                ?>
                <div class="d-flex justify-content-between mb-2">
                    <span><?= formatCurrency($googleAdsUsed) ?> / <?= formatCurrency($googleBudgetLimit) ?></span>
                    <span><?= number_format($googlePercent, 1) ?>%</span>
                </div>
                <div class="progress">
                    <div class="progress-bar" role="progressbar" style="width: <?= min($googlePercent, 100) ?>%"></div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-md-6 mb-3">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Facebook Ads Budget</h5>
            </div>
            <div class="card-body">
                <?php 
                $facebookBudgetLimit = $subscription['facebook_ads_budget'];
                $facebookPercent = $facebookBudgetLimit > 0 ? ($facebookAdsUsed / $facebookBudgetLimit * 100) : 0;
                ?>
                <div class="d-flex justify-content-between mb-2">
                    <span><?= formatCurrency($facebookAdsUsed) ?> / <?= formatCurrency($facebookBudgetLimit) ?></span>
                    <span><?= number_format($facebookPercent, 1) ?>%</span>
                </div>
                <div class="progress">
                    <div class="progress-bar bg-success" role="progressbar" style="width: <?= min($facebookPercent, 100) ?>%"></div>
                </div>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Campaign Performance -->
<?php if (!empty($adPerformance)): ?>
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Ad Performance (Last 30 Days)</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Platform</th>
                                <th>Impressions</th>
                                <th>Clicks</th>
                                <th>CTR</th>
                                <th>Conversions</th>
                                <th>Cost</th>
                                <th>Revenue</th>
                                <th>ROAS</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($adPerformance as $perf): ?>
                                <tr>
                                    <td><?= ucfirst(str_replace('_', ' ', $perf['ad_platform'])) ?></td>
                                    <td><?= number_format($perf['total_impressions']) ?></td>
                                    <td><?= number_format($perf['total_clicks']) ?></td>
                                    <td><?= number_format($perf['ctr'], 2) ?>%</td>
                                    <td><?= number_format($perf['total_conversions']) ?></td>
                                    <td><?= formatCurrency($perf['total_cost']) ?></td>
                                    <td><?= formatCurrency($perf['total_revenue']) ?></td>
                                    <td>
                                        <?php 
                                        $roas = $perf['total_cost'] > 0 ? ($perf['total_revenue'] / $perf['total_cost']) : 0;
                                        echo number_format($roas, 2);
                                        ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Recent Campaigns & Active Discounts -->
<div class="row">
    <div class="col-md-6 mb-4">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Recent Campaigns</h5>
                <a href="<?= url('client/campaigns.php') ?>" class="btn btn-sm btn-outline-primary">View All</a>
            </div>
            <div class="card-body">
                <?php if (empty($recentCampaigns)): ?>
                    <p class="text-muted mb-0">No campaigns yet. <a href="<?= url('client/campaigns.php') ?>">Create your first campaign</a></p>
                <?php else: ?>
                    <div class="list-group list-group-flush">
                        <?php foreach ($recentCampaigns as $campaign): ?>
                            <div class="list-group-item">
                                <div class="d-flex w-100 justify-content-between">
                                    <h6 class="mb-1"><?= escape($campaign['campaign_name']) ?></h6>
                                    <span class="badge bg-<?= $campaign['status'] === 'active' ? 'success' : 'secondary' ?>">
                                        <?= escape($campaign['status']) ?>
                                    </span>
                                </div>
                                <p class="mb-1"><small><?= ucfirst(str_replace('_', ' ', $campaign['campaign_type'])) ?></small></p>
                                <?php if ($campaign['budget']): ?>
                                    <small class="text-muted">Budget: <?= formatCurrency($campaign['budget']) ?></small>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <div class="col-md-6 mb-4">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Active Discount Codes</h5>
                <a href="<?= url('client/discounts.php') ?>" class="btn btn-sm btn-outline-primary">Manage</a>
            </div>
            <div class="card-body">
                <?php if (empty($activeDiscounts)): ?>
                    <p class="text-muted mb-0">No active discount codes. <a href="<?= url('client/discounts.php') ?>">Create one</a></p>
                <?php else: ?>
                    <div class="list-group list-group-flush">
                        <?php foreach ($activeDiscounts as $discount): ?>
                            <div class="list-group-item">
                                <div class="d-flex w-100 justify-content-between">
                                    <h6 class="mb-1"><?= escape($discount['code']) ?></h6>
                                    <span class="badge bg-success">Active</span>
                                </div>
                                <p class="mb-1">
                                    <?php if ($discount['discount_type'] === 'percentage'): ?>
                                        <?= number_format($discount['discount_value']) ?>% off
                                    <?php else: ?>
                                        <?= formatCurrency($discount['discount_value']) ?> off
                                    <?php endif; ?>
                                </p>
                                <small class="text-muted">
                                    Used: <?= $discount['usage_count'] ?> / <?= $discount['usage_limit'] ?? '∞' ?>
                                    <?php if ($discount['end_date']): ?>
                                        | Expires: <?= formatDate($discount['end_date']) ?>
                                    <?php endif; ?>
                                </small>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

