<?php
/**
 * CRM - Customer Relationship Management
 * Manage customers and email lists
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireClient();
$pageTitle = 'CRM - Customers & Email Lists';

$userId = $_SESSION['user_id'];
$db = getDB();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    require_once __DIR__ . '/../includes/csrf.php';
    
    $action = $_POST['action'] ?? '';
    
    if ($action === 'import_emails') {
        if (isset($_FILES['email_file']) && $_FILES['email_file']['error'] === UPLOAD_ERR_OK) {
            $file = $_FILES['email_file'];
            $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            
            if (!in_array($extension, ['csv', 'xlsx', 'xls'])) {
                setFlashMessage('danger', 'Invalid file type. Please upload CSV or Excel file.');
                header('Location: ' . url('client/crm.php'));
                exit;
            }
            
            $uploadDir = __DIR__ . '/../uploads/user_' . $userId . '/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $fileName = 'email_import_' . uniqid() . '_' . time() . '.' . $extension;
            $filePath = $uploadDir . $fileName;
            
            if (move_uploaded_file($file['tmp_name'], $filePath)) {
                // Process the file
                $imported = 0;
                $skipped = 0;
                $errors = [];
                
                if ($extension === 'csv') {
                    $handle = fopen($filePath, 'r');
                    if ($handle) {
                        $headers = fgetcsv($handle); // Skip header row
                        
                        // Auto-detect column mapping
                        $emailCol = null;
                        $firstNameCol = null;
                        $lastNameCol = null;
                        $phoneCol = null;
                        
                        foreach ($headers as $index => $header) {
                            $header = strtolower(trim($header));
                            if (in_array($header, ['email', 'e-mail', 'email address', 'email_address'])) {
                                $emailCol = $index;
                            } elseif (in_array($header, ['first name', 'firstname', 'first_name', 'fname', 'first'])) {
                                $firstNameCol = $index;
                            } elseif (in_array($header, ['last name', 'lastname', 'last_name', 'lname', 'last'])) {
                                $lastNameCol = $index;
                            } elseif (in_array($header, ['phone', 'phone number', 'phone_number', 'telephone', 'tel'])) {
                                $phoneCol = $index;
                            }
                        }
                        
                        if ($emailCol === null) {
                            // Try first column as email
                            $emailCol = 0;
                        }
                        
                        while (($row = fgetcsv($handle)) !== false) {
                            if (count($row) <= $emailCol) continue;
                            
                            $email = trim($row[$emailCol] ?? '');
                            $firstName = $firstNameCol !== null ? trim($row[$firstNameCol] ?? '') : '';
                            $lastName = $lastNameCol !== null ? trim($row[$lastNameCol] ?? '') : '';
                            $phone = $phoneCol !== null ? trim($row[$phoneCol] ?? '') : null;
                            
                            if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                                $skipped++;
                                continue;
                            }
                            
                            // Insert or update subscriber
                            $stmt = $db->prepare("
                                INSERT INTO email_subscribers (user_id, email, first_name, last_name, status)
                                VALUES (?, ?, ?, ?, 'subscribed')
                                ON DUPLICATE KEY UPDATE
                                    first_name = COALESCE(VALUES(first_name), first_name),
                                    last_name = COALESCE(VALUES(last_name), last_name),
                                    status = 'subscribed',
                                    unsubscribed_at = NULL
                            ");
                            
                            try {
                                $stmt->execute([$userId, $email, $firstName ?: null, $lastName ?: null]);
                                $imported++;
                                
                                // If phone provided, store in customers table or notes
                                if (!empty($phone)) {
                                    // Store in a notes/metadata field or separate table
                                    // For now, we'll store in a JSON field if available
                                }
                            } catch (PDOException $e) {
                                $skipped++;
                                $errors[] = "Error with email {$email}: " . $e->getMessage();
                            }
                        }
                        fclose($handle);
                    }
                } else {
                    // Excel file - use PhpSpreadsheet if available, otherwise show error
                    setFlashMessage('warning', 'Excel files require PhpSpreadsheet library. Please convert to CSV for now.');
                    header('Location: ' . url('client/crm.php'));
                    exit;
                }
                
                setFlashMessage('success', "Import complete: {$imported} emails imported, {$skipped} skipped.");
                if (!empty($errors)) {
                    setFlashMessage('warning', 'Some errors occurred during import.');
                }
            } else {
                setFlashMessage('danger', 'Failed to upload file.');
            }
        } else {
            setFlashMessage('danger', 'Please select a file to import.');
        }
        
        header('Location: ' . url('client/crm.php'));
        exit;
    } elseif ($action === 'delete_customer') {
        $customerId = intval($_POST['customer_id'] ?? 0);
        
        $stmt = $db->prepare("DELETE FROM email_subscribers WHERE id = ? AND user_id = ?");
        $stmt->execute([$customerId, $userId]);
        
        setFlashMessage('success', 'Customer removed successfully.');
        header('Location: ' . url('client/crm.php'));
        exit;
    } elseif ($action === 'update_status') {
        $customerId = intval($_POST['customer_id'] ?? 0);
        $status = $_POST['status'] ?? 'subscribed';
        
        $stmt = $db->prepare("
            UPDATE email_subscribers 
            SET status = ?, 
                unsubscribed_at = CASE WHEN ? = 'unsubscribed' THEN NOW() ELSE NULL END
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$status, $status, $customerId, $userId]);
        
        setFlashMessage('success', 'Customer status updated successfully.');
        header('Location: ' . url('client/crm.php'));
        exit;
    } elseif ($action === 'add_customer') {
        $email = trim($_POST['email'] ?? '');
        $firstName = trim($_POST['first_name'] ?? '');
        $lastName = trim($_POST['last_name'] ?? '');
        $phone = trim($_POST['phone'] ?? '');
        
        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            setFlashMessage('danger', 'Please provide a valid email address.');
            header('Location: ' . url('client/crm.php'));
            exit;
        }
        
        $stmt = $db->prepare("
            INSERT INTO email_subscribers (user_id, email, first_name, last_name, status)
            VALUES (?, ?, ?, ?, 'subscribed')
            ON DUPLICATE KEY UPDATE
                first_name = COALESCE(VALUES(first_name), first_name),
                last_name = COALESCE(VALUES(last_name), last_name),
                status = 'subscribed',
                unsubscribed_at = NULL
        ");
        
        try {
            $stmt->execute([$userId, $email, $firstName ?: null, $lastName ?: null]);
            setFlashMessage('success', 'Customer added successfully.');
        } catch (PDOException $e) {
            setFlashMessage('danger', 'Failed to add customer: ' . $e->getMessage());
        }
        
        header('Location: ' . url('client/crm.php'));
        exit;
    }
}

// Get search and filter parameters
$search = $_GET['search'] ?? '';
$statusFilter = $_GET['status'] ?? '';
$page = max(1, intval($_GET['page'] ?? 1));
$perPage = 50;
$offset = ($page - 1) * $perPage;

// Build query
$where = ['user_id = ?'];
$params = [$userId];

if (!empty($search)) {
    $where[] = "(email LIKE ? OR first_name LIKE ? OR last_name LIKE ?)";
    $searchTerm = '%' . $search . '%';
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

if (!empty($statusFilter)) {
    $where[] = "status = ?";
    $params[] = $statusFilter;
}

$whereClause = implode(' AND ', $where);

// Get total count
$stmt = $db->prepare("SELECT COUNT(*) FROM email_subscribers WHERE {$whereClause}");
$stmt->execute($params);
$totalCustomers = $stmt->fetchColumn();
$totalPages = ceil($totalCustomers / $perPage);

// Get customers
$stmt = $db->prepare("
    SELECT * FROM email_subscribers 
    WHERE {$whereClause}
    ORDER BY subscribed_at DESC
    LIMIT ? OFFSET ?
");
$params[] = $perPage;
$params[] = $offset;
$stmt->execute($params);
$customers = $stmt->fetchAll();

// Get statistics for email subscribers
$stmt = $db->prepare("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'subscribed' THEN 1 ELSE 0 END) as subscribed,
        SUM(CASE WHEN status = 'unsubscribed' THEN 1 ELSE 0 END) as unsubscribed,
        SUM(CASE WHEN status = 'bounced' THEN 1 ELSE 0 END) as bounced
    FROM email_subscribers
    WHERE user_id = ?
");
$stmt->execute([$userId]);
$stats = $stmt->fetch();

// Get lead statistics (from client_leads table)
// Since client_leads doesn't have user_id directly, we'll match by email_subscribers or campaign associations
// For now, we'll show all leads where email matches an email_subscriber of this user
$stmt = $db->prepare("
    SELECT 
        COUNT(*) as total_leads,
        SUM(CASE WHEN cl.status = 'new' THEN 1 ELSE 0 END) as new_leads,
        SUM(CASE WHEN cl.status = 'contacted' THEN 1 ELSE 0 END) as contacted,
        SUM(CASE WHEN cl.status = 'qualified' THEN 1 ELSE 0 END) as qualified,
        SUM(CASE WHEN cl.status = 'converted' THEN 1 ELSE 0 END) as converted
    FROM client_leads cl
    INNER JOIN email_subscribers es ON cl.lead_email = es.email
    WHERE es.user_id = ?
");
$stmt->execute([$userId]);
$leadStats = $stmt->fetch() ?: ['total_leads' => 0, 'new_leads' => 0, 'contacted' => 0, 'qualified' => 0, 'converted' => 0];

// Get recent leads
$stmt = $db->prepare("
    SELECT cl.*, 
           ap.partner_name,
           ap.partner_code,
           es.user_id
    FROM client_leads cl
    INNER JOIN email_subscribers es ON cl.lead_email = es.email
    LEFT JOIN affiliate_partners ap ON cl.partner_id = ap.id
    WHERE es.user_id = ?
    ORDER BY cl.created_at DESC
    LIMIT 10
");
$stmt->execute([$userId]);
$recentLeads = $stmt->fetchAll();

// Determine which tab to show
$tab = $_GET['tab'] ?? 'customers';

require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/../includes/csrf.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <h1><i class="bi bi-people"></i> CRM - Customers & Leads</h1>
            <div>
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#importModal">
                    <i class="bi bi-upload"></i> Import Email List
                </button>
                <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#addCustomerModal">
                    <i class="bi bi-person-plus"></i> Add Customer
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Tabs Navigation -->
<ul class="nav nav-tabs mb-4" role="tablist">
    <li class="nav-item">
        <a class="nav-link <?= $tab === 'customers' ? 'active' : '' ?>" href="?tab=customers">
            <i class="bi bi-people"></i> Email Subscribers (<?= number_format($stats['total'] ?? 0) ?>)
        </a>
    </li>
    <li class="nav-item">
        <a class="nav-link <?= $tab === 'leads' ? 'active' : '' ?>" href="?tab=leads">
            <i class="bi bi-person-lines-fill"></i> Leads (<?= number_format($leadStats['total_leads']) ?>)
        </a>
    </li>
</ul>

<!-- Statistics Cards -->
<?php if ($tab === 'customers'): ?>
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card text-white bg-primary">
            <div class="card-body">
                <h5 class="card-title">Total Customers</h5>
                <h2 class="mb-0"><?= number_format($stats['total'] ?? 0) ?></h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-success">
            <div class="card-body">
                <h5 class="card-title">Subscribed</h5>
                <h2 class="mb-0"><?= number_format($stats['subscribed'] ?? 0) ?></h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-warning">
            <div class="card-body">
                <h5 class="card-title">Unsubscribed</h5>
                <h2 class="mb-0"><?= number_format($stats['unsubscribed'] ?? 0) ?></h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-danger">
            <div class="card-body">
                <h5 class="card-title">Bounced</h5>
                <h2 class="mb-0"><?= number_format($stats['bounced'] ?? 0) ?></h2>
            </div>
        </div>
    </div>
</div>
<?php else: ?>
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card text-white bg-primary">
            <div class="card-body">
                <h5 class="card-title">Total Leads</h5>
                <h2 class="mb-0"><?= number_format($leadStats['total_leads']) ?></h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-info">
            <div class="card-body">
                <h5 class="card-title">New Leads</h5>
                <h2 class="mb-0"><?= number_format($leadStats['new_leads']) ?></h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-warning">
            <div class="card-body">
                <h5 class="card-title">Qualified</h5>
                <h2 class="mb-0"><?= number_format($leadStats['qualified']) ?></h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-success">
            <div class="card-body">
                <h5 class="card-title">Converted</h5>
                <h2 class="mb-0"><?= number_format($leadStats['converted']) ?></h2>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Filters and Search -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <input type="hidden" name="tab" value="<?= escape($tab) ?>">
                    <div class="col-md-6">
                        <input type="text" class="form-control" name="search" placeholder="Search by email, name..." value="<?= escape($search) ?>">
                    </div>
                    <div class="col-md-3">
                        <?php if ($tab === 'customers'): ?>
                        <select class="form-select" name="status">
                            <option value="">All Statuses</option>
                            <option value="subscribed" <?= $statusFilter === 'subscribed' ? 'selected' : '' ?>>Subscribed</option>
                            <option value="unsubscribed" <?= $statusFilter === 'unsubscribed' ? 'selected' : '' ?>>Unsubscribed</option>
                            <option value="bounced" <?= $statusFilter === 'bounced' ? 'selected' : '' ?>>Bounced</option>
                        </select>
                        <?php else: ?>
                        <select class="form-select" name="status">
                            <option value="">All Statuses</option>
                            <option value="new" <?= $statusFilter === 'new' ? 'selected' : '' ?>>New</option>
                            <option value="contacted" <?= $statusFilter === 'contacted' ? 'selected' : '' ?>>Contacted</option>
                            <option value="qualified" <?= $statusFilter === 'qualified' ? 'selected' : '' ?>>Qualified</option>
                            <option value="converted" <?= $statusFilter === 'converted' ? 'selected' : '' ?>>Converted</option>
                            <option value="lost" <?= $statusFilter === 'lost' ? 'selected' : '' ?>>Lost</option>
                        </select>
                        <?php endif; ?>
                    </div>
                    <div class="col-md-3">
                        <button type="submit" class="btn btn-primary w-100">
                            <i class="bi bi-search"></i> Search
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<?php if ($tab === 'leads'): ?>
<!-- Leads Table -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <?php if (empty($recentLeads)): ?>
                    <p class="text-muted text-center py-5">No leads found. Leads from your AcceleratedLeadz forms will appear here.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Email</th>
                                    <th>Name</th>
                                    <th>Phone</th>
                                    <th>Company</th>
                                    <th>Source</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recentLeads as $lead): ?>
                                    <tr>
                                        <td><?= escape($lead['lead_email']) ?></td>
                                        <td><?= escape($lead['lead_name'] ?? '-') ?></td>
                                        <td><?= escape($lead['lead_phone'] ?? '-') ?></td>
                                        <td><?= escape($lead['company_name'] ?? '-') ?></td>
                                        <td>
                                            <?php if ($lead['partner_name']): ?>
                                                <span class="badge bg-info"><?= escape($lead['partner_name']) ?></span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary"><?= escape($lead['lead_source'] ?? 'direct') ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?= 
                                                $lead['status'] === 'new' ? 'info' : 
                                                ($lead['status'] === 'qualified' ? 'warning' : 
                                                ($lead['status'] === 'converted' ? 'success' : 
                                                ($lead['status'] === 'lost' ? 'danger' : 'secondary')))
                                            ?>">
                                                <?= ucfirst($lead['status'] ?? 'new') ?>
                                            </span>
                                        </td>
                                        <td><?= formatDate($lead['created_at']) ?></td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <?php if ($lead['notes']): ?>
                                                    <button class="btn btn-info" title="<?= escape($lead['notes']) ?>">
                                                        <i class="bi bi-info-circle"></i>
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>
<?php else: ?>
<!-- Customers Table -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <?php if (empty($customers)): ?>
                    <p class="text-muted text-center py-5">No customers found. Import an email list or add customers manually.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Email</th>
                                    <th>Name</th>
                                    <th>Status</th>
                                    <th>Subscribed Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($customers as $customer): ?>
                                    <tr>
                                        <td><?= escape($customer['email']) ?></td>
                                        <td>
                                            <?php if ($customer['first_name'] || $customer['last_name']): ?>
                                                <?= escape(trim(($customer['first_name'] ?? '') . ' ' . ($customer['last_name'] ?? ''))) ?>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?= 
                                                $customer['status'] === 'subscribed' ? 'success' : 
                                                ($customer['status'] === 'unsubscribed' ? 'warning' : 'danger') 
                                            ?>">
                                                <?= ucfirst($customer['status']) ?>
                                            </span>
                                        </td>
                                        <td><?= formatDate($customer['subscribed_at']) ?></td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <button class="btn btn-info" data-bs-toggle="modal" data-bs-target="#editCustomerModal<?= $customer['id'] ?>">
                                                    <i class="bi bi-pencil"></i> Edit
                                                </button>
                                                <form method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to remove this customer?');">
                                                    <input type="hidden" name="action" value="delete_customer">
                                                    <input type="hidden" name="customer_id" value="<?= $customer['id'] ?>">
                                                    <?= csrfField() ?>
                                                    <button type="submit" class="btn btn-danger">
                                                        <i class="bi bi-trash"></i> Delete
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                    
                                    <!-- Edit Customer Modal -->
                                    <div class="modal fade" id="editCustomerModal<?= $customer['id'] ?>" tabindex="-1">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <div class="modal-header">
                                                    <h5 class="modal-title">Edit Customer</h5>
                                                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                                </div>
                                                <form method="POST">
                                                    <?= csrfField() ?>
                                                    <input type="hidden" name="action" value="update_status">
                                                    <input type="hidden" name="customer_id" value="<?= $customer['id'] ?>">
                                                    <div class="modal-body">
                                                        <div class="mb-3">
                                                            <label class="form-label">Email</label>
                                                            <input type="email" class="form-control" value="<?= escape($customer['email']) ?>" disabled>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label class="form-label">Name</label>
                                                            <input type="text" class="form-control" value="<?= escape(trim(($customer['first_name'] ?? '') . ' ' . ($customer['last_name'] ?? ''))) ?>" disabled>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label class="form-label">Status</label>
                                                            <select class="form-select" name="status">
                                                                <option value="subscribed" <?= $customer['status'] === 'subscribed' ? 'selected' : '' ?>>Subscribed</option>
                                                                <option value="unsubscribed" <?= $customer['status'] === 'unsubscribed' ? 'selected' : '' ?>>Unsubscribed</option>
                                                                <option value="bounced" <?= $customer['status'] === 'bounced' ? 'selected' : '' ?>>Bounced</option>
                                                            </select>
                                                        </div>
                                                    </div>
                                                    <div class="modal-footer">
                                                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                        <button type="submit" class="btn btn-primary">Update</button>
                                                    </div>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination -->
                    <?php if ($totalPages > 1): ?>
                        <nav aria-label="Page navigation">
                            <ul class="pagination justify-content-center">
                                <?php if ($page > 1): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($statusFilter) ?>">Previous</a>
                                    </li>
                                <?php endif; ?>
                                
                                <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                    <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                                        <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($statusFilter) ?>"><?= $i ?></a>
                                    </li>
                                <?php endfor; ?>
                                
                                <?php if ($page < $totalPages): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($statusFilter) ?>">Next</a>
                                    </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Import Email List Modal -->
<div class="modal fade" id="importModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bi bi-upload"></i> Import Email List</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" enctype="multipart/form-data">
                <?= csrfField() ?>
                <input type="hidden" name="action" value="import_emails">
                <div class="modal-body">
                    <div class="alert alert-info">
                        <div class="d-flex justify-content-between align-items-start">
                            <div>
                                <strong>File Format Requirements:</strong>
                                <ul class="mb-0">
                                    <li>CSV or Excel format (.csv, .xlsx, .xls)</li>
                                    <li>First row should contain column headers</li>
                                    <li>Required column: <strong>Email</strong> (or E-mail, Email Address)</li>
                                    <li>Optional columns: First Name, Last Name, Phone</li>
                                    <li>Duplicate emails will be updated (not duplicated)</li>
                                </ul>
                            </div>
                            <div class="ms-3">
                                <a href="<?= url('client/download_email_template.php') ?>" class="btn btn-sm btn-outline-primary" download>
                                    <i class="bi bi-download"></i> Download Template
                                </a>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="email_file" class="form-label">Select File <span class="text-danger">*</span></label>
                        <input type="file" class="form-control" id="email_file" name="email_file" accept=".csv,.xlsx,.xls" required>
                        <small class="text-muted">Maximum file size: 10MB. <a href="<?= url('client/download_email_template.php') ?>" class="text-decoration-none">Download CSV template</a> to see the correct format.</small>
                    </div>
                    
                    <div class="alert alert-warning">
                        <strong>Column Mapping:</strong> The system will automatically detect columns named:
                        <ul class="mb-0">
                            <li><strong>Email:</strong> email, e-mail, email address, email_address</li>
                            <li><strong>First Name:</strong> first name, firstname, first_name, fname, first</li>
                            <li><strong>Last Name:</strong> last name, lastname, last_name, lname, last</li>
                            <li><strong>Phone:</strong> phone, phone number, phone_number, telephone, tel</li>
                        </ul>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-upload"></i> Import Emails
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Add Customer Modal -->
<div class="modal fade" id="addCustomerModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bi bi-person-plus"></i> Add Customer</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <?= csrfField() ?>
                <input type="hidden" name="action" value="add_customer">
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="email" class="form-label">Email <span class="text-danger">*</span></label>
                        <input type="email" class="form-control" id="email" name="email" required>
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="first_name" class="form-label">First Name</label>
                            <input type="text" class="form-control" id="first_name" name="first_name">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="last_name" class="form-label">Last Name</label>
                            <input type="text" class="form-control" id="last_name" name="last_name">
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="phone" class="form-label">Phone (Optional)</label>
                        <input type="tel" class="form-control" id="phone" name="phone">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Add Customer</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

