<?php
/**
 * Client Campaigns Page
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/webhook_helper.php';

requireClient();
$pageTitle = 'Campaigns';

$userId = $_SESSION['user_id'];
$db = getDB();

// Handle campaign creation/updates
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        $campaignName = $_POST['campaign_name'] ?? '';
        $campaignType = $_POST['campaign_type'] ?? '';
        $budget = !empty($_POST['budget']) ? floatval($_POST['budget']) : null;
        $startDate = $_POST['start_date'] ?? null;
        $endDate = $_POST['end_date'] ?? null;
        $description = $_POST['description'] ?? '';
        
        $stmt = $db->prepare("
            INSERT INTO campaigns (user_id, campaign_name, campaign_type, budget, start_date, end_date, description, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, 'draft')
        ");
        $stmt->execute([$userId, $campaignName, $campaignType, $budget, $startDate, $endDate, $description]);
        $campaignId = $db->lastInsertId();
        
        // Trigger webhook for campaign creation
        triggerWebhookEvent('campaign.created', [
            'campaign_id' => $campaignId,
            'campaign_name' => $campaignName,
            'campaign_type' => $campaignType,
            'budget' => $budget,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'status' => 'draft'
        ]);
        
        setFlashMessage('success', 'Campaign created successfully!');
        header('Location: ' . url('client/campaigns.php'));
        exit;
    } elseif ($action === 'delete') {
        $campaignId = intval($_POST['campaign_id'] ?? 0);
        
        // Verify ownership
        $stmt = $db->prepare("SELECT * FROM campaigns WHERE id = ? AND user_id = ?");
        $stmt->execute([$campaignId, $userId]);
        $campaign = $stmt->fetch();
        
        if ($campaign) {
            // Get campaign details for webhook before deletion
            $campaignName = $campaign['campaign_name'];
            
            // Delete campaign (cascade will handle campaign_ads, email_campaigns should use SET NULL)
            $stmt = $db->prepare("DELETE FROM campaigns WHERE id = ? AND user_id = ?");
            $stmt->execute([$campaignId, $userId]);
            
            // Trigger webhook for campaign deletion
            triggerWebhookEvent('campaign.deleted', [
                'campaign_id' => $campaignId,
                'campaign_name' => $campaignName,
                'campaign_type' => $campaign['campaign_type'],
                'deleted_at' => date('c')
            ]);
            
            setFlashMessage('success', 'Campaign deleted successfully!');
        } else {
            setFlashMessage('danger', 'Campaign not found or you do not have permission to delete it.');
        }
        
        header('Location: ' . url('client/campaigns.php'));
        exit;
    } elseif ($action === 'update_status') {
        $campaignId = intval($_POST['campaign_id'] ?? 0);
        $status = $_POST['status'] ?? 'draft';
        
        // Get campaign details before update for webhook
        $stmt = $db->prepare("SELECT * FROM campaigns WHERE id = ? AND user_id = ?");
        $stmt->execute([$campaignId, $userId]);
        $campaign = $stmt->fetch();
        
        if ($campaign) {
            $oldStatus = $campaign['status'];
            
            // Update campaign status
            $stmt = $db->prepare("UPDATE campaigns SET status = ? WHERE id = ? AND user_id = ?");
            $stmt->execute([$status, $campaignId, $userId]);
            
            // Trigger webhook for campaign status update
            triggerWebhookEvent('campaign.updated', [
                'campaign_id' => $campaignId,
                'campaign_name' => $campaign['campaign_name'],
                'campaign_type' => $campaign['campaign_type'],
                'old_status' => $oldStatus,
                'new_status' => $status,
                'budget' => $campaign['budget']
            ]);
        }
        
        setFlashMessage('success', 'Campaign updated successfully!');
        header('Location: ' . url('client/campaigns.php'));
        exit;
    }
}

// Get all campaigns with ad counts
$stmt = $db->prepare("
    SELECT c.*, COUNT(ca.id) as ad_count
    FROM campaigns c
    LEFT JOIN campaign_ads ca ON c.id = ca.campaign_id AND ca.status = 'active'
    WHERE c.user_id = ? 
    GROUP BY c.id
    ORDER BY c.created_at DESC
");
$stmt->execute([$userId]);
$campaigns = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';

// Handle campaign view
$viewCampaignId = isset($_GET['view']) ? intval($_GET['view']) : 0;
$viewCampaign = null;

if ($viewCampaignId) {
    $stmt = $db->prepare("SELECT * FROM campaigns WHERE id = ? AND user_id = ?");
    $stmt->execute([$viewCampaignId, $userId]);
    $viewCampaign = $stmt->fetch();
    
    if ($viewCampaign) {
        // Get ads for this campaign
        $stmt = $db->prepare("
            SELECT ca.*, sa.ad_data 
            FROM campaign_ads ca
            LEFT JOIN saved_ads sa ON ca.saved_ad_id = sa.id
            WHERE ca.campaign_id = ? AND ca.status = 'active'
            ORDER BY ca.created_at DESC
        ");
        $stmt->execute([$viewCampaignId]);
        $campaignAds = $stmt->fetchAll();
        
        foreach ($campaignAds as &$ad) {
            if ($ad['ad_data']) {
                $ad['ad_data'] = json_decode($ad['ad_data'], true);
            }
        }
    }
}
?>

<?php if ($viewCampaign): ?>
    <!-- Campaign Detail View -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1><?= escape($viewCampaign['campaign_name']) ?></h1>
                    <p class="text-muted"><?= ucfirst(str_replace('_', ' ', $viewCampaign['campaign_type'])) ?> Campaign</p>
                </div>
                <div>
                    <a href="?view=" class="btn btn-secondary">
                        <i class="bi bi-arrow-left"></i> Back to Campaigns
                    </a>
                    <button type="button" class="btn btn-danger delete-campaign-btn" data-campaign-id="<?= $viewCampaign['id'] ?>" data-campaign-name="<?= escape($viewCampaign['campaign_name']) ?>">
                        <i class="bi bi-trash"></i> Delete Campaign
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <div class="row mb-4">
        <div class="col-md-4">
            <div class="card">
                <div class="card-body">
                    <h6>Status</h6>
                    <span class="badge bg-<?= $viewCampaign['status'] === 'active' ? 'success' : ($viewCampaign['status'] === 'paused' ? 'warning' : 'secondary') ?>">
                        <?= escape($viewCampaign['status']) ?>
                    </span>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card">
                <div class="card-body">
                    <h6>Budget</h6>
                    <p class="mb-0"><?= $viewCampaign['budget'] ? formatCurrency($viewCampaign['budget']) : 'N/A' ?></p>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card">
                <div class="card-body">
                    <h6>Ads</h6>
                    <p class="mb-0"><strong><?= count($campaignAds) ?></strong> active ad(s)</p>
                </div>
            </div>
        </div>
    </div>
    
    <?php if (!empty($campaignAds)): ?>
        <div class="row">
            <div class="col-12">
                <h3>Campaign Ads</h3>
                <div class="row">
                    <?php foreach ($campaignAds as $ad): 
                        $adData = $ad['ad_data'] ?? [];
                        // Generate URL for acceleratedadz.php
                        $adUrl = url('acceleratedadz.php') . '?campaign_ad=' . $ad['id'] . '&source=campaign';
                    ?>
                        <div class="col-md-6 col-lg-4 mb-4">
                            <div class="card">
                                <div class="card-header d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="mb-0"><?= escape($ad['ad_name']) ?></h6>
                                        <small class="text-muted"><?= escape($ad['ad_platform'] ?? 'N/A') ?></small>
                                    </div>
                                    <a href="<?= $adUrl ?>" target="_blank" class="btn btn-sm btn-primary" title="View Ad">
                                        <i class="bi bi-eye"></i>
                                    </a>
                                </div>
                                <div class="card-body">
                                    <?php if ($adData['headline'] || $adData['productImage']): ?>
                                        <a href="<?= $adUrl ?>" target="_blank" style="text-decoration: none; color: inherit;">
                                            <div style="background: <?= escape($adData['bgColor'] ?? '#ffffff') ?>; padding: 15px; border-radius: 8px; min-height: 200px; text-align: center; border: 1px solid #ddd; transition: transform 0.2s;">
                                                <?php if ($adData['productImage']): ?>
                                                    <img src="<?= escape($adData['productImage']) ?>" alt="<?= escape($adData['headline'] ?? 'Ad') ?>" style="max-width: 100%; max-height: 150px; height: auto; margin-bottom: 10px;">
                                                <?php endif; ?>
                                                <?php if ($adData['headline']): ?>
                                                    <h5 style="color: <?= escape($adData['textColor'] ?? '#000000') ?>; margin: 10px 0;"><?= escape($adData['headline']) ?></h5>
                                                <?php endif; ?>
                                                <?php if ($adData['description']): ?>
                                                    <p style="color: <?= escape($adData['textColor'] ?? '#000000') ?>; margin: 10px 0;"><?= escape($adData['description']) ?></p>
                                                <?php endif; ?>
                                                <?php if ($adData['cta']): ?>
                                                    <button style="background: <?= escape($adData['ctaColor'] ?? '#007bff') ?>; color: white; border: none; padding: 8px 20px; border-radius: 5px; margin-top: 10px;">
                                                        <?= escape($adData['cta']) ?>
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        </a>
                                        <div class="mt-2">
                                            <a href="<?= $adUrl ?>" target="_blank" class="btn btn-sm btn-outline-primary w-100">
                                                <i class="bi bi-box-arrow-up-right"></i> View Live Ad
                                            </a>
                                        </div>
                                    <?php else: ?>
                                        <p class="text-muted">Ad preview not available</p>
                                        <a href="<?= $adUrl ?>" target="_blank" class="btn btn-sm btn-outline-primary">
                                            <i class="bi bi-eye"></i> View Ad
                                        </a>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    <?php else: ?>
        <div class="alert alert-info">
            <i class="bi bi-info-circle"></i> No ads in this campaign yet. Use the Ad Builder to create ads and add them to this campaign.
        </div>
    <?php endif; ?>
    
<?php else: ?>
    <!-- Campaign List View -->
<div class="row">
    <div class="col-12 mb-4">
        <div class="d-flex justify-content-between align-items-center">
            <h1>Campaigns</h1>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createCampaignModal">
                <i class="bi bi-plus-circle"></i> Create Campaign
            </button>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <?php if (empty($campaigns)): ?>
                    <p class="text-muted text-center py-5">No campaigns yet. Create your first campaign!</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Campaign Name</th>
                                    <th>Type</th>
                                    <th>Status</th>
                                    <th>Budget</th>
                                    <th>Ads</th>
                                    <th>Start Date</th>
                                    <th>End Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($campaigns as $campaign): ?>
                                    <tr>
                                        <td><strong><?= escape($campaign['campaign_name']) ?></strong></td>
                                        <td><?= ucfirst(str_replace('_', ' ', $campaign['campaign_type'])) ?></td>
                                        <td>
                                            <span class="badge bg-<?= 
                                                $campaign['status'] === 'active' ? 'success' : 
                                                ($campaign['status'] === 'paused' ? 'warning' : 'secondary') 
                                            ?>">
                                                <?= escape($campaign['status']) ?>
                                            </span>
                                        </td>
                                        <td><?= $campaign['budget'] ? formatCurrency($campaign['budget']) : 'N/A' ?></td>
                                        <td>
                                            <?php if ($campaign['ad_count'] > 0): ?>
                                                <button class="btn btn-sm btn-info view-ads-btn" data-campaign-id="<?= $campaign['id'] ?>" data-campaign-name="<?= escape($campaign['campaign_name']) ?>">
                                                    <i class="bi bi-image"></i> <?= $campaign['ad_count'] ?> Ad(s)
                                                </button>
                                            <?php else: ?>
                                                <span class="text-muted">No ads</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= $campaign['start_date'] ? formatDate($campaign['start_date']) : 'N/A' ?></td>
                                        <td><?= $campaign['end_date'] ? formatDate($campaign['end_date']) : 'No end date' ?></td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <a href="?view=<?= $campaign['id'] ?>" class="btn btn-sm btn-info" title="View Details">
                                                    <i class="bi bi-eye"></i> View
                                                </a>
                                            <form method="POST" class="d-inline">
                                                <input type="hidden" name="action" value="update_status">
                                                <input type="hidden" name="campaign_id" value="<?= $campaign['id'] ?>">
                                                <?php if ($campaign['status'] === 'active'): ?>
                                                    <input type="hidden" name="status" value="paused">
                                                    <button type="submit" class="btn btn-sm btn-warning">Pause</button>
                                                <?php elseif ($campaign['status'] === 'draft' || $campaign['status'] === 'paused'): ?>
                                                    <input type="hidden" name="status" value="active">
                                                    <button type="submit" class="btn btn-sm btn-success">Activate</button>
                                                <?php endif; ?>
                                            </form>
                                            <button type="button" class="btn btn-sm btn-danger delete-campaign-btn" data-campaign-id="<?= $campaign['id'] ?>" data-campaign-name="<?= escape($campaign['campaign_name']) ?>" title="Delete Campaign">
                                                <i class="bi bi-trash"></i> Delete
                                            </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>
</div>

<?php endif; ?>

<!-- Create Campaign Modal -->
<div class="modal fade" id="createCampaignModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create New Campaign</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="create">
                    
                    <div class="mb-3">
                        <label for="campaign_name" class="form-label">Campaign Name</label>
                        <input type="text" class="form-control" id="campaign_name" name="campaign_name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="campaign_type" class="form-label">Campaign Type</label>
                        <select class="form-select" id="campaign_type" name="campaign_type" required>
                            <option value="">Select type...</option>
                            <option value="email" selected>Email Marketing (Recommended)</option>
                            <option value="google_ads">Google Ads</option>
                            <option value="facebook_ads">Facebook Ads</option>
                            <option value="direct_mail">Direct Mail</option>
                            <option value="multi_channel">Multi-Channel</option>
                        </select>
                        <small class="text-muted">Email Marketing is the default campaign type for easier workflow.</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="budget" class="form-label">Budget (Optional)</label>
                        <input type="number" step="0.01" class="form-control" id="budget" name="budget">
                    </div>
                    
                    <div class="mb-3">
                        <label for="start_date" class="form-label">Start Date</label>
                        <input type="date" class="form-control" id="start_date" name="start_date" value="<?= date('Y-m-d') ?>">
                    </div>
                    
                    <div class="mb-3">
                        <label for="end_date" class="form-label">End Date (Optional)</label>
                        <input type="date" class="form-control" id="end_date" name="end_date">
                    </div>
                    
                    <div class="mb-3">
                        <label for="description" class="form-label">Description (Optional)</label>
                        <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Campaign</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- View Ads Modal -->
<div class="modal fade" id="viewAdsModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Ads in Campaign - <span id="adsCampaignName"></span></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="adsContent">
                <p class="text-center"><i class="spinner-border"></i> Loading ads...</p>
            </div>
        </div>
    </div>
</div>

<!-- Delete Campaign Confirmation Modal -->
<div class="modal fade" id="deleteCampaignModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Delete Campaign</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the campaign <strong id="deleteCampaignName"></strong>?</p>
                <p class="text-danger"><small>This action cannot be undone. All ads linked to this campaign will also be removed.</small></p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <form method="POST" id="deleteCampaignForm" class="d-inline">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="campaign_id" id="deleteCampaignId">
                    <button type="submit" class="btn btn-danger">Delete Campaign</button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Delete campaign handlers
    document.querySelectorAll('.delete-campaign-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const campaignId = this.dataset.campaignId;
            const campaignName = this.dataset.campaignName;
            document.getElementById('deleteCampaignId').value = campaignId;
            document.getElementById('deleteCampaignName').textContent = campaignName;
            new bootstrap.Modal(document.getElementById('deleteCampaignModal')).show();
        });
    });
    
    document.querySelectorAll('.view-ads-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const campaignId = this.dataset.campaignId;
            const campaignName = this.dataset.campaignName;
            loadCampaignAds(campaignId, campaignName);
        });
    });
    
    function loadCampaignAds(campaignId, campaignName) {
        const modal = document.getElementById('viewAdsModal');
        const content = document.getElementById('adsContent');
        document.getElementById('adsCampaignName').textContent = campaignName;
        content.innerHTML = '<p class="text-center"><i class="spinner-border"></i> Loading ads...</p>';
        
        fetch(`<?= url('api/ad_builder.php') ?>?action=list_campaign_ads&campaign_id=${campaignId}`)
            .then(async r => {
                const text = await r.text();
                if (!r.ok) {
                    throw new Error(`HTTP ${r.status}: ${text}`);
                }
                return JSON.parse(text);
            })
            .then(data => {
                if (data.success && data.data && data.data.length > 0) {
                    let html = '<div class="row">';
                    data.data.forEach(ad => {
                        const adData = ad.ad_data || {};
                        html += '<div class="col-md-6 mb-4">';
                        html += '<div class="card">';
                        html += '<div class="card-header">';
                        html += `<h6 class="mb-0">${escapeHtml(ad.ad_name)}</h6>`;
                        html += `<small class="text-muted">${ad.ad_platform || 'N/A'}</small>`;
                        html += '</div>';
                        html += '<div class="card-body">';
                        
                        // Render ad preview
                        if (adData.bgImage || adData.headline) {
                            const bgColor = adData.bgColor || '#ffffff';
                            const textColor = adData.textColor || '#000000';
                            html += `<div style="background: ${bgColor}; background-image: ${adData.bgImage ? `url('${adData.bgImage}')` : 'none'}; background-size: cover; padding: 20px; border-radius: 8px; min-height: 200px; display: flex; flex-direction: column; justify-content: center; align-items: center; text-align: center;">`;
                            if (adData.productImage) {
                                html += `<img src="${escapeHtml(adData.productImage)}" style="max-width: 40%; max-height: 100px; margin-bottom: 10px;" onerror="this.style.display='none'">`;
                            }
                            if (adData.headline) {
                                html += `<h4 style="color: ${textColor}; margin: 10px 0;">${escapeHtml(adData.headline)}</h4>`;
                            }
                            if (adData.description) {
                                html += `<p style="color: ${textColor}; margin: 10px 0;">${escapeHtml(adData.description)}</p>`;
                            }
                            if (adData.cta) {
                                html += `<button style="background: ${adData.ctaColor || '#007bff'}; color: white; border: none; padding: 10px 20px; border-radius: 5px; margin-top: 10px;">${escapeHtml(adData.cta)}</button>`;
                            }
                            html += '</div>';
                        } else {
                            html += '<p class="text-muted">Ad preview not available</p>';
                        }
                        
                        html += '</div>';
                        html += '</div>';
                        html += '</div>';
                    });
                    html += '</div>';
                    content.innerHTML = html;
                } else {
                    content.innerHTML = '<p class="text-center text-muted py-5">No ads found for this campaign.</p>';
                }
                new bootstrap.Modal(modal).show();
            })
            .catch(err => {
                console.error('Error loading ads:', err);
                content.innerHTML = `<p class="text-center text-danger">Error loading ads: ${err.message}</p>`;
                new bootstrap.Modal(modal).show();
            });
    }
    
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

