<?php
/**
 * Client Analytics Page
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireClient();
$pageTitle = 'Analytics';

$userId = $_SESSION['user_id'];
$db = getDB();

// Get date range filter
$days = isset($_GET['days']) ? intval($_GET['days']) : 30;
$days = max(7, min(365, $days)); // Between 7 and 365 days

// Get metrics summary
$metrics = getUserMetricsSummary($userId, $days);

// Get campaign performance
$adPerformance = getCampaignPerformanceSummary($userId, $days);

// Get top traffic sources
$stmt = $db->prepare("
    SELECT utm_source, COUNT(*) as visits, COUNT(DISTINCT session_id) as unique_visitors
    FROM user_metrics
    WHERE user_id = ? AND visited_at >= DATE_SUB(NOW(), INTERVAL ? DAY) AND utm_source IS NOT NULL
    GROUP BY utm_source
    ORDER BY visits DESC
    LIMIT 10
");
$stmt->execute([$userId, $days]);
$trafficSources = $stmt->fetchAll();

// Get device breakdown
$stmt = $db->prepare("
    SELECT device_type, COUNT(*) as visits, COUNT(DISTINCT session_id) as unique_visitors
    FROM user_metrics
    WHERE user_id = ? AND visited_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
    GROUP BY device_type
    ORDER BY visits DESC
");
$stmt->execute([$userId, $days]);
$deviceBreakdown = $stmt->fetchAll();

// Get conversions by type
$stmt = $db->prepare("
    SELECT conversion_type, COUNT(*) as count, SUM(conversion_value) as total_value
    FROM user_metrics
    WHERE user_id = ? AND visited_at >= DATE_SUB(NOW(), INTERVAL ? DAY) AND conversion_type IS NOT NULL
    GROUP BY conversion_type
    ORDER BY total_value DESC
");
$stmt->execute([$userId, $days]);
$conversions = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row">
    <div class="col-12 mb-4">
        <div class="d-flex justify-content-between align-items-center">
            <h1>Analytics</h1>
            <div>
                <select class="form-select d-inline-block" style="width: auto;" onchange="window.location.href='?days=' + this.value">
                    <option value="7" <?= $days == 7 ? 'selected' : '' ?>>Last 7 days</option>
                    <option value="30" <?= $days == 30 ? 'selected' : '' ?>>Last 30 days</option>
                    <option value="90" <?= $days == 90 ? 'selected' : '' ?>>Last 90 days</option>
                    <option value="365" <?= $days == 365 ? 'selected' : '' ?>>Last year</option>
                </select>
            </div>
        </div>
    </div>
</div>

<!-- Key Metrics -->
<div class="row mb-4">
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-primary">
            <div class="card-body">
                <h5 class="card-title">Total Visits</h5>
                <h2 class="mb-0"><?= number_format($metrics['total_visits'] ?? 0) ?></h2>
                <small>Last <?= $days ?> days</small>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-success">
            <div class="card-body">
                <h5 class="card-title">Unique Visitors</h5>
                <h2 class="mb-0"><?= number_format($metrics['unique_visitors'] ?? 0) ?></h2>
                <small>Last <?= $days ?> days</small>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-info">
            <div class="card-body">
                <h5 class="card-title">Conversions</h5>
                <h2 class="mb-0"><?= number_format($metrics['conversions'] ?? 0) ?></h2>
                <small>Last <?= $days ?> days</small>
            </div>
        </div>
    </div>
    <div class="col-md-3 mb-3">
        <div class="card text-white bg-warning">
            <div class="card-body">
                <h5 class="card-title">Revenue</h5>
                <h2 class="mb-0"><?= formatCurrency($metrics['total_revenue'] ?? 0) ?></h2>
                <small>Last <?= $days ?> days</small>
            </div>
        </div>
    </div>
</div>

<!-- Traffic Sources -->
<div class="row mb-4">
    <div class="col-md-6">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Top Traffic Sources</h5>
            </div>
            <div class="card-body">
                <?php if (empty($trafficSources)): ?>
                    <p class="text-muted mb-0">No traffic source data available.</p>
                <?php else: ?>
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Source</th>
                                <th>Visits</th>
                                <th>Unique Visitors</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($trafficSources as $source): ?>
                                <tr>
                                    <td><?= escape($source['utm_source']) ?></td>
                                    <td><?= number_format($source['visits']) ?></td>
                                    <td><?= number_format($source['unique_visitors']) ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <div class="col-md-6">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Device Breakdown</h5>
            </div>
            <div class="card-body">
                <?php if (empty($deviceBreakdown)): ?>
                    <p class="text-muted mb-0">No device data available.</p>
                <?php else: ?>
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Device</th>
                                <th>Visits</th>
                                <th>Unique Visitors</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($deviceBreakdown as $device): ?>
                                <tr>
                                    <td><?= ucfirst($device['device_type']) ?></td>
                                    <td><?= number_format($device['visits']) ?></td>
                                    <td><?= number_format($device['unique_visitors']) ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Conversions -->
<?php if (!empty($conversions)): ?>
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Conversions by Type</h5>
            </div>
            <div class="card-body">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Conversion Type</th>
                            <th>Count</th>
                            <th>Total Value</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($conversions as $conv): ?>
                            <tr>
                                <td><?= escape($conv['conversion_type']) ?></td>
                                <td><?= number_format($conv['count']) ?></td>
                                <td><?= formatCurrency($conv['total_value'] ?? 0) ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Ad Performance -->
<?php if (!empty($adPerformance)): ?>
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Ad Performance (Last <?= $days ?> Days)</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-striped">
                        <thead>
                            <tr>
                                <th>Platform</th>
                                <th>Impressions</th>
                                <th>Clicks</th>
                                <th>CTR</th>
                                <th>Conversions</th>
                                <th>Cost</th>
                                <th>Revenue</th>
                                <th>ROAS</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($adPerformance as $perf): ?>
                                <tr>
                                    <td><?= ucfirst(str_replace('_', ' ', $perf['ad_platform'])) ?></td>
                                    <td><?= number_format($perf['total_impressions']) ?></td>
                                    <td><?= number_format($perf['total_clicks']) ?></td>
                                    <td><?= number_format($perf['ctr'], 2) ?>%</td>
                                    <td><?= number_format($perf['total_conversions']) ?></td>
                                    <td><?= formatCurrency($perf['total_cost']) ?></td>
                                    <td><?= formatCurrency($perf['total_revenue']) ?></td>
                                    <td>
                                        <?php 
                                        $roas = $perf['total_cost'] > 0 ? ($perf['total_revenue'] / $perf['total_cost']) : 0;
                                        echo number_format($roas, 2);
                                        ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Charts Section -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Revenue & Cost Trends</h5>
            </div>
            <div class="card-body">
                <canvas id="revenueChart" height="100"></canvas>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const days = <?= $days ?>;
    
    // Fetch time series data
    fetch(`<?= url('api/comprehensive_analytics.php') ?>?action=time_series&metric_type=revenue&start_date=<?= date('Y-m-d', strtotime("-{$days} days")) ?>&end_date=<?= date('Y-m-d') ?>`)
        .then(r => r.json())
                .then(data => {
            if (data.success && data.data.length > 0) {
                const ctx = document.getElementById('revenueChart');
                if (ctx) {
                    new Chart(ctx, {
                        type: 'line',
                        data: {
                            labels: data.data.map(d => d.period),
                            datasets: [{
                                label: 'Revenue',
                                data: data.data.map(d => parseFloat(d.value || 0)),
                                borderColor: 'rgb(75, 192, 192)',
                                backgroundColor: 'rgba(75, 192, 192, 0.2)',
                                tension: 0.1
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: true,
                            plugins: {
                                title: {
                                    display: true,
                                    text: 'Revenue Over Time'
                                },
                                legend: {
                                    display: true
                                }
                            },
                            scales: {
                                y: {
                                    beginAtZero: true
                                }
                            }
                        }
                    });
                }
            } else {
                // Show message if no data
                const ctx = document.getElementById('revenueChart');
                if (ctx) {
                    ctx.getContext('2d').fillText('No revenue data available', 10, 20);
                }
            }
        })
        .catch(err => {
            console.error('Error loading revenue data:', err);
        });
    
    // Fetch cost data
    fetch(`<?= url('api/comprehensive_analytics.php') ?>?action=time_series&metric_type=cost&start_date=<?= date('Y-m-d', strtotime("-{$days} days")) ? Arkivert>&end_date=<?= date('Y-m-d') ?>`)
        .then(r => r.json())
        .then(data => {
            if (data.success && data.data && data.data.length > 0) {
                const ctx = document.getElementById('revenueChart');
                if (ctx) {
                    const chartInstance = Chart.getChart(ctx);
                    if (chartInstance) {
                        chartInstance.data.datasets.push({
                            label: 'Cost',
                            data: data.data.map(d => parseFloat(d.value || 0)),
                            borderColor: 'rgb(255, 99, 132)',
                            backgroundColor: 'rgba(255, 99, 132, 0.2)',
                            tension: 0.1
                        });
                        chartInstance.update();
                    }
                }
            }
        })
        .catch(err => {
            console.error('Error loading cost data:', err);
        });
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

