<?php
/**
 * Ad Builder - Visual Ad Creation Platform
 * Industry templates: Retail, Food, Beauty, Fashion, Service, Custom
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireClient();
$pageTitle = 'Ad Builder';

$userId = $_SESSION['user_id'];
$db = getDB();

// Get campaigns for adding ads (include all types that can have ads)
$stmt = $db->prepare("
    SELECT id, campaign_name, campaign_type 
    FROM campaigns 
    WHERE user_id = ? AND campaign_type IN ('google_ads', 'facebook_ads', 'multi_channel', 'email')
    ORDER BY campaign_name
");
$stmt->execute([$userId]);
$campaigns = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/../includes/csrf.php';

// Generate CSRF token if not exists
if (!isset($_SESSION['csrf_token'])) {
    generateCSRFToken();
}
?>

<?= csrfField() ?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <h1><i class="bi bi-palette"></i> Ad Builder</h1>
            <div>
                <button class="btn btn-success" id="saveAdBtn">
                    <i class="bi bi-save"></i> Save Ad
                </button>
                <button class="btn btn-primary" id="addToCampaignBtn">
                    <i class="bi bi-plus-circle"></i> Add to Campaign
                </button>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Template Selection Sidebar -->
    <div class="col-md-3">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Industry Templates</h5>
            </div>
            <div class="card-body">
                <div class="list-group">
                    <button class="list-group-item list-group-item-action template-btn active" data-template="retail">
                        <i class="bi bi-shop"></i> Retail
                    </button>
                    <button class="list-group-item list-group-item-action template-btn" data-template="food">
                        <i class="bi bi-cup-hot"></i> Food & Restaurant
                    </button>
                    <button class="list-group-item list-group-item-action template-btn" data-template="beauty">
                        <i class="bi bi-star"></i> Beauty & Cosmetics
                    </button>
                    <button class="list-group-item list-group-item-action template-btn" data-template="fashion">
                        <i class="bi bi-bag"></i> Fashion
                    </button>
                    <button class="list-group-item list-group-item-action template-btn" data-template="service">
                        <i class="bi bi-tools"></i> Service Industry
                    </button>
                    <button class="list-group-item list-group-item-action template-btn" data-template="custom">
                        <i class="bi bi-pencil"></i> Custom (Blank)
                    </button>
                </div>
            </div>
        </div>

        <!-- Ad Settings -->
        <div class="card mt-3">
            <div class="card-header">
                <h6 class="mb-0">Ad Settings</h6>
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <label class="form-label">Ad Name</label>
                    <input type="text" class="form-control" id="adName" value="My Ad">
                </div>
                <div class="mb-3">
                    <label class="form-label">Platform</label>
                    <select class="form-select" id="adPlatform">
                        <option value="google_ads">Google Ads</option>
                        <option value="facebook_ads">Facebook Ads</option>
                    </select>
                </div>
                <div class="mb-3">
                    <label class="form-label">Ad Size</label>
                    <select class="form-select" id="adSize">
                        <option value="1200x628">Facebook Standard (1200x628)</option>
                        <option value="1080x1080">Square (1080x1080)</option>
                        <option value="1200x1200">Instagram Square (1200x1200)</option>
                        <option value="1080x1920">Stories (1080x1920)</option>
                        <option value="300x250">Google Banner (300x250)</option>
                        <option value="728x90">Google Leaderboard (728x90)</option>
                        <option value="160x600">Google Skyscraper (160x600)</option>
                    </select>
                </div>
            </div>
        </div>
    </div>

    <!-- Ad Canvas -->
    <div class="col-md-6">
        <div class="card">
            <div class="card-header d-flex justify-content-between">
                <h5 class="mb-0">Preview</h5>
                <div>
                    <button class="btn btn-sm btn-secondary" id="downloadBtn">
                        <i class="bi bi-download"></i> Download
                    </button>
                </div>
            </div>
            <div class="card-body">
                <div id="adCanvas" style="
                    width: 100%;
                    max-width: 600px;
                    margin: 0 auto;
                    background: #fff;
                    border: 2px solid #ddd;
                    border-radius: 8px;
                    overflow: hidden;
                    position: relative;
                    aspect-ratio: 16/9;
                    min-height: 350px;
                ">
                    <!-- Ad content will be rendered here -->
                </div>
            </div>
        </div>
    </div>

    <!-- Editor Panel -->
    <div class="col-md-3">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Edit Ad</h5>
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <label class="form-label">Headline</label>
                    <input type="text" class="form-control" id="headline" placeholder="Enter headline">
                </div>
                <div class="mb-3">
                    <label class="form-label">Description</label>
                    <textarea class="form-control" id="description" rows="3" placeholder="Enter description"></textarea>
                </div>
                <div class="mb-3">
                    <label class="form-label">Call-to-Action</label>
                    <input type="text" class="form-control" id="cta" placeholder="Shop Now, Learn More, etc.">
                </div>
                <div class="mb-3">
                    <label class="form-label">CTA Button Color</label>
                    <input type="color" class="form-control form-control-color" id="ctaColor" value="#007bff">
                </div>
                <div class="mb-3">
                    <label class="form-label">Background Color</label>
                    <input type="color" class="form-control form-control-color" id="bgColor" value="#ffffff">
                </div>
                <div class="mb-3">
                    <label class="form-label">Text Color</label>
                    <input type="color" class="form-control form-control-color" id="textColor" value="#000000">
                </div>
                <div class="mb-3">
                    <label class="form-label">Background Image URL (Optional)</label>
                    <input type="url" class="form-control" id="bgImage" placeholder="https://...">
                    <small class="form-text text-muted">Or upload: <input type="file" id="bgImageFile" accept="image/*"></small>
                </div>
                <div class="mb-3">
                    <label class="form-label">Logo/Product Image URL</label>
                    <input type="url" class="form-control" id="productImage" placeholder="https://...">
                    <small class="form-text text-muted">Or upload: <input type="file" id="productImageFile" accept="image/*"></small>
                </div>
                <div class="mb-3">
                    <label class="form-label">Destination URL (Link)</label>
                    <input type="url" class="form-control" id="destinationUrl" placeholder="https://your-website.com/product">
                    <small class="form-text text-muted">Where users will go when they click the ad</small>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Campaign Selection Modal -->
<div class="modal fade" id="campaignModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add Ad to Campaign</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label class="form-label">Select Campaign</label>
                    <select class="form-select" id="selectedCampaign">
                        <option value="">Choose a campaign...</option>
                        <?php foreach ($campaigns as $campaign): ?>
                            <option value="<?= $campaign['id'] ?>">
                                <?= escape($campaign['campaign_name']) ?> (<?= ucfirst(str_replace('_', ' ', $campaign['campaign_type'])) ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="confirmAddCampaign">Add to Campaign</button>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/html2canvas@1.4.1/dist/html2canvas.min.js"></script>
<script>
const adTemplates = {
    retail: {
        headline: "Shop the Latest Deals!",
        description: "Discover amazing products at unbeatable prices. Limited time offer!",
        cta: "Shop Now",
        ctaColor: "#ff6b6b",
        bgColor: "#ffffff",
        textColor: "#2c3e50",
        bgImage: "",
        style: "modern"
    },
    food: {
        headline: "Fresh & Delicious",
        description: "Experience mouth-watering flavors that will tantalize your taste buds.",
        cta: "Order Now",
        ctaColor: "#ff9800",
        bgColor: "#fff5e6",
        textColor: "#8b4513",
        bgImage: "",
        style: "warm"
    },
    beauty: {
        headline: "Feel Beautiful Inside & Out",
        description: "Transform your beauty routine with our premium collection.",
        cta: "Explore Products",
        ctaColor: "#e91e63",
        bgColor: "#fce4ec",
        textColor: "#880e4f",
        bgImage: "",
        style: "elegant"
    },
    fashion: {
        headline: "Style That Speaks",
        description: "Trendsetting fashion pieces for the modern you.",
        cta: "Shop Collection",
        ctaColor: "#000000",
        bgColor: "#f5f5f5",
        textColor: "#212121",
        bgImage: "",
        style: "minimalist"
    },
    service: {
        headline: "Expert Service You Can Trust",
        description: "Professional solutions tailored to your needs.",
        cta: "Get Started",
        ctaColor: "#2196f3",
        bgColor: "#ffffff",
        textColor: "#1565c0",
        bgImage: "",
        style: "professional"
    },
    custom: {
        headline: "Your Headline Here",
        description: "Your description goes here",
        cta: "Learn More",
        ctaColor: "#007bff",
        bgColor: "#ffffff",
        textColor: "#000000",
        bgImage: "",
        style: "custom"
    }
};

// Initialize currentAd with empty fields - user must enter their own data
let currentAd = {
    template: 'custom',
    headline: '',
    description: '',
    cta: '',
    destinationUrl: '',
    ctaColor: '#007bff',
    bgColor: '#ffffff',
    textColor: '#000000',
    bgImage: '',
    productImage: '',
    name: 'My Ad',
    platform: 'google_ads',
    size: '1200x628'
};

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    // Template selection
    document.querySelectorAll('.template-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            document.querySelectorAll('.template-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            
            const template = this.dataset.template;
            loadTemplate(template);
        });
    });
    
    // Editor inputs - collect all form values including destinationUrl
    ['headline', 'description', 'cta', 'ctaColor', 'bgColor', 'textColor', 'bgImage', 'productImage', 'destinationUrl', 'adSize'].forEach(id => {
        const element = document.getElementById(id);
        if (element) {
            element.addEventListener('input', updateAd);
            element.addEventListener('change', updateAd);
        }
    });
    
    // File upload handlers
    document.getElementById('bgImageFile').addEventListener('change', function() {
        uploadFile(this.files[0], 'bgImage');
    });
    
    document.getElementById('productImageFile').addEventListener('change', function() {
        uploadFile(this.files[0], 'productImage');
    });
    
    document.getElementById('adName').addEventListener('input', function() {
        currentAd.name = this.value;
    });
    
    document.getElementById('adPlatform').addEventListener('change', function() {
        currentAd.platform = this.value;
        updateAdSizeOptions(this.value);
    });
    
    document.getElementById('adSize').addEventListener('change', updateAd);
    
    // Save ad
    document.getElementById('saveAdBtn').addEventListener('click', saveAd);
    
    // Add to campaign
    document.getElementById('addToCampaignBtn').addEventListener('click', function() {
        new bootstrap.Modal(document.getElementById('campaignModal')).show();
    });
    
    document.getElementById('confirmAddCampaign').addEventListener('click', addToCampaign);
    
    // Download
    document.getElementById('downloadBtn').addEventListener('click', downloadAd);
    
    // Initialize form with empty fields (no mock data)
    // User starts with a blank slate and enters their own content
    document.getElementById('headline').value = '';
    document.getElementById('description').value = '';
    document.getElementById('cta').value = '';
    document.getElementById('destinationUrl').value = '';
    
    // Set default styling colors (not content)
    document.getElementById('ctaColor').value = '#007bff';
    document.getElementById('bgColor').value = '#ffffff';
    document.getElementById('textColor').value = '#000000';
    
    // Update the preview with empty state
    updateAd();
    
    // Note: loadTemplate('retail') was removed - user starts with empty fields
    // If they want a template, they can click on one in the sidebar
});

function loadTemplate(templateName) {
    currentAd.template = templateName;
    const template = adTemplates[templateName];
    
    if (template) {
        // Only update template fields if they're empty (don't overwrite user data)
        // Templates are starting points, not replacements for user content
        const headlineEl = document.getElementById('headline');
        const descriptionEl = document.getElementById('description');
        const ctaEl = document.getElementById('cta');
        
        // Only fill if empty - preserve user-entered data
        if (headlineEl && !headlineEl.value.trim()) {
            headlineEl.value = template.headline;
        }
        if (descriptionEl && !descriptionEl.value.trim()) {
            descriptionEl.value = template.description;
        }
        if (ctaEl && !ctaEl.value.trim()) {
            ctaEl.value = template.cta;
        }
        
        // Colors can be updated (these are styling, not content)
        document.getElementById('ctaColor').value = template.ctaColor;
        document.getElementById('bgColor').value = template.bgColor;
        document.getElementById('textColor').value = template.textColor;
        
        const bgImageEl = document.getElementById('bgImage');
        if (bgImageEl && !bgImageEl.value.trim()) {
            bgImageEl.value = template.bgImage || '';
        }
        
        // Clear destination URL when loading template (user should set their own)
        const destinationUrlEl = document.getElementById('destinationUrl');
        if (destinationUrlEl) {
            destinationUrlEl.value = '';
        }
        
        updateAd();
    }
}

function updateAd() {
    // ALWAYS collect form values - never use mock/template data unless form is empty
    // This ensures user-entered data is always used
    const headlineEl = document.getElementById('headline');
    const descriptionEl = document.getElementById('description');
    const ctaEl = document.getElementById('cta');
    const destinationUrlEl = document.getElementById('destinationUrl');
    
    currentAd.headline = headlineEl ? headlineEl.value.trim() : '';
    currentAd.description = descriptionEl ? descriptionEl.value.trim() : '';
    currentAd.cta = ctaEl ? ctaEl.value.trim() : '';
    currentAd.destinationUrl = destinationUrlEl ? destinationUrlEl.value.trim() : '';
    currentAd.ctaColor = document.getElementById('ctaColor')?.value || '#007bff';
    currentAd.bgColor = document.getElementById('bgColor')?.value || '#ffffff';
    currentAd.textColor = document.getElementById('textColor')?.value || '#000000';
    currentAd.bgImage = document.getElementById('bgImage')?.value.trim() || '';
    currentAd.productImage = document.getElementById('productImage')?.value.trim() || '';
    currentAd.size = document.getElementById('adSize')?.value || '1200x628';
    
    const [width, height] = currentAd.size.split('x').map(Number);
    const aspectRatio = width / height;
    
    const canvas = document.getElementById('adCanvas');
    canvas.style.aspectRatio = `${width} / ${height}`;
    
    let html = `
        <div style="
            width: 100%;
            height: 100%;
            background: ${currentAd.bgColor};
            background-image: ${currentAd.bgImage ? `url('${currentAd.bgImage}')` : 'none'};
            background-size: cover;
            background-position: center;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            padding: 40px;
            position: relative;
            overflow: hidden;
            box-sizing: border-box;
        ">
    `;
    
    if (currentAd.productImage) {
        html += `
            <img src="${currentAd.productImage}" alt="Product" style="
                max-width: 40%;
                max-height: 40%;
                object-fit: contain;
                margin-bottom: 20px;
                border-radius: 8px;
            " onerror="this.style.display='none'">
        `;
    }
    
    html += `
        <h2 style="
            color: ${currentAd.textColor};
            font-size: clamp(18px, 4vw, ${Math.max(24, width / 20)}px);
            font-weight: bold;
            text-align: center;
            margin: 0 0 15px 0;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
            z-index: 1;
        ">${escapeHtml(currentAd.headline)}</h2>
    `;
    
    html += `
        <p style="
            color: ${currentAd.textColor};
            font-size: clamp(12px, 2.5vw, ${Math.max(14, width / 40)}px);
            text-align: center;
            margin: 0 0 25px 0;
            opacity: 0.9;
            z-index: 1;
            line-height: 1.4;
        ">${escapeHtml(currentAd.description)}</p>
    `;
    
    html += `
        <button style="
            background: ${currentAd.ctaColor};
            color: white;
            border: none;
            padding: 15px 40px;
            font-size: clamp(14px, 3vw, ${Math.max(16, width / 35)}px);
            font-weight: bold;
            border-radius: 50px;
            cursor: pointer;
            box-shadow: 0 4px 6px rgba(0,0,0,0.3);
            transition: transform 0.2s;
            z-index: 1;
        ">${escapeHtml(currentAd.cta)}</button>
    `;
    
    html += '</div>';
    canvas.innerHTML = html;
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function updateAdSizeOptions(platform) {
    const sizeSelect = document.getElementById('adSize');
    sizeSelect.innerHTML = '';
    
    if (platform === 'facebook_ads') {
        sizeSelect.innerHTML = `
            <option value="1200x628">Facebook Standard (1200x628)</option>
            <option value="1080x1080">Square (1080x1080)</option>
            <option value="1200x1200">Instagram Square (1200x1200)</option>
            <option value="1080x1920">Stories (1080x1920)</option>
        `;
    } else {
        sizeSelect.innerHTML = `
            <option value="300x250">Banner (300x250)</option>
            <option value="728x90">Leaderboard (728x90)</option>
            <option value="160x600">Skyscraper (160x600)</option>
            <option value="300x600">Half Page (300x600)</option>
        `;
    }
    
    currentAd.size = sizeSelect.value;
    updateAd();
}

function uploadFile(file, type) {
    if (!file) return;
    
    const formData = new FormData();
    formData.append('file', file);
    formData.append('upload_type', type === 'bgImage' ? 'direct_mail_design' : 'direct_mail_design');
    formData.append('csrf_token', '<?= $_SESSION['csrf_token'] ?? '' ?>');
    
    fetch('<?= url("api/file_upload.php") ?>', {
        method: 'POST',
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            const imageUrl = '<?= url("") ?>' + data.file_path;
            if (type === 'bgImage') {
                document.getElementById('bgImage').value = imageUrl;
            } else {
                document.getElementById('productImage').value = imageUrl;
            }
            updateAd();
        } else {
            alert('Upload failed: ' + (data.error || 'Unknown error'));
        }
    })
    .catch(err => {
        alert('Upload error: ' + err.message);
    });
}

function saveAd() {
    // CRITICAL: Update ad data from form fields FIRST to ensure we save actual user data, not mock/template data
    updateAd(); // This collects all form values including headline, description, cta, destinationUrl, etc.
    
    // Update metadata
    currentAd.name = document.getElementById('adName')?.value.trim() || 'My Ad';
    currentAd.platform = document.getElementById('adPlatform')?.value || 'google_ads';
    
    // Validate that user has entered actual content (not just template defaults)
    const hasUserContent = currentAd.headline || currentAd.description || currentAd.productImage;
    if (!hasUserContent) {
        alert('Please enter ad content (headline, description, or image) before saving.');
        return;
    }
    
    // Log what we're saving for debugging
    console.log('Saving ad with data:', {
        name: currentAd.name,
        headline: currentAd.headline,
        description: currentAd.description,
        cta: currentAd.cta,
        destinationUrl: currentAd.destinationUrl,
        hasProductImage: !!currentAd.productImage
    });
    
    const formData = new FormData();
    formData.append('action', 'save_ad');
    formData.append('ad_data', JSON.stringify(currentAd));
    formData.append('csrf_token', document.querySelector('input[name="csrf_token"]')?.value || '<?= generateCSRFToken() ?>');
    
    fetch('<?= url("api/ad_builder.php") ?>', {
        method: 'POST',
        body: formData
    })
    .then(r => {
        if (!r.ok) {
            throw new Error(`HTTP error! status: ${r.status}`);
        }
        return r.json();
    })
    .then(data => {
        if (data.success) {
            alert('Ad saved successfully! Ad ID: ' + (data.ad_id || 'N/A'));
        } else {
            alert('Error: ' + (data.error || 'Unknown error'));
        }
    })
    .catch(err => {
        console.error('Save error:', err);
        alert('Error saving ad: ' + err.message);
    });
}

function addToCampaign() {
    const campaignId = document.getElementById('selectedCampaign').value;
    
    if (!campaignId) {
        alert('Please select a campaign');
        return;
    }
    
    // CRITICAL: Update ad data from form fields FIRST to ensure we save actual user data
    updateAd(); // This collects all form values
    
    // Update metadata
    currentAd.name = document.getElementById('adName')?.value.trim() || 'My Ad';
    currentAd.platform = document.getElementById('adPlatform')?.value || 'google_ads';
    
    // Validate that user has entered actual content
    const hasUserContent = currentAd.headline || currentAd.description || currentAd.productImage;
    if (!hasUserContent) {
        alert('Please enter ad content (headline, description, or image) before adding to campaign.');
        return;
    }
    
    // Log what we're saving for debugging
    console.log('Adding ad to campaign with data:', {
        name: currentAd.name,
        headline: currentAd.headline,
        description: currentAd.description,
        cta: currentAd.cta,
        destinationUrl: currentAd.destinationUrl
    });
    
    const formData = new FormData();
    formData.append('action', 'add_to_campaign');
    formData.append('campaign_id', campaignId);
    formData.append('ad_data', JSON.stringify(currentAd));
    formData.append('csrf_token', document.querySelector('input[name="csrf_token"]')?.value || '<?= generateCSRFToken() ?>');
    
    // Show loading state
    const confirmBtn = document.getElementById('confirmAddCampaign');
    const originalText = confirmBtn.innerHTML;
    confirmBtn.disabled = true;
    confirmBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Adding...';
    
    fetch('<?= url("api/ad_builder.php") ?>', {
        method: 'POST',
        body: formData
    })
    .then(async r => {
        if (!r.ok) {
            const text = await r.text();
            throw new Error(`HTTP error! status: ${r.status}, response: ${text}`);
        }
        
        const text = await r.text();
        
        // Check if response is empty
        if (!text || text.trim() === '') {
            throw new Error('Empty response from server');
        }
        
        // Try to parse JSON
        try {
            return JSON.parse(text);
        } catch (parseError) {
            console.error('JSON parse error:', parseError);
            console.error('Response text:', text);
            throw new Error('Invalid JSON response: ' + text.substring(0, 200));
        }
    })
    .then(data => {
        if (data.success) {
            const campaignName = data.campaign_name || 'campaign';
            alert('Ad "' + currentAd.name + '" added to campaign "' + campaignName + '" successfully!');
            bootstrap.Modal.getInstance(document.getElementById('campaignModal')).hide();
            // Reset campaign selection
            document.getElementById('selectedCampaign').value = '';
        } else {
            alert('Error: ' + (data.error || 'Unknown error'));
        }
    })
    .catch(err => {
        console.error('Add to campaign error:', err);
        alert('Error adding ad to campaign: ' + err.message);
    })
    .finally(() => {
        confirmBtn.disabled = false;
        confirmBtn.innerHTML = originalText;
    });
}

function downloadAd() {
    // Generate email-compatible HTML (same format as used in email campaigns)
    const adData = currentAd;
    const adName = document.getElementById('adName').value || currentAd.name || 'My Ad';
    
    // Generate email-compatible HTML for the ad (same format as insertAdHTML)
    let adHTML = '<!-- Ad: ' + escapeHtml(adName) + ' -->\n';
    adHTML += '<table cellpadding="0" cellspacing="0" border="0" width="100%" style="max-width: 600px; margin: 20px auto;">\n';
    adHTML += '<tr><td style="padding: 20px; background: ' + (adData.bgColor || '#ffffff') + ';';
    if (adData.bgImage) {
        adHTML += ' background-image: url(\'' + escapeHtml(adData.bgImage) + '\'); background-size: cover;';
    }
    adHTML += ' text-align: center;">\n';
    
    if (adData.productImage) {
        adHTML += '<img src="' + escapeHtml(adData.productImage) + '" alt="' + escapeHtml(adData.headline || 'Ad') + '" style="max-width: 300px; height: auto; display: block; margin: 0 auto 15px;">\n';
    }
    
    if (adData.headline) {
        adHTML += '<h2 style="color: ' + (adData.textColor || '#000000') + '; margin: 10px 0;">' + escapeHtml(adData.headline) + '</h2>\n';
    }
    
    if (adData.description) {
        adHTML += '<p style="color: ' + (adData.textColor || '#000000') + '; margin: 10px 0;">' + escapeHtml(adData.description) + '</p>\n';
    }
    
    if (adData.cta) {
        adHTML += '<a href="#" style="display: inline-block; background: ' + (adData.ctaColor || '#007bff') + '; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin-top: 15px;">' + escapeHtml(adData.cta) + '</a>\n';
    }
    
    adHTML += '</td></tr>\n';
    adHTML += '</table>\n';
    adHTML += '<!-- End Ad -->\n';
    
    // Create download link for HTML
    const blob = new Blob([adHTML], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.download = adName.replace(/[^a-z0-9]/gi, '_') + '_email_ad.html';
    link.href = url;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
}

function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

