<?php
/**
 * AcceleratedLeadz - Email Automation & CRM Integration
 * Processes IMAP emails from roundbox and sends automated emails to CRM customers
 * (1 email per minute like Mailchimp)
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/csrf.php';

requireClient();
$pageTitle = 'AcceleratedLeadz - Email Automation';

$userId = $_SESSION['user_id'];
$db = getDB();

// Process IMAP emails if requested
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'process_imap') {
    require_once __DIR__ . '/../includes/csrf.php';
    
    // Call the API endpoint
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, url('api/acceleratedleadz.php?action=process_imap&user_id=' . $userId));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $result = curl_exec($ch);
    curl_close($ch);
    
    $resultData = json_decode($result, true);
    if ($resultData['success'] ?? false) {
        setFlashMessage('success', "Processed {$resultData['processed']} emails from IMAP");
    } else {
        setFlashMessage('danger', 'Error processing IMAP: ' . ($resultData['error'] ?? 'Unknown error'));
    }
    
    header('Location: ' . url('client/acceleratedleadz.php'));
    exit;
}

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <h1><i class="bi bi-envelope-at"></i> AcceleratedLeadz - Email Automation</h1>
            <div>
                <form method="POST" class="d-inline">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="process_imap">
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-download"></i> Process IMAP Emails
                    </button>
                </form>
                <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#createCampaignModal">
                    <i class="bi bi-plus-circle"></i> Create Campaign
                </button>
            </div>
        </div>
        <p class="text-muted">Automated email campaigns powered by IMAP integration with roundbox</p>
    </div>
</div>

<!-- Statistics Cards -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card text-white bg-primary">
            <div class="card-body">
                <h5 class="card-title">Active Campaigns</h5>
                <h2 class="mb-0" id="activeCampaigns">-</h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-success">
            <div class="card-body">
                <h5 class="card-title">Pending Emails</h5>
                <h2 class="mb-0" id="pendingEmails">-</h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-info">
            <div class="card-body">
                <h5 class="card-title">Sent Today</h5>
                <h2 class="mb-0" id="sentToday">-</h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-warning">
            <div class="card-body">
                <h5 class="card-title">Failed</h5>
                <h2 class="mb-0" id="failedEmails">-</h2>
            </div>
        </div>
    </div>
</div>

<!-- Campaigns List -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <h5 class="card-title">Email Automation Campaigns</h5>
                <div id="campaignsList">
                    <div class="text-center py-5">
                        <div class="spinner-border" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Create Campaign Modal -->
<div class="modal fade" id="createCampaignModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bi bi-plus-circle"></i> Create Email Automation Campaign</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="createCampaignForm">
                <?= csrfField() ?>
                <div class="modal-body">
                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i> This campaign will automatically send emails to your CRM customers 
                        at a rate of 1 email per minute (like Mailchimp automation).
                    </div>
                    
                    <div class="mb-3">
                        <label for="campaign_name" class="form-label">Campaign Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="campaign_name" name="campaign_name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="campaign_type" class="form-label">Campaign Type</label>
                        <select class="form-select" id="campaign_type" name="campaign_type">
                            <option value="drip">Drip Campaign (Sequential Emails)</option>
                            <option value="imap_trigger">IMAP Triggered (Based on incoming emails)</option>
                            <option value="scheduled">Scheduled Campaign</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="send_rate" class="form-label">Send Rate (emails per minute)</label>
                        <input type="number" class="form-control" id="send_rate" name="send_rate" value="1" min="1" max="60">
                        <small class="text-muted">Recommended: 1 email per minute for best deliverability</small>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="start_date" class="form-label">Start Date</label>
                            <input type="datetime-local" class="form-control" id="start_date" name="start_date">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="end_date" class="form-label">End Date</label>
                            <input type="datetime-local" class="form-control" id="end_date" name="end_date">
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Campaign</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
const csrfToken = document.querySelector('input[name="csrf_token"]')?.value || '';

// Load campaigns and stats
function loadData() {
    // Load campaigns
    fetch('<?= url('api/acceleratedleadz.php') ?>?action=get_campaigns', {
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            displayCampaigns(data.campaigns);
        }
    })
    .catch(err => console.error('Error loading campaigns:', err));
    
    // Load queue stats
    fetch('<?= url('api/acceleratedleadz.php') ?>?action=get_queue_stats', {
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            updateStats(data.stats);
        }
    })
    .catch(err => console.error('Error loading stats:', err));
}

function displayCampaigns(campaigns) {
    if (!campaigns || campaigns.length === 0) {
        document.getElementById('campaignsList').innerHTML = 
            '<p class="text-muted text-center py-5">No campaigns yet. Create your first email automation campaign!</p>';
        return;
    }
    
    let html = '<div class="list-group">';
    
    campaigns.forEach(camp => {
        const statusBadge = {
            'active': 'success',
            'paused': 'warning',
            'completed': 'secondary',
            'draft': 'info'
        }[camp.status] || 'secondary';
        
        html += `
            <div class="list-group-item">
                <div class="d-flex w-100 justify-content-between">
                    <h5 class="mb-1">${escapeHtml(camp.campaign_name)}</h5>
                    <span class="badge bg-${statusBadge}">${escapeHtml(camp.status)}</span>
                </div>
                <p class="mb-1">
                    <small class="text-muted">
                        Type: ${escapeHtml(camp.campaign_type)} | 
                        Rate: ${escapeHtml(camp.send_rate)} email/min | 
                        IMAP Source: ${escapeHtml(camp.imap_source)}
                    </small>
                </p>
                <small class="text-muted">
                    Created: ${new Date(camp.created_at).toLocaleString()}
                </small>
            </div>
        `;
    });
    
    html += '</div>';
    document.getElementById('campaignsList').innerHTML = html;
    
    // Update active campaigns count
    const active = campaigns.filter(c => c.status === 'active').length;
    document.getElementById('activeCampaigns').textContent = active;
}

function updateStats(stats) {
    document.getElementById('pendingEmails').textContent = parseInt(stats.pending || 0);
    document.getElementById('sentToday').textContent = parseInt(stats.sent || 0);
    document.getElementById('failedEmails').textContent = parseInt(stats.failed || 0);
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Create campaign form
document.getElementById('createCampaignForm')?.addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', 'create_campaign');
    
    fetch('<?= url('api/acceleratedleadz.php') ?>', {
        method: 'POST',
        body: formData
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            bootstrap.Modal.getInstance(document.getElementById('createCampaignModal')).hide();
            loadData();
            alert('Campaign created successfully!');
        } else {
            alert('Error: ' + (data.error || 'Unknown error'));
        }
    })
    .catch(err => {
        console.error('Error:', err);
        alert('Error creating campaign');
    });
});

// Load on page load
loadData();
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

