<?php
/**
 * AcceleratedAdz - Native Ad Platform
 * Similar to Facebook Ads and Google Ads, hosted on ai-itinc.com domain
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/csrf.php';

requireClient();
$pageTitle = 'AcceleratedAdz - Native Ads';

$userId = $_SESSION['user_id'];
$db = getDB();

require_once __DIR__ . '/../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <h1><i class="bi bi-bullseye"></i> AcceleratedAdz - Native Advertising</h1>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createCampaignModal">
                <i class="bi bi-plus-circle"></i> Create Campaign
            </button>
        </div>
        <p class="text-muted">Manage your native ad campaigns powered by AI-IT Inc</p>
    </div>
</div>

<!-- Statistics Cards -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card text-white bg-primary">
            <div class="card-body">
                <h5 class="card-title">Total Campaigns</h5>
                <h2 class="mb-0" id="totalCampaigns">-</h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-success">
            <div class="card-body">
                <h5 class="card-title">Impressions</h5>
                <h2 class="mb-0" id="totalImpressions">-</h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-info">
            <div class="card-body">
                <h5 class="card-title">Clicks</h5>
                <h2 class="mb-0" id="totalClicks">-</h2>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-white bg-warning">
            <div class="card-body">
                <h5 class="card-title">Total Spend</h5>
                <h2 class="mb-0" id="totalSpend">$0.00</h2>
            </div>
        </div>
    </div>
</div>

<!-- Campaigns Table -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <div id="campaignsTable">
                    <div class="text-center py-5">
                        <div class="spinner-border" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Create Campaign Modal -->
<div class="modal fade" id="createCampaignModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bi bi-plus-circle"></i> Create AcceleratedAdz Campaign</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="createCampaignForm">
                <?= csrfField() ?>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="campaign_name" class="form-label">Campaign Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="campaign_name" name="campaign_name" required>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="campaign_type" class="form-label">Campaign Type <span class="text-danger">*</span></label>
                            <select class="form-select" id="campaign_type" name="campaign_type" required>
                                <option value="display">Display Ads</option>
                                <option value="search">Search Ads</option>
                                <option value="video">Video Ads</option>
                                <option value="social">Social Ads</option>
                                <option value="native">Native Ads</option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="objective" class="form-label">Objective <span class="text-danger">*</span></label>
                            <select class="form-select" id="objective" name="objective" required>
                                <option value="traffic">Traffic</option>
                                <option value="conversions">Conversions</option>
                                <option value="awareness">Awareness</option>
                                <option value="engagement">Engagement</option>
                                <option value="leads">Leads</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="budget_type" class="form-label">Budget Type</label>
                            <select class="form-select" id="budget_type" name="budget_type">
                                <option value="daily">Daily Budget</option>
                                <option value="lifetime">Lifetime Budget</option>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="daily_budget" class="form-label">Daily Budget ($)</label>
                            <input type="number" step="0.01" class="form-control" id="daily_budget" name="daily_budget">
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="start_date" class="form-label">Start Date</label>
                            <input type="datetime-local" class="form-control" id="start_date" name="start_date">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label for="end_date" class="form-label">End Date</label>
                            <input type="datetime-local" class="form-control" id="end_date" name="end_date">
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="optimization_goal" class="form-label">Optimization Goal</label>
                        <select class="form-select" id="optimization_goal" name="optimization_goal">
                            <option value="clicks">Clicks</option>
                            <option value="impressions">Impressions</option>
                            <option value="conversions">Conversions</option>
                            <option value="cpa">CPA (Cost Per Acquisition)</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="bid_strategy" class="form-label">Bid Strategy</label>
                        <select class="form-select" id="bid_strategy" name="bid_strategy">
                            <option value="cpc">CPC (Cost Per Click)</option>
                            <option value="cpm">CPM (Cost Per 1000 Impressions)</option>
                            <option value="cpa">CPA (Cost Per Acquisition)</option>
                            <option value="cpl">CPL (Cost Per Lead)</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="default_bid" class="form-label">Default Bid ($)</label>
                        <input type="number" step="0.01" class="form-control" id="default_bid" name="default_bid" value="1.00">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Campaign</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
const csrfToken = document.querySelector('input[name="csrf_token"]')?.value || '';

// Load campaigns
function loadCampaigns() {
    fetch('<?= url('api/acceleratedadz.php') ?>?action=get_campaigns', {
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            displayCampaigns(data.campaigns);
            updateStats(data.campaigns);
        } else {
            document.getElementById('campaignsTable').innerHTML = 
                '<p class="text-danger">Error loading campaigns: ' + (data.error || 'Unknown error') + '</p>';
        }
    })
    .catch(err => {
        console.error('Error:', err);
        document.getElementById('campaignsTable').innerHTML = 
            '<p class="text-danger">Error loading campaigns</p>';
    });
}

function displayCampaigns(campaigns) {
    if (!campaigns || campaigns.length === 0) {
        document.getElementById('campaignsTable').innerHTML = 
            '<p class="text-muted text-center py-5">No campaigns yet. Create your first AcceleratedAdz campaign!</p>';
        return;
    }
    
    let html = `
        <div class="table-responsive">
            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Campaign Name</th>
                        <th>Type</th>
                        <th>Status</th>
                        <th>Budget</th>
                        <th>Impressions</th>
                        <th>Clicks</th>
                        <th>CTR</th>
                        <th>Spend</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
    `;
    
    campaigns.forEach(camp => {
        const ctr = camp.total_impressions > 0 ? ((camp.total_clicks / camp.total_impressions) * 100).toFixed(2) : '0.00';
        const statusBadge = {
            'active': 'success',
            'paused': 'warning',
            'archived': 'secondary',
            'draft': 'info'
        }[camp.status] || 'secondary';
        
        html += `
            <tr>
                <td><strong>${escapeHtml(camp.campaign_name)}</strong></td>
                <td>${escapeHtml(camp.campaign_type)}</td>
                <td><span class="badge bg-${statusBadge}">${escapeHtml(camp.status)}</span></td>
                <td>$${parseFloat(camp.daily_budget || 0).toFixed(2)}/day</td>
                <td>${parseInt(camp.total_impressions || 0).toLocaleString()}</td>
                <td>${parseInt(camp.total_clicks || 0).toLocaleString()}</td>
                <td>${ctr}%</td>
                <td>$${parseFloat(camp.total_spend || 0).toFixed(2)}</td>
                <td>
                    <button class="btn btn-sm btn-info" onclick="viewCampaign(${camp.id})">
                        <i class="bi bi-eye"></i> View
                    </button>
                </td>
            </tr>
        `;
    });
    
    html += `
                </tbody>
            </table>
        </div>
    `;
    
    document.getElementById('campaignsTable').innerHTML = html;
}

function updateStats(campaigns) {
    let total = campaigns.length;
    let impressions = 0;
    let clicks = 0;
    let spend = 0;
    
    campaigns.forEach(camp => {
        impressions += parseInt(camp.total_impressions || 0);
        clicks += parseInt(camp.total_clicks || 0);
        spend += parseFloat(camp.total_spend || 0);
    });
    
    document.getElementById('totalCampaigns').textContent = total;
    document.getElementById('totalImpressions').textContent = impressions.toLocaleString();
    document.getElementById('totalClicks').textContent = clicks.toLocaleString();
    document.getElementById('totalSpend').textContent = '$' + spend.toFixed(2);
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function viewCampaign(campaignId) {
    window.location.href = '<?= url('client/campaigns.php') ?>?view=' + campaignId + '&type=acceleratedadz';
}

// Create campaign form
document.getElementById('createCampaignForm')?.addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('action', 'create_campaign');
    
    fetch('<?= url('api/acceleratedadz.php') ?>', {
        method: 'POST',
        body: formData
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            bootstrap.Modal.getInstance(document.getElementById('createCampaignModal')).hide();
            loadCampaigns();
            alert('Campaign created successfully!');
        } else {
            alert('Error: ' + (data.error || 'Unknown error'));
        }
    })
    .catch(err => {
        console.error('Error:', err);
        alert('Error creating campaign');
    });
});

// Load on page load
loadCampaigns();
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

