<?php
/**
 * A/B Testing Dashboard
 * Manage campaign variants and A/B tests
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';

requireClient();
$pageTitle = 'A/B Testing';

$userId = $_SESSION['user_id'];
$db = getDB();

// Get campaigns with variants
$stmt = $db->prepare("
    SELECT c.*, COUNT(cv.id) as variant_count,
           SUM(CASE WHEN cv.status = 'active' THEN 1 ELSE 0 END) as active_variants
    FROM campaigns c
    LEFT JOIN campaign_variants cv ON c.id = cv.campaign_id
    WHERE c.user_id = ?
    GROUP BY c.id
    HAVING variant_count > 0 OR c.status IN ('active', 'draft')
    ORDER BY c.created_at DESC
");
$stmt->execute([$userId]);
$campaigns = $stmt->fetchAll();

// Get all available campaigns for variant creation
$stmt = $db->prepare("
    SELECT id, campaign_name, campaign_type, status
    FROM campaigns 
    WHERE user_id = ? AND status IN ('active', 'draft')
    ORDER BY campaign_name ASC
");
$stmt->execute([$userId]);
$availableCampaigns = $stmt->fetchAll();

require_once __DIR__ . '/../includes/header.php';
require_once __DIR__ . '/../includes/csrf.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center">
            <h1><i class="bi bi-clipboard-data"></i> A/B Testing</h1>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createVariantModal">
                <i class="bi bi-plus-circle"></i> Create Variant
            </button>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <?php if (empty($campaigns)): ?>
                    <p class="text-muted text-center py-5">No A/B tests yet. Create variants for your campaigns!</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Campaign</th>
                                    <th>Type</th>
                                    <th>Variants</th>
                                    <th>Active</th>
                                    <th>Winner</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($campaigns as $campaign): 
                                    // Get winner info
                                    $stmt = $db->prepare("SELECT variant_name FROM campaign_variants WHERE campaign_id = ? AND is_winner = 1 LIMIT 1");
                                    $stmt->execute([$campaign['id']]);
                                    $winner = $stmt->fetch();
                                ?>
                                    <tr>
                                        <td><strong><?= escape($campaign['campaign_name']) ?></strong></td>
                                        <td><?= ucfirst(str_replace('_', ' ', $campaign['campaign_type'])) ?></td>
                                        <td>
                                            <span class="badge bg-info"><?= $campaign['variant_count'] ?> total</span>
                                        </td>
                                        <td>
                                            <span class="badge bg-success"><?= $campaign['active_variants'] ?? 0 ?> active</span>
                                        </td>
                                        <td>
                                            <?php if ($winner): ?>
                                                <span class="badge bg-success"><i class="bi bi-trophy"></i> <?= escape($winner['variant_name']) ?></span>
                                            <?php else: ?>
                                                <span class="text-muted">No winner yet</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?= $campaign['status'] === 'active' ? 'success' : ($campaign['status'] === 'draft' ? 'warning' : 'secondary') ?>">
                                                <?= escape($campaign['status']) ?>
                                            </span>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <button class="btn btn-sm btn-primary view-variants-btn" data-campaign-id="<?= $campaign['id'] ?>">
                                                    <i class="bi bi-eye"></i> View
                                                </button>
                                                <?php if ($campaign['variant_count'] == 0): ?>
                                                    <button class="btn btn-sm btn-success add-variant-btn" data-campaign-id="<?= $campaign['id'] ?>" data-campaign-name="<?= escape($campaign['campaign_name']) ?>">
                                                        <i class="bi bi-plus"></i> Add Variant
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Create Variant Modal -->
<div class="modal fade" id="createVariantModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Create Campaign Variant</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="createVariantForm">
                <div class="modal-body">
                    <?= csrfField() ?>
                    <input type="hidden" name="action" value="create_variant">
                    <div class="mb-3">
                        <label for="variant_campaign_id" class="form-label">Campaign</label>
                        <select class="form-select" id="variant_campaign_id" name="campaign_id" required>
                            <option value="">Select a campaign...</option>
                            <?php foreach ($availableCampaigns as $camp): ?>
                                <option value="<?= $camp['id'] ?>"><?= escape($camp['campaign_name']) ?> (<?= ucfirst($camp['campaign_type']) ?>)</option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label for="variant_name" class="form-label">Variant Name</label>
                        <input type="text" class="form-control" id="variant_name" name="variant_name" placeholder="e.g., Version A, Headline Test" required>
                    </div>
                    <div class="mb-3">
                        <label for="traffic_percentage" class="form-label">Traffic Percentage (%)</label>
                        <input type="number" class="form-control" id="traffic_percentage" name="traffic_percentage" min="0" max="100" step="0.1" value="50" required>
                        <div class="form-text">Percentage of traffic to send to this variant (total across all variants cannot exceed 100%)</div>
                    </div>
                    <div class="mb-3">
                        <label for="variant_data" class="form-label">Variant Data (JSON)</label>
                        <textarea class="form-control font-monospace" id="variant_data" name="variant_data" rows="8" placeholder='{"headline": "New Headline", "image": "url", "cta": "Button Text"}'>{}</textarea>
                        <div class="form-text">Enter variant-specific settings as JSON (e.g., different headlines, images, CTA text)</div>
                    </div>
                    <div id="variantFormResult" class="alert" style="display:none;"></div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Variant</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Variants Modal -->
<div class="modal fade" id="variantsModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Campaign Variants - <span id="variantsCampaignName"></span></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="variantsContent">
                <p class="text-center">Loading variants...</p>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const csrfToken = document.querySelector('input[name="csrf_token"]')?.value || '<?= generateCSRFToken() ?>';
    
    // View variants
    document.querySelectorAll('.view-variants-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const campaignId = this.dataset.campaignId;
            loadVariants(campaignId);
        });
    });
    
    // Add variant button (quick add)
    document.querySelectorAll('.add-variant-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const campaignId = this.dataset.campaignId;
            const campaignName = this.dataset.campaignName;
            document.getElementById('variant_campaign_id').value = campaignId;
            const modal = new bootstrap.Modal(document.getElementById('createVariantModal'));
            modal.show();
        });
    });
    
    // Create variant form
    const createVariantForm = document.getElementById('createVariantForm');
    if (createVariantForm) {
        createVariantForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const resultDiv = document.getElementById('variantFormResult');
            resultDiv.style.display = 'block';
            resultDiv.className = 'alert alert-info';
            resultDiv.textContent = 'Creating variant...';
            
            const formData = new FormData(createVariantForm);
            
            fetch('<?= url('api/ab_testing.php') ?>', {
                method: 'POST',
                body: formData
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    resultDiv.className = 'alert alert-success';
                    resultDiv.textContent = data.message || 'Variant created successfully!';
                    setTimeout(() => {
                        bootstrap.Modal.getInstance(document.getElementById('createVariantModal')).hide();
                        location.reload();
                    }, 1500);
                } else {
                    resultDiv.className = 'alert alert-danger';
                    resultDiv.textContent = data.error || 'Failed to create variant';
                }
            })
            .catch(err => {
                resultDiv.className = 'alert alert-danger';
                resultDiv.textContent = 'Network error: ' + err.message;
            });
        });
    }
    
    function loadVariants(campaignId) {
        const modal = document.getElementById('variantsModal');
        const content = document.getElementById('variantsContent');
        content.innerHTML = '<p class="text-center"><i class="spinner-border"></i> Loading variants...</p>';
        
        // Get campaign name
        const campaignRow = document.querySelector(`[data-campaign-id="${campaignId}"]`)?.closest('tr');
        const campaignName = campaignRow?.querySelector('td:first-child strong')?.textContent || 'Campaign';
        document.getElementById('variantsCampaignName').textContent = campaignName;
        
        fetch(`<?= url('api/ab_testing.php') ?>?action=variants&campaign_id=${campaignId}`)
            .then(r => r.json())
            .then(data => {
                if (data.success && data.data.length > 0) {
                    let html = '<div class="row mb-3"><div class="col-12">';
                    html += '<div class="table-responsive"><table class="table table-hover">';
                    html += '<thead><tr><th>Variant</th><th>Status</th><th>Traffic %</th><th>Impressions</th><th>Clicks</th><th>CTR</th><th>Conversions</th><th>Conv. Rate</th><th>Cost</th><th>Revenue</th><th>ROI</th><th>Actions</th></tr></thead><tbody>';
                    
                    data.data.forEach(variant => {
                        const ctr = variant.impressions > 0 ? ((variant.clicks / variant.impressions) * 100).toFixed(2) : '0.00';
                        const convRate = variant.clicks > 0 ? ((variant.conversions / variant.clicks) * 100).toFixed(2) : '0.00';
                        const roi = variant.cost > 0 ? (((variant.revenue - variant.cost) / variant.cost) * 100).toFixed(2) : '0.00';
                        const statusBadge = variant.status === 'active' ? 'success' : (variant.status === 'paused' ? 'warning' : 'secondary');
                        
                        html += `<tr ${variant.is_winner ? 'class="table-success"' : ''}>`;
                        html += `<td><strong>${escapeHtml(variant.variant_name)}</strong> ${variant.is_winner ? '<span class="badge bg-success ms-2"><i class="bi bi-trophy"></i> Winner</span>' : ''}</td>`;
                        html += `<td><span class="badge bg-${statusBadge}">${variant.status}</span></td>`;
                        html += `<td><strong>${variant.traffic_percentage}%</strong></td>`;
                        html += `<td>${variant.impressions.toLocaleString()}</td>`;
                        html += `<td>${variant.clicks.toLocaleString()}</td>`;
                        html += `<td><strong>${ctr}%</strong></td>`;
                        html += `<td>${variant.conversions.toLocaleString()}</td>`;
                        html += `<td><strong>${convRate}%</strong></td>`;
                        html += `<td>$${parseFloat(variant.cost).toFixed(2)}</td>`;
                        html += `<td>$${parseFloat(variant.revenue).toFixed(2)}</td>`;
                        html += `<td><strong class="${parseFloat(roi) >= 0 ? 'text-success' : 'text-danger'}">${roi}%</strong></td>`;
                        html += '<td><div class="btn-group btn-group-sm">';
                        
                        if (!variant.is_winner) {
                            html += `<button class="btn btn-sm btn-success set-winner-btn" data-variant-id="${variant.id}" title="Set as Winner"><i class="bi bi-trophy"></i></button>`;
                        }
                        
                        if (variant.status === 'active') {
                            html += `<button class="btn btn-sm btn-warning pause-variant-btn" data-variant-id="${variant.id}" title="Pause"><i class="bi bi-pause"></i></button>`;
                        } else if (variant.status === 'paused') {
                            html += `<button class="btn btn-sm btn-success resume-variant-btn" data-variant-id="${variant.id}" title="Resume"><i class="bi bi-play"></i></button>`;
                        }
                        
                        html += '</div></td>';
                        html += '</tr>';
                    });
                    
                    html += '</tbody></table></div>';
                    
                    // Add summary stats
                    const totalImpressions = data.data.reduce((sum, v) => sum + v.impressions, 0);
                    const totalClicks = data.data.reduce((sum, v) => sum + v.clicks, 0);
                    const totalConversions = data.data.reduce((sum, v) => sum + v.conversions, 0);
                    const totalCost = data.data.reduce((sum, v) => sum + parseFloat(v.cost), 0);
                    const totalRevenue = data.data.reduce((sum, v) => sum + parseFloat(v.revenue), 0);
                    const avgCTR = totalImpressions > 0 ? ((totalClicks / totalImpressions) * 100).toFixed(2) : '0.00';
                    const avgConvRate = totalClicks > 0 ? ((totalConversions / totalClicks) * 100).toFixed(2) : '0.00';
                    const totalROI = totalCost > 0 ? (((totalRevenue - totalCost) / totalCost) * 100).toFixed(2) : '0.00';
                    
                    html += '<div class="row mt-4"><div class="col-12"><div class="card bg-light"><div class="card-body">';
                    html += '<h6 class="card-title">Overall Performance Summary</h6>';
                    html += `<div class="row text-center"><div class="col-md-2"><strong>Impressions</strong><br>${totalImpressions.toLocaleString()}</div>`;
                    html += `<div class="col-md-2"><strong>Clicks</strong><br>${totalClicks.toLocaleString()}</div>`;
                    html += `<div class="col-md-2"><strong>CTR</strong><br>${avgCTR}%</div>`;
                    html += `<div class="col-md-2"><strong>Conversions</strong><br>${totalConversions.toLocaleString()}</div>`;
                    html += `<div class="col-md-2"><strong>Conv. Rate</strong><br>${avgConvRate}%</div>`;
                    html += `<div class="col-md-2"><strong>ROI</strong><br><span class="${parseFloat(totalROI) >= 0 ? 'text-success' : 'text-danger'}">${totalROI}%</span></div></div>`;
                    html += '</div></div></div>';
                    
                    html += '</div></div>';
                    
                    content.innerHTML = html;
                    
                    // Add event listeners
                    document.querySelectorAll('.set-winner-btn').forEach(btn => {
                        btn.addEventListener('click', function() {
                            const variantId = this.dataset.variantId;
                            setWinner(variantId);
                        });
                    });
                    
                    document.querySelectorAll('.pause-variant-btn').forEach(btn => {
                        btn.addEventListener('click', function() {
                            const variantId = this.dataset.variantId;
                            updateVariantStatus(variantId, 'paused');
                        });
                    });
                    
                    document.querySelectorAll('.resume-variant-btn').forEach(btn => {
                        btn.addEventListener('click', function() {
                            const variantId = this.dataset.variantId;
                            updateVariantStatus(variantId, 'active');
                        });
                    });
                    
                    new bootstrap.Modal(modal).show();
                } else if (data.success && data.data.length === 0) {
                    content.innerHTML = '<p class="text-center text-muted py-5">No variants found for this campaign. Create your first variant!</p>';
                    new bootstrap.Modal(modal).show();
                } else {
                    content.innerHTML = `<p class="text-center text-danger">Error: ${data.error || 'Failed to load variants'}</p>`;
                    new bootstrap.Modal(modal).show();
                }
            })
            .catch(err => {
                content.innerHTML = `<p class="text-center text-danger">Network error: ${err.message}</p>`;
            });
    }
    
    function setWinner(variantId) {
        if (!confirm('Set this variant as the winner? Other variants will be paused automatically.')) return;
        
        const formData = new FormData();
        formData.append('action', 'set_winner');
        formData.append('variant_id', variantId);
        formData.append('csrf_token', csrfToken);
        
        fetch('<?= url('api/ab_testing.php') ?>', {
            method: 'POST',
            body: formData
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                alert('Winner set successfully!');
                location.reload();
            } else {
                alert('Error: ' + (data.error || 'Failed to set winner'));
            }
        })
        .catch(err => {
            alert('Network error: ' + err.message);
        });
    }
    
    function updateVariantStatus(variantId, status) {
        const action = status === 'paused' ? 'pause' : 'resume';
        if (!confirm(`${status === 'paused' ? 'Pause' : 'Resume'} this variant?`)) return;
        
        const formData = new FormData();
        formData.append('action', 'update_status');
        formData.append('variant_id', variantId);
        formData.append('status', status);
        formData.append('csrf_token', csrfToken);
        
        fetch('<?= url('api/ab_testing.php') ?>', {
            method: 'POST',
            body: formData
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert('Error: ' + (data.error || `Failed to ${action} variant`));
            }
        })
        .catch(err => {
            alert('Network error: ' + err.message);
        });
    }
    
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
});
</script>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>

