<?php
/**
 * Webhook Management API
 * Handles webhook creation, management, and triggering
 */

require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/database.php';

header('Content-Type: application/json');
requireAuth();
$db = getDB();
$userId = $_SESSION['user_id'];

$action = $_GET['action'] ?? $_POST['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

function triggerWebhook($db, $userId, $eventType, $payload) {
    // Get active webhooks for user that listen to this event
    $stmt = $db->prepare("
        SELECT * FROM webhooks 
        WHERE user_id = ? 
        AND status = 'active'
        AND JSON_CONTAINS(events, ?, '$')
    ");
    $stmt->execute([$userId, json_encode($eventType)]);
    $webhooks = $stmt->fetchAll();
    
    foreach ($webhooks as $webhook) {
        $events = json_decode($webhook['events'], true);
        if (in_array($eventType, $events)) {
            sendWebhookRequest($db, $webhook, $eventType, $payload);
        }
    }
}

function sendWebhookRequest($db, $webhook, $eventType, $payload) {
    $ch = curl_init($webhook['webhook_url']);
    
    $postData = [
        'event' => $eventType,
        'timestamp' => date('c'),
        'payload' => $payload
    ];
    
    // Add signature if secret token exists
    if (!empty($webhook['secret_token'])) {
        $signature = hash_hmac('sha256', json_encode($postData), $webhook['secret_token']);
        $postData['signature'] = $signature;
    }
    
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($postData),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'User-Agent: MarketingPlatform/1.0'
        ],
        CURLOPT_TIMEOUT => 10,
        CURLOPT_CONNECTTIMEOUT => 5
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    // Log webhook call
    $stmt = $db->prepare("
        INSERT INTO webhook_logs 
        (webhook_id, event_type, payload, response_status, response_body, error_message)
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    
    $errorMessage = $error ?: null;
    $responseBody = $response ? substr($response, 0, 1000) : null;
    
    $stmt->execute([
        $webhook['id'],
        $eventType,
        json_encode($payload),
        $httpCode,
        $responseBody,
        $errorMessage
    ]);
    
    // Update webhook status if it failed
    if ($httpCode < 200 || $httpCode >= 300) {
        $failureCount = $webhook['failure_count'] + 1;
        $status = ($failureCount >= 5) ? 'failed' : 'active';
        
        $stmt = $db->prepare("
            UPDATE webhooks 
            SET failure_count = ?, status = ?, last_triggered_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$failureCount, $status, $webhook['id']]);
    } else {
        $stmt = $db->prepare("UPDATE webhooks SET last_triggered_at = NOW(), failure_count = 0 WHERE id = ?");
        $stmt->execute([$webhook['id']]);
    }
    
    return ['success' => ($httpCode >= 200 && $httpCode < 300), 'http_code' => $httpCode];
}

try {
    switch ($method) {
        case 'GET':
            if ($action === 'list') {
                $stmt = $db->prepare("SELECT * FROM webhooks WHERE user_id = ? ORDER BY created_at DESC");
                $stmt->execute([$userId]);
                $webhooks = $stmt->fetchAll();
                
                foreach ($webhooks as &$webhook) {
                    $webhook['events'] = json_decode($webhook['events'], true);
                }
                
                echo json_encode(['success' => true, 'data' => $webhooks]);
                
            } elseif ($action === 'logs') {
                $webhookId = isset($_GET['webhook_id']) ? intval($_GET['webhook_id']) : null;
                
                $query = "
                    SELECT wl.*, w.webhook_name, w.webhook_url
                    FROM webhook_logs wl
                    JOIN webhooks w ON wl.webhook_id = w.id
                    WHERE w.user_id = ?
                ";
                $params = [$userId];
                
                if ($webhookId) {
                    $query .= " AND wl.webhook_id = ?";
                    $params[] = $webhookId;
                }
                
                $query .= " ORDER BY wl.triggered_at DESC LIMIT 100";
                
                $stmt = $db->prepare($query);
                $stmt->execute($params);
                $logs = $stmt->fetchAll();
                
                foreach ($logs as &$log) {
                    $log['payload'] = json_decode($log['payload'], true);
                }
                
                echo json_encode(['success' => true, 'data' => $logs]);
            }
            break;
            
        case 'POST':
            requireCSRFToken();
            
            if ($action === 'create') {
                $webhookName = trim($_POST['webhook_name'] ?? '');
                $webhookUrl = trim($_POST['webhook_url'] ?? '');
                $events = $_POST['events'] ?? '[]';
                $secretToken = $_POST['secret_token'] ?? null;
                
                if (empty($webhookName) || empty($webhookUrl)) {
                    throw new Exception('Webhook name and URL are required');
                }
                
                // Validate URL
                if (!filter_var($webhookUrl, FILTER_VALIDATE_URL)) {
                    throw new Exception('Invalid webhook URL');
                }
                
                // Validate events JSON
                $decodedEvents = json_decode($events, true);
                if (json_last_error() !== JSON_ERROR_NONE || !is_array($decodedEvents)) {
                    throw new Exception('Invalid events JSON');
                }
                
                // Validate events are allowed
                $allowedEvents = [
                    'campaign.created',
                    'campaign.updated',
                    'campaign.completed',
                    'email.sent',
                    'email.opened',
                    'email.clicked',
                    'discount.used',
                    'conversion.tracked'
                ];
                
                foreach ($decodedEvents as $event) {
                    if (!in_array($event, $allowedEvents)) {
                        throw new Exception('Invalid event type: ' . $event);
                    }
                }
                
                $stmt = $db->prepare("
                    INSERT INTO webhooks 
                    (user_id, webhook_name, webhook_url, events, secret_token)
                    VALUES (?, ?, ?, ?, ?)
                ");
                $stmt->execute([$userId, $webhookName, $webhookUrl, $events, $secretToken]);
                
                $webhookId = $db->lastInsertId();
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Webhook created successfully',
                    'webhook_id' => $webhookId
                ]);
                
            } elseif ($action === 'test') {
                // Test webhook
                $webhookId = intval($_POST['webhook_id'] ?? 0);
                
                $stmt = $db->prepare("SELECT * FROM webhooks WHERE id = ? AND user_id = ?");
                $stmt->execute([$webhookId, $userId]);
                $webhook = $stmt->fetch();
                
                if (!$webhook) {
                    throw new Exception('Webhook not found');
                }
                
                $result = sendWebhookRequest($db, $webhook, 'webhook.test', ['test' => true]);
                
                echo json_encode([
                    'success' => $result['success'],
                    'http_code' => $result['http_code']
                ]);
            }
            break;
            
        case 'DELETE':
            requireCSRFToken();
            
            if ($action === 'delete') {
                $webhookId = intval($_GET['id'] ?? $_POST['id'] ?? 0);
                
                $stmt = $db->prepare("SELECT user_id FROM webhooks WHERE id = ?");
                $stmt->execute([$webhookId]);
                $webhook = $stmt->fetch();
                
                if (!$webhook || $webhook['user_id'] != $userId) {
                    throw new Exception('Webhook not found or access denied');
                }
                
                $stmt = $db->prepare("DELETE FROM webhooks WHERE id = ?");
                $stmt->execute([$webhookId]);
                
                echo json_encode(['success' => true, 'message' => 'Webhook deleted']);
            }
            break;
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

// Export function for use in other files
if (!function_exists('triggerWebhook')) {
    function triggerWebhook($db, $userId, $eventType, $payload) {
        // Implementation above
    }
}

