<?php
/**
 * REST API v1
 * Main API endpoint router
 */

require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/database.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-API-Key');

// Handle OPTIONS request for CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Authenticate request
$apiKey = null;
$authHeader = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
if (preg_match('/Bearer (.+)/', $authHeader, $matches)) {
    $apiKey = $matches[1];
} else {
    $apiKey = $_GET['api_key'] ?? $_SERVER['HTTP_X_API_KEY'] ?? null;
}

if (!$apiKey) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'API key required']);
    exit;
}

// Validate API key
$db = getDB();
$stmt = $db->prepare("
    SELECT ak.*, u.id as user_id, u.email 
    FROM api_keys ak
    JOIN users u ON ak.user_id = u.id
    WHERE ak.api_key = ? AND ak.status = 'active'
    AND (ak.expires_at IS NULL OR ak.expires_at > NOW())
");
$stmt->execute([$apiKey]);
$keyData = $stmt->fetch();

if (!$keyData) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Invalid or expired API key']);
    exit;
}

// Update last used timestamp
$stmt = $db->prepare("UPDATE api_keys SET last_used_at = NOW() WHERE id = ?");
$stmt->execute([$keyData['id']]);

// Set user context
$_SESSION['user_id'] = $keyData['user_id'];
$_SESSION['user_email'] = $keyData['email'];
$_SESSION['api_key_id'] = $keyData['id'];

// Parse request
$method = $_SERVER['REQUEST_METHOD'];
$path = trim($_SERVER['PATH_INFO'] ?? $_SERVER['REQUEST_URI'], '/');
$pathParts = explode('/', $path);

// Remove 'api/v1' from path
array_shift($pathParts);
array_shift($pathParts);

$resource = $pathParts[0] ?? '';
$resourceId = $pathParts[1] ?? null;

// Route to appropriate handler
switch ($resource) {
    case 'campaigns':
        require_once __DIR__ . '/campaigns.php';
        break;
        
    case 'email-campaigns':
        require_once __DIR__ . '/email_campaigns.php';
        break;
        
    case 'discount-codes':
        require_once __DIR__ . '/discount_codes.php';
        break;
        
    case 'analytics':
        require_once __DIR__ . '/analytics.php';
        break;
        
    case 'webhooks':
        require_once __DIR__ . '/webhooks.php';
        break;
        
    default:
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Resource not found']);
}

